﻿
using System;
using System.Collections;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.IO;
using ESRI.ArcGIS.SpatialAnalyst;
using ESRI.ArcGIS.Carto;
using ESRI.ArcGIS.Framework;
using ESRI.ArcGIS.ArcMapUI;
using ESRI.ArcGIS.Geodatabase;
using ESRI.ArcGIS.DataSourcesRaster;
using ESRI.ArcGIS.Geometry;
using ESRI.ArcGIS.Geoprocessing;
using ESRI.ArcGIS.Geoprocessor;
using ESRI.ArcGIS.DataManagementTools;
using ESRI.ArcGIS.GeoDatabaseUI;
using ESRI.ArcGIS.AnalysisTools;
using ESRI.ArcGIS.esriSystem;
using ESRI.ArcGIS.GeoAnalyst;
using BMPSitingTool.Common;
using BMPSitingTool.Classes;

namespace BMPSitingTool.Utilities
{
    class RasterUtilities
    {

        /// <summary>
        /// Convert the raster data to shape file
        /// </summary>
        /// <param name="strWorkspace"></param>
        /// <param name="strRaster">Name of the raster data, could be "FLOW", "SLOPE", Landuse Raster data, Impervious Raster data</param>
        /// <param name="pFilter"> is true only when strRaster is "FLOW" or "SLOPE"</param>
        /// <param name="pParse"> is true only when strRaster is "FLOW"</param>
        /// <param name="pBMPText"> contains BMPType only when strRaster is "FLOW" or "SLOPE", otherwise is blank </param>
        /// <param name="settings"></param> 
        /// <param name="strDACriteria"></param>
        /// <param name="pBMPCriteria"></param>

        /// <returns></returns>
        public static IFeatureClass ConvertRastertoFeature(string strWorkspace, string strRaster, bool pFilter, bool pParse, string pBMPText, 
            SitingToolSettings settings, BMPCriteria pBMPCriteria, IMap pMap)
        {
            IFeatureClass returnValue = null;
            BMPobj oBMP = null;
            IWorkspaceFactory pRasterWspFact = null;

            IRasterWorkspace pRasterWorksp = null;
            IRasterDataset pRasterDataset = null;
            IWorkspaceFactory pFeatWspFact = null;
            IFeatureWorkspace pFeatWsp = null;
            IFeatureClass pFeatClass = null;
            IDataset pDataset = null;
            IWorkspace pWkspace = null;
            IRaster pRaster = null;
            IRaster pOutputRaster = null;
            IRasterWorkspace pRasterWorksp1 = null;
            IWorkspaceFactory pRasterWspFact1 = null;
            try
            {
                string strDA_DS_Criteria = "";
                if (pBMPText != "")
                {
                    oBMP = pBMPCriteria.GetOneBMP(pBMPText);
                    if (strRaster == Resource.SLOPE)
                        strDA_DS_Criteria = pBMPCriteria.GetOneBMP(pBMPText).DC_DS;
                    if (strRaster == Resource.FLOW)
                        strDA_DS_Criteria = pBMPCriteria.GetOneBMP(pBMPText).DC_DA;
                }

                ///Open the grid
                pRasterWspFact = new RasterWorkspaceFactory();

                pRasterWorksp = (IRasterWorkspace)pRasterWspFact.OpenFromFile(strWorkspace, 0);

                pRasterDataset = pRasterWorksp.OpenRasterDataset(strRaster);

                /// Filter the Raster for Flow Criteria.....
                string strExp = null;
                string fullExp = null;

                if (pFilter)
                {

                    pRaster = pRasterDataset.CreateDefaultRaster();

                    RasterOperators pRasterUtilities = new RasterOperators(settings, pMap);
                    IMapAlgebraOp pMapAlgebraOp = (IMapAlgebraOp)pRasterUtilities.MapAlgebraOp;
                    pMapAlgebraOp.BindRaster((IGeoDataset)pRaster, Resource.RasterCon);

                    strExp = SitingToolUtilities.Parse_Expression(strDA_DS_Criteria, pParse, pRasterUtilities.CellSize);

                    if (strRaster == Resource.SLOPE)
                    {

                        //if (!string.IsNullOrEmpty(oBMP.DC_DS_Weight))
                        if ((oBMP.DC_DS_Weight_State) == "1")
                        {
                            fullExp = ParseCriteriaToExp(SitingToolUtilities.Parse_Expression2(oBMP.DC_DS_Weight, pParse, pRasterUtilities.CellSize), strExp);
                        }
                        else
                        {

                            //fullExp = "(con([" + Resource.RasterCon + "] " + strExp + ", 1,0))";
                            fullExp = "(con([" + Resource.RasterCon + "] " + strExp + ", 0,-1))";
                        }

                    }
                    else // "FLOW"
                    {
                        fullExp = "(con([" + Resource.RasterCon + "] " + strExp + ", 1,0))";
                    }

                    pOutputRaster = (IRaster)pMapAlgebraOp.Execute(fullExp);
                    pMapAlgebraOp.UnbindRaster(Resource.RasterCon);

                    ///Write it to the disk
                    WriteRasterDatasetToDisk(pOutputRaster, Resource.RasterCon, settings);
                    pRasterWspFact1 = new RasterWorkspaceFactory();

                    pRasterWorksp1 = (IRasterWorkspace)pRasterWspFact1.OpenFromFile(strWorkspace, 0);
                    pRasterDataset = pRasterWorksp1.OpenRasterDataset(Resource.RasterCon);

                }

                ///Delete the shape file if exists.......
                Delete_Dataset_ST(strWorkspace, strRaster + "_Ras");

                ///now convert the raster dataset to a feature dataset
                ///start by making a new, empty shapefile
                pFeatWspFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();

                pFeatWsp = (IFeatureWorkspace)pFeatWspFact.OpenFromFile(strWorkspace, 0);

                //make an IConversionOp object
                IConversionOp pConvOp = (IConversionOp)new RasterConversionOp();
                //pFeatClass = (IFeatureClass)pConvOp.RasterDataToPolygonFeatureData((IGeoDataset)pRasterDataset, (IWorkspace)pFeatWsp, strRaster + "_Ras.shp", true);
                //set weeding to be false --- will not simplify the polygon so that the raster and polygon result will be exactly the same shape
                pFeatClass = (IFeatureClass)pConvOp.RasterDataToPolygonFeatureData((IGeoDataset)pRasterDataset, (IWorkspace)pFeatWsp, strRaster + "_Ras.shp", false);
                //Delete the shape file if exists.......
                Delete_Dataset_ST(settings.WorkingfolderParent, strRaster + "_Ras");

                pDataset = (IDataset)pFeatClass;

                if (pDataset.CanCopy())
                {
                    pWkspace = (IWorkspace)pFeatWsp;
                    pWkspace = SitingToolUtilities.GetWorkspace(settings.WorkingfolderParent);
                    pDataset.Copy(strRaster + "_Ras", (IWorkspace)pWkspace);

                    pFeatClass = SitingToolUtilities.OpenShapeFile(settings.WorkingfolderParent, strRaster + "_Ras");
                }


                returnValue = pFeatClass;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at ConvertRastertoFeature: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at ConvertRastertoFeature: " + e.Message);
            }
            finally
            {
                if (pRasterWspFact != null)
                    Marshal.FinalReleaseComObject(pRasterWspFact);
                if (pRasterWorksp != null)
                    Marshal.FinalReleaseComObject(pRasterWorksp);
                if (pRasterDataset != null)
                    Marshal.FinalReleaseComObject(pRasterDataset);
                if (pFeatWspFact != null)
                    Marshal.FinalReleaseComObject(pFeatWspFact);
                if (pFeatWsp != null)
                    Marshal.FinalReleaseComObject(pFeatWsp);
                if (pDataset != null)
                    Marshal.FinalReleaseComObject(pDataset);
                if (pWkspace != null)
                    Marshal.FinalReleaseComObject(pWkspace);
                if (pRaster != null)
                    Marshal.FinalReleaseComObject(pRaster);
                if (pOutputRaster != null)
                    Marshal.FinalReleaseComObject(pOutputRaster);
                if (pRasterWorksp1 != null)
                    Marshal.FinalReleaseComObject(pRasterWorksp1);
                if (pRasterWspFact1 != null)
                    Marshal.FinalReleaseComObject(pRasterWspFact1);

            }
            return returnValue;

        }
        public static string ParseCriteriaToExp(string allCriteria, string strExp)
        {
            string returnValue = null;

            int itemLen = 0;
            string[] criteriaItems = null;
            string[] criteriaItem = null;
            string expressionStr = null;
            try
            {
                criteriaItems = allCriteria.Split(';');
                itemLen = criteriaItems.GetUpperBound(0) - criteriaItems.GetLowerBound(0) + 1;

                if (itemLen > 0)
                {
                    expressionStr = "con(";
                }

                short iIndex = 0;
                for (iIndex = 0; iIndex <= 3; iIndex++)
                {
                    ///if there is content, populate to the fields
                    if (itemLen > iIndex)
                    {
                        criteriaItem = criteriaItems[iIndex].Split(':');
                        expressionStr = expressionStr + "[" + Resource.RasterCon + "] " + criteriaItem[0] + " and " + "[" + Resource.RasterCon + "] " + strExp + "," + criteriaItem[1].Trim() + ",";
                    }
                }

                if (expressionStr.Length > 0)
                {
                    //expressionStr = expressionStr + "0)";
                    expressionStr = expressionStr + "-1)";
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at ParseCriteriaToExp: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at ParseCriteriaToExp: " + e.Message);
            }
            returnValue = expressionStr;
            return returnValue;


        }
        static void WriteRasterDatasetToDisk(IRaster pRaster, string pOutName, SitingToolSettings settings)
        {

            /// Create a raster workspace
            IRasterWorkspace pRWS = null;
            IWorkspaceFactory pWSF = null;
            IRasterDataset pRasterDataset = null;
            IDataset pDataset = null;
            IDataset pDS = null;
            IRasterBandCollection pRasBandCol = null;
            try
            {
                pWSF = new RasterWorkspaceFactory();
                pRWS = (IRasterWorkspace)pWSF.OpenFromFile(settings.Rasterfolder, 0);
                ///Delete the raster dataset if present on disk
                if (Directory.Exists(settings.Rasterfolder + "\\" + pOutName))
                {
                    pRasterDataset = pRWS.OpenRasterDataset(pOutName);
                    if ((pRasterDataset != null))
                    {

                        pDataset = (IDataset)pRasterDataset;
                        if (pDataset.CanDelete())
                            pDataset.Delete();
                        pDataset = null;
                        pRasterDataset = null;
                    }
                }
                /// SaveAs the projected raster
                pRasBandCol = (IRasterBandCollection)pRaster;
                pDS = pRasBandCol.SaveAs(pOutName, (IWorkspace)pRWS, Resource.GRID);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at WriteRasterDatasetToDisk: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at WriteRasterDatasetToDisk: " + e.Message);
            }
            finally
            {
                if (pRWS != null)
                    Marshal.FinalReleaseComObject(pRWS);
                if (pWSF != null)
                    Marshal.FinalReleaseComObject(pWSF);
                if (pDataset != null)
                    Marshal.FinalReleaseComObject(pDataset);
                if (pRasterDataset != null)
                    Marshal.FinalReleaseComObject(pRasterDataset);
                if (pDS != null)
                    Marshal.FinalReleaseComObject(pDS);
                //if (pRasBandCol != null)
                //    Marshal.FinalReleaseComObject(pRasBandCol);// cannot be released 
            }

        }
        public static void Delete_Dataset_ST(string strWorkspace, string strDataset)
        {

            ///Delete the FeatureClass.....
            IWorkspaceFactory pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();
            
            IPropertySet ConnectionProperties = new PropertySet();
            ConnectionProperties.SetProperty("DATABASE", strWorkspace);
            IFeatureWorkspace pShapeWS = (IFeatureWorkspace)pWsFact.Open(ConnectionProperties, 0);

            IWorkspace2 pWS2 = (IWorkspace2)pShapeWS;
            try
                {
                if (pWS2.get_NameExists(esriDatasetType.esriDTFeatureClass, strDataset))
                {
                    IFeatureClass pFClass = SitingToolUtilities.OpenShapeFile(strWorkspace, strDataset);
                    IDataset pDataset = null;
                    if ((pFClass != null))
                    {
                        pDataset = (IDataset)pFClass;
                        if (pDataset.CanDelete())
                            pDataset.Delete();
                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Delete_Dataset_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Delete_Dataset_ST: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pShapeWS != null)
                    Marshal.FinalReleaseComObject(pShapeWS);
                if (pWS2 != null)
                    Marshal.FinalReleaseComObject(pWS2);
            }
            return;


        }
        public static string Create_Join(IFeatureLayer pLayer, ITable pTable, string pFld1, ITable pStTab, string pFld2, SitingToolSettings settings)
        {
            IWorkspaceFactory pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();
            IPropertySet ConnectionProperties = null;
            ConnectionProperties = new PropertySet();
            ConnectionProperties.SetProperty("DATABASE", settings.WorkingfolderParent);
            IFeatureWorkspace pShapeWS = (IFeatureWorkspace)pWsFact.Open(ConnectionProperties, 0);
            IWorkspace2 pWS2 = (IWorkspace2)pShapeWS;
            /// Create virtual relate
            IMemoryRelationshipClassFactory pMemRelFact = null;
            IRelationshipClass pRelClass = null;

            IFeatureClass pSelFeatureClass = null;
            IGeoFeatureLayer pGLayer = null;
            IFeatureLayerDefinition pFeatureLayerDef = null;
            IFeatureClass pFeatureClass = null;
            IQueryFilter pQueryFilter = null;
            IFeatureLayer pSelFeatLayer = null;
            IDataset pDataset = null;
            IWorkspaceName pWorkspaceName = null;

            pMemRelFact = new MemoryRelationshipClassFactory();
            try
            {
                pRelClass = pMemRelFact.Open(pLayer.Name, (IObjectClass)pStTab, pFld2, (IObjectClass)pTable, pFld1, "forward", "backward", esriRelCardinality.esriRelCardinalityOneToOne);
                
                /// use Relate to perform a join
                IDisplayRelationshipClass pDispRC = (IDisplayRelationshipClass)pLayer;
                pDispRC.DisplayRelationshipClass(pRelClass, esriJoinType.esriLeftInnerJoin);

                pGLayer = (IGeoFeatureLayer)pLayer;

                pFeatureLayerDef = (IFeatureLayerDefinition)pGLayer;

                pFeatureClass = pGLayer.DisplayFeatureClass;

                pQueryFilter = new QueryFilter();

                pQueryFilter.WhereClause = pFeatureLayerDef.DefinitionExpression;

                pSelFeatLayer = pFeatureLayerDef.CreateSelectionLayer(pLayer.Name, true, "", "");
            
                pSelFeatureClass = pSelFeatLayer.FeatureClass;

                pDataset = (IDataset)pFeatureClass;
            
                IDatasetName pInDsName = (IDatasetName)pDataset.FullName;

                IFeatureClassName pFeatureClassName = (IFeatureClassName)new FeatureClassName();

                IDatasetName pOutDatasetName = (IDatasetName)pFeatureClassName;
                pOutDatasetName.Name = pSelFeatureClass.AliasName + "_Join";


                pWorkspaceName = (IWorkspaceName)new WorkspaceName();
                pWorkspaceName.PathName = settings.WorkingfolderParent;
                pWorkspaceName.WorkspaceFactoryProgID = "esriDataSourcesFile.shapefileworkspacefactory.1";

                pOutDatasetName.WorkspaceName = pWorkspaceName;

                ///Give the output shapefile the same props as the input dataset
                pFeatureClassName.FeatureType = pFeatureClass.FeatureType;
                pFeatureClassName.ShapeType = pFeatureClass.ShapeType;
                pFeatureClassName.ShapeFieldName = pFeatureClass.ShapeFieldName;

                ///Export selected features
                IExportOperation pExportOp = null;
                pExportOp = new ExportOperation();
                if (!pWS2.get_NameExists(esriDatasetType.esriDTFeatureClass, pOutDatasetName.Name))
                {
                    pExportOp.ExportFeatureClass((IDatasetName)pInDsName, null, null, null, (IFeatureClassName)pOutDatasetName, 0);
                }


                //IFeatureLayer2 actFeatLyr = (IFeatureLayer2)pLayer;
                //ITable actLyr = (ITable)pLayer;
                //IDisplayTable actDispTable = actFeatLyr as IDisplayTable;
                //IRelQueryTable actQueryTable = actDispTable.DisplayTable as IRelQueryTable;
                //IDisplayRelationshipClass actDispRelClass = actFeatLyr as IDisplayRelationshipClass;

                //actDispRelClass.DisplayRelationshipClass(null, esriJoinType.esriLeftInnerJoin);

            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Create_Join: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Create_Join: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pShapeWS != null)
                    Marshal.FinalReleaseComObject(pShapeWS);
                if (pWS2 != null)
                    Marshal.FinalReleaseComObject(pWS2);
                if (pMemRelFact != null)
                    Marshal.FinalReleaseComObject(pMemRelFact);
                if (pRelClass != null)
                    Marshal.FinalReleaseComObject(pRelClass);

                if (pGLayer != null)
                    Marshal.FinalReleaseComObject(pGLayer);
                if (pFeatureLayerDef != null)
                    Marshal.FinalReleaseComObject(pFeatureLayerDef);
                if (pFeatureClass != null)
                    Marshal.FinalReleaseComObject(pFeatureClass);
                if (pQueryFilter != null)
                    Marshal.FinalReleaseComObject(pQueryFilter);
                if (pSelFeatLayer != null)
                    Marshal.FinalReleaseComObject(pSelFeatLayer);
                if (pDataset != null)
                    Marshal.FinalReleaseComObject(pDataset);
                if (pWorkspaceName != null)
                    Marshal.FinalReleaseComObject(pWorkspaceName);

            }
            return pSelFeatureClass.AliasName + "_Join";
            

        }
        public static bool Delete_Raster(string name, SitingToolSettings settings)
        {
            bool returnValue = true;
            IRasterDataset pRDS = null;
            try
            {
                pRDS = OpenRasterDataset(name, settings);
                if (pRDS == null)
                    return returnValue;

                IDataset pDS = null;
                pDS = (IDataset)pRDS;
                if ((pDS.CanDelete()))
                {
                    pDS.Delete();
                    returnValue = true;
                }
                else
                {
                    returnValue = false;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Delete_Raster: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Delete_Raster: " + e.Message);
            }
            finally
            {

                if (pRDS != null)
                    Marshal.FinalReleaseComObject(pRDS);
            }

            return returnValue;
        }
        public static IRasterDataset OpenRasterDataset(string sRasterDs, SitingToolSettings settings)
        {
            IRasterDataset returnValue = null;
            IWorkspaceFactory pWsFact = null;
            IRasterWorkspace pWs = null;
            IRaster pRaster = null;
            try
            {
                if (!Directory.Exists(settings.Rasterfolder))
                {
                    return null;
                }
                pRaster = OpenRasterDatasetFromDisk(sRasterDs, settings);

                if (pRaster == null)
                    return null;

                pWsFact = new RasterWorkspaceFactory();
                pWs = (IRasterWorkspace)pWsFact.OpenFromFile(settings.Rasterfolder, 0);
                returnValue = pWs.OpenRasterDataset(sRasterDs);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at OpenRasterDataset: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at OpenRasterDataset: " + e.Message);
            }
            finally
            {
                if (pRaster != null)
                    Marshal.FinalReleaseComObject(pRaster);
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (pWs != null)
                    Marshal.FinalReleaseComObject(pWs);
            }
            return returnValue;

        }
        public static IRaster OpenRasterDatasetFromDisk(string pRasterName, SitingToolSettings settings)
        {
            IRaster returnValue = null;
            IWorkspaceFactory pWF = null;
            IRasterWorkspace pRW = null;
            IRasterDataset pRDS = null;

            try
            {
                /// check if raster dataset exist
                if (!Directory.Exists(settings.Rasterfolder + "\\" + pRasterName))
                {
                    return null;
                }

                ///Open workspace
                pWF = new RasterWorkspaceFactory();
                pRW = (IRasterWorkspace)pWF.OpenFromFile(settings.Rasterfolder, 0);

                if ((pRW.IsWorkspace(settings.Rasterfolder)))
                {
                    pRDS = pRW.OpenRasterDataset(pRasterName.ToLower());
                }
                if (pRDS == null)
                {
                    return null;
                }
                ///Get Raster from the raster dataset
                IRaster pRaster = null;
                pRaster = pRDS.CreateDefaultRaster();
                ///Return raster
                returnValue = pRaster;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at OpenRasterDatasetFromDisk: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at OpenRasterDatasetFromDisk: " + e.Message);
            }

            if (pWF != null)
                Marshal.FinalReleaseComObject(pWF);
            if (pRW != null)
                Marshal.FinalReleaseComObject(pRW);
            if (pRDS != null)
                Marshal.FinalReleaseComObject(pRDS);
            return returnValue;
        }
        public static void Calculate_Slope(IRasterLayer pRasterLayer, double dScalefac, SitingToolSettings settings, IMap pMap)
        {

            IWorkspace pRasterWS = null;
            IWorkspaceFactory pRasterWSFact = null;
            ISurfaceOp pSurfaceOp = null;
            IRaster pRaster = null;

            try
            {
                pRasterWSFact = new RasterWorkspaceFactory();

                ///perform slope
                ///Create a RasterSurfaceOp operator
                pSurfaceOp = (ISurfaceOp)new RasterSurfaceOp();

                IRasterAnalysisGlobalEnvironment analysisGlobalEnv;
                analysisGlobalEnv = (IRasterAnalysisGlobalEnvironment)pSurfaceOp;
                analysisGlobalEnv.AvoidDataConversion = true;


                ///get raster from rasterlayer            
                pRaster = pRasterLayer.Raster;
                double zFactor = 0;
                if (settings.ZUnit == "Meter")
                {
                    zFactor = dScalefac * 3.2808;
                }
                else
                {
                    zFactor = dScalefac;
                }

                ///Perform spatial operation
                IGeoDataset pOutRaster = pSurfaceOp.Slope((IGeoDataset)pRaster, esriGeoAnalysisSlopeEnum.esriGeoAnalysisSlopePercentrise, zFactor);

                ISaveAs2 outSave = (ISaveAs2)pOutRaster;
                pRasterWS = pRasterWSFact.OpenFromFile(settings.Rasterfolder, 0);
                outSave.SaveAsRasterDataset("dSlope", pRasterWS, Resource.GRID, null);

                /// Create a integer raster......................................
                RasterOperators pRasterOperators = new RasterOperators(settings, pMap);
                IMapAlgebraOp pMapAlgebraOp = pRasterOperators.MapAlgebraOp;

                pMapAlgebraOp.BindRaster((IGeoDataset)pOutRaster, Resource.SLOPE);

                pRaster = (IRaster)pMapAlgebraOp.Execute("Int([SLOPE])");
                pMapAlgebraOp.UnbindRaster(Resource.SLOPE);

                ///Write it to the disk
                WriteRasterDatasetToDisk(pRaster, Resource.SLOPE, settings);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Calculate_Slope: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Calculate_Slope: " + e.Message);
            }
            finally
            {
                if (pRasterWS != null)
                    Marshal.FinalReleaseComObject(pRasterWS);
                if (pRasterWSFact != null)
                    Marshal.FinalReleaseComObject(pRasterWSFact);
                if (pRaster != null)
                    Marshal.FinalReleaseComObject(pRaster);
            }
            return;
        }
        public static void Create_FlowDirectionandAccumulation(IRaster pRaster, SitingToolSettings settings, IMap pMap)
        {

            ///fill raw dem
            IWorkspaceFactory pRasterWSFact = null;
            IWorkspace pRasterWS = null;
            IRaster pFillDEMRaster = null;
            ISaveAs2 outSave = null;
            IRaster pFlowdirRaster = null;
            IRaster pFlowAccRaster = null;
            try
            {
                pFillDEMRaster = FillRawDEM(pRaster, settings, pMap);


                pRasterWSFact = new RasterWorkspaceFactory();
                pRasterWS = pRasterWSFact.OpenFromFile(settings.Rasterfolder, 0);

                ///Call the subroutine to create flow direction

                RasterOperators pRasterOperators = new RasterOperators(settings, pMap);
                IHydrologyOp pHydrologyOp = pRasterOperators.HydrologyOp;
                pFlowdirRaster = (IRaster)pHydrologyOp.FlowDirection((IGeoDataset)pFillDEMRaster, true, true);
                outSave = (ISaveAs2)pFlowdirRaster;
                outSave.SaveAsRasterDataset("dirFLOW", pRasterWS, Resource.GRID, null);

                ///Call the subroutine to create flow accumulation

                pFlowAccRaster = (IRaster)pHydrologyOp.FlowAccumulation((IGeoDataset)pFlowdirRaster);
                outSave = (ISaveAs2)pFlowAccRaster;
                outSave.SaveAsRasterDataset(Resource.FLOW, pRasterWS, Resource.GRID, null);

                pFlowAccRaster = null;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Create_FlowDirectionandAccumulation: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Create_FlowDirectionandAccumulation: " + e.Message);
            }
            finally
            {
                if (pRasterWSFact != null)
                    Marshal.FinalReleaseComObject(pRasterWSFact);
                if (pRasterWS != null)
                    Marshal.FinalReleaseComObject(pRasterWS);
                if (pFillDEMRaster != null)
                    Marshal.FinalReleaseComObject(pFillDEMRaster);
                if (pFlowdirRaster != null)
                    Marshal.FinalReleaseComObject(pFlowdirRaster);
                if (pFlowAccRaster != null)
                    Marshal.FinalReleaseComObject(pFlowAccRaster);

            }
            return;

        }
        private static IRaster FillRawDEM(IRaster pDEMRaster, SitingToolSettings settings, IMap pMap)
        {
            
            IRaster returnValue = null;
            IRaster pFillDEMRaster = null;
            ///Fill the raw dem
            try
            {

                RasterOperators pRasterOperators = new RasterOperators(settings, pMap);
                IHydrologyOp pHydrologyOp = pRasterOperators.HydrologyOp;
                pFillDEMRaster = (IRaster)pHydrologyOp.Fill((IGeoDataset)pDEMRaster);
                ///Write it to the disk
                WriteRasterDatasetToDisk(pFillDEMRaster, "FillDEM", settings);

                returnValue = pFillDEMRaster;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at FillRawDEM: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at FillRawDEM: " + e.Message);
            }
            finally
            {
                //if (pFillDEMRaster != null)
                //    Marshal.FinalReleaseComObject(pFillDEMRaster);// cannot be released 
            }
            return returnValue;
        }
 
    }
}
