using System;
using System.Collections;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.IO;
using ESRI.ArcGIS.SpatialAnalyst;
using ESRI.ArcGIS.Carto;
using ESRI.ArcGIS.Framework;
using ESRI.ArcGIS.ArcMapUI;
using ESRI.ArcGIS.Geodatabase;
using ESRI.ArcGIS.DataSourcesRaster;
using ESRI.ArcGIS.DataSourcesGDB;
using ESRI.ArcGIS.Geometry;
using ESRI.ArcGIS.Geoprocessing;
using ESRI.ArcGIS.Geoprocessor;
using ESRI.ArcGIS.DataManagementTools;
using ESRI.ArcGIS.AnalysisTools;
using ESRI.ArcGIS.esriSystem;
using ESRI.ArcGIS.Display;
using ESRI.ArcGIS.CartoUI;
using BMPSitingTool.Common;
using BMPSitingTool.Classes;
namespace BMPSitingTool.Utilities
{
    class SitingToolUtilities
	{

        public static bool CheckMapDocumentSavedStatus_ST(IApplication m_application)
        {
            bool returnValue = false;
            try
            {
                string pAppPath = null;
                pAppPath = DefineApplicationPath_ST(m_application);

                string dataSrcFN = null;

                dataSrcFN = m_application.Document.Title;

                ///Check if the .mxd is saved, if not force the user to save the .mxd
                if ((dataSrcFN.Replace(".mxd", "") == "Untitled"))
                {
                    MessageBox.Show("Please save .mxd file to continue. ", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                    return returnValue;
                }
                returnValue = true;

            }
            catch (Exception e)
            {
                MessageBox.Show("Error at CheckMapDocumentSavedStatus_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at CheckMapDocumentSavedStatus_ST: " + e.Message);
            }
            
            return returnValue;
            
            
        }

        public static string DefineApplicationPath_ST(IApplication m_application)
        {
            string returnValue = null;
            ITemplates pTemplates = null;
            IDocument pDoc = null;
            try
            {

                int lTempCount = 0;
                string strDocPath = null;
                if (m_application == null)
                    return returnValue;

                pTemplates = m_application.Templates;
                pDoc = m_application.Document;

                lTempCount = pTemplates.Count;

                strDocPath = pTemplates.Item[lTempCount - 1];
                string strMatch = null;
                strMatch = pDoc.Title;
                if ((pDoc.Title).ToUpper().Replace(".MXD", "") == (pDoc.Title).ToUpper())
                    strMatch = pDoc.Title + ".mxd";

                strDocPath = strDocPath.Replace(strMatch, "");
                returnValue = strDocPath;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at DefineApplicationPath_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at DefineApplicationPath_ST: " + e.Message);
            }
            finally
            {
                if (pTemplates != null)
                    Marshal.FinalReleaseComObject(pTemplates);
                if (pDoc != null)
                    Marshal.FinalReleaseComObject(pDoc);               
            }
            return returnValue;
        }

        public static bool CheckDataPath_LayerName(IApplication m_application, SitingToolSettings m_settings)
        {
            bool returnValue = false;

            try
            {
                string dataSrcFN = null;
                string errorMsg = "";
                /// Get the complete path of the application
                dataSrcFN = m_application.Document.Title;

                dataSrcFN = dataSrcFN.Replace(".mxd", "");
                dataSrcFN = DefineApplicationPath_ST(m_application) + dataSrcFN;

                if (dataSrcFN.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in the map document name and path." + System.Environment.NewLine;
                }

                if (m_settings.DEMdata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in DEM layer." + System.Environment.NewLine;
                }
                if (m_settings.Landusedata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in land use layer." + System.Environment.NewLine;
                }

                if (m_settings.RasterLanduseData.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in land use raster layer." + System.Environment.NewLine;
                }

                if (m_settings.Roaddata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in road layer." + System.Environment.NewLine;
                }

                if (m_settings.Soildata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in soil layer." + System.Environment.NewLine;
                }

                if (m_settings.Streamdata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in stream layer." + System.Environment.NewLine;
                }

                if (m_settings.Imperviousdata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in impervious layer." + System.Environment.NewLine;
                }

                if (m_settings.WTdata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in ground water depth layer." + System.Environment.NewLine;
                }

                if (m_settings.SoilTable.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in soil table." + System.Environment.NewLine;
                }

                if (m_settings.RasterLanduseTable.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in land use lookup table." + System.Environment.NewLine;
                }

                if (m_settings.Landownerdata.IndexOf(" ", 0) >= 0)
                {
                    errorMsg = errorMsg + "There should not be spaces in land ownership layer." + System.Environment.NewLine;
                }

                if (errorMsg != "")
                {
                    MessageBox.Show(errorMsg, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    return returnValue;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at CheckDataPath_LayerName: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at CheckDataPath_LayerName: " + e.Message);
            }
            returnValue = true;
            return returnValue;

        }

        public static void SetDataDirectory_ResultFolder(IApplication m_application, SitingToolSettings m_settings)
        {

            string pAppPath = null;

            try
            {
                pAppPath = DefineApplicationPath_ST(m_application);
                int resultFolderIndex = 1;

                while (Directory.Exists(pAppPath + "Result" + Convert.ToString(resultFolderIndex)))
                {
                    resultFolderIndex++;
                }
                Directory.CreateDirectory(pAppPath + "Result" + Convert.ToString(resultFolderIndex));
                m_settings.LoadSettings(m_settings.SitingToolSettingsFile);
                m_settings.WorkingfolderParent = pAppPath + "Result" + Convert.ToString(resultFolderIndex);

                //m_settings.Rasterfolder = pAppPath + "Cache";
                m_settings.SaveSettings(m_settings.SitingToolSettingsFile);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at SetDataDirectory_ResultFolder: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at SetDataDirectory_ResultFolder: " + e.Message);
            }
            return;

        }
        /// <summary>
        /// Check the data directory path
        /// </summary>
        public static void SetDataDirectory_CatchFolder(IApplication m_application, SitingToolSettings m_settings)
        {

            string pAppPath = null;

            try
            {
                pAppPath = DefineApplicationPath_ST(m_application);

                m_settings.LoadSettings(m_settings.SitingToolSettingsFile);

                m_settings.Rasterfolder = pAppPath + "Cache";
                m_settings.SaveSettings(m_settings.SitingToolSettingsFile);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at SetDataDirectory_CatchFolder: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at SetDataDirectory_CatchFolder: " + e.Message);
            }
            return;

        }



        /// <summary>
        /// 
        /// </summary>
        /// <param name="RLayerName"></param>
        /// <returns></returns>
        public static IRasterLayer GetInputRasterLayer(string RLayerName, IMap pMap)
        {
            IRasterLayer returnValue = null;

            short i = 0;
            ILayer pRLayer = null;
            try
            {
                for (i = 0; i <= (pMap.LayerCount - 1); i++)
                {
                    pRLayer = pMap.Layer[i];
                    if ((((pRLayer.Name).ToUpper() == (RLayerName).ToUpper()) & (pRLayer is IRasterLayer) & pRLayer.Valid))
                    {
                        returnValue = (IRasterLayer)pRLayer;
                   
                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetInputRasterLayer: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetInputRasterLayer: " + e.Message);
            }
            return returnValue;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="RLayerName"></param>
        /// <returns></returns>

        public static int GetInputRasterLayerCount(string RLayerName, IMap pMap)
        {
            int returnValue = 0;
            short i = 0;
            ILayer pRLayer = null;
            try
            {
                for (i = 0; i <= (pMap.LayerCount - 1); i++)
                {
                    pRLayer = pMap.Layer[i];
                    if ((((pRLayer.Name).ToUpper() == (RLayerName).ToUpper()) & (pRLayer is IRasterLayer) & pRLayer.Valid))
                    {
                        returnValue = returnValue + 1;
                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetInputRasterLayerCount: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetInputRasterLayerCount: " + e.Message);
            }
            if (pRLayer != null)
                Marshal.FinalReleaseComObject(pRLayer);

            return returnValue;
        }

        /// <summary>
        /// Function to get feature layer from map. If featurelayer does not contain feature class, no feature layer is returned.
        /// </summary>
        /// <param name="FLayerName"></param>
        /// <param name="pMap"></param>
        /// <returns></returns>
        public static ILayer GetInputFeatureLayer(string FLayerName, IMap pMap)
        {
            ILayer returnValue = null;
            short i = 0;
            ILayer pFLayer = null;
            try
            {
                for (i = 0; i <= (pMap.LayerCount - 1); i++)
                {
                    pFLayer = pMap.Layer[i];
                    if (((pFLayer.Name == FLayerName) & (pFLayer is IFeatureLayer | pFLayer is IRasterLayer) & pFLayer.Valid))
                    {
                        returnValue = pFLayer;
                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetInputFeatureLayer: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetInputFeatureLayer: " + e.Message);
            }
            return returnValue;
        }

        /// <summary>
        /// Function to get feature layer from map. If featurelayer does not contain feature class, no feature layer is returned.
        /// </summary>
        /// <param name="tableName"></param>
        /// <param name="pMap"></param>
        /// <returns></returns>
        public static ITable GetInputTable(string tableName, IMap pMap)
        {
            ITable returnValue = null;

            IStandaloneTableCollection pStandCol = null;
            pStandCol = (IStandaloneTableCollection)pMap;
            short i = 0;
            try
            {
                for (i = 0; i <= pStandCol.StandaloneTableCount - 1; i++)
                {
                    if ((pStandCol.StandaloneTable[i].Name).ToUpper() == (tableName).ToUpper())
                    {
                        returnValue = pStandCol.StandaloneTable[i].Table;

                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetInputTable: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetInputTable: " + e.Message);
            }
            return returnValue;
        }

        /// <summary>
        /// Function to get feature layer from Workspace. If featurelayer does not contain feature class, no feature layer is returned.
        /// </summary>
        /// <param name="strWorkspace"></param>
        /// <param name="FLayerName"></param>
        /// <returns></returns>
        public static ILayer GetFeatureLayer(string strWorkspace, string FLayerName)
        {
            ILayer returnValue = null;
            IFeatureClass pFeatClass = null;
            IFeatureLayer pFeatLyr = null;
            try
            {
                pFeatClass = OpenShapeFile(strWorkspace, FLayerName);
                if (pFeatClass == null)
                    return returnValue;

                pFeatLyr = new FeatureLayer();
                pFeatLyr.FeatureClass = pFeatClass;
                pFeatLyr.Name = pFeatClass.AliasName;
                pFeatLyr.Visible = false;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetFeatureLayer: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetFeatureLayer: " + e.Message);
            }
            returnValue = pFeatLyr;

            if (pFeatClass != null)
                Marshal.FinalReleaseComObject(pFeatClass);
            return returnValue;
        }

        /// <summary>
        /// Function to get standalone table from Workspace.
        /// </summary>
        /// <param name="strWorkspace"></param>
        /// <param name="tableName"></param>
        /// <returns></returns>
        public static IStandaloneTable GetDataTable(string strWorkspace, string tableName)
        {

            IWorkspaceFactory pWsFact = null;
            IPropertySet ConnectionProperties = null;
            IFeatureWorkspace pWs = null;

            IStandaloneTable returnValue = null;
            ITable pTable = null;
            IStandaloneTable psTable = null;
            pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();

            try
            {
                ConnectionProperties = new PropertySet();
                ConnectionProperties.SetProperty("DATABASE", strWorkspace);

                pWs = (IFeatureWorkspace)pWsFact.Open(ConnectionProperties, 0);
                if (!(string.IsNullOrEmpty(tableName)))
                {
                    pTable = pWs.OpenTable(tableName);

                    psTable = new StandaloneTable();
                    psTable.Table = pTable;

                    returnValue = psTable;
                }
                else
                {
                    returnValue = null;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetDataTable: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetDataTable: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pWs != null)
                    Marshal.FinalReleaseComObject(pWs);

            }
            return returnValue;
        }

        public static IFeatureClass OpenInMemory(string name)
        {
            IFeatureClass returnValue = null;

            IWorkspaceFactory pWsFact = null;
            IPropertySet ConnectionProperties = null;
            IFeatureWorkspace pShapeWS = null;
            bool isShapeWS = false;
            try
            {

                IWorkspaceFactory workspaceFactory = new InMemoryWorkspaceFactory();
                IWorkspaceName workspaceName = workspaceFactory.Create("", "in_memory", null, 0);
                //ESRI.ArcGIS.esriSystem.IName name = (ESRI.ArcGIS.esriSystem.IName)workspaceName;
                //IWorkspace workspace = (IWorkspace)name.Open(); 
                ///The shapefile workspace exist
                pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();

                ConnectionProperties = new PropertySet();
                ConnectionProperties.SetProperty("DATABASE", "in_memory");
                pShapeWS = (IFeatureWorkspace)workspaceFactory.Open(ConnectionProperties, 0);


                returnValue = pShapeWS.OpenFeatureClass(name);



            }
            catch (Exception e)
            {
                MessageBox.Show("Error at OpenShapeFile: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at OpenShapeFile: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pShapeWS != null)
                    Marshal.FinalReleaseComObject(pShapeWS);
            }

            return returnValue;

        }
        /// <summary>
        /// Open feature dataset (.shp) from disk and return the featureclass.
        /// </summary>
        /// <param name="dir_Renamed"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public static IFeatureClass OpenShapeFile(string dir_Renamed, string name)
        {
            IFeatureClass returnValue = null;

            IWorkspaceFactory pWsFact = null;
            IPropertySet ConnectionProperties = null;
            IFeatureWorkspace pShapeWS = null;
            bool isShapeWS = false;
            try
            {
                ///The shapefile workspace exist
                pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();
                isShapeWS = pWsFact.IsWorkspace(dir_Renamed);
                
                if ((isShapeWS))
                {
                    ConnectionProperties = new PropertySet();
                    ConnectionProperties.SetProperty("DATABASE", dir_Renamed);
                    pShapeWS = (IFeatureWorkspace)pWsFact.Open(ConnectionProperties, 0);

                    IWorkspace2 pWS2 = (IWorkspace2)pShapeWS;
                    ///Check if the shapefile exist
                    if (pWS2.get_NameExists(esriDatasetType.esriDTFeatureClass, name))
                        returnValue = pShapeWS.OpenFeatureClass(name);
                    else
                        returnValue = null;

                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at OpenShapeFile: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at OpenShapeFile: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pShapeWS != null)
                    Marshal.FinalReleaseComObject(pShapeWS);
            }

            return returnValue;

        }

        public static ITable OpenTable(string dir_Renamed, string name)
        {
            ITable returnValue = null;

            IWorkspaceFactory pWsFact = null;
            IPropertySet ConnectionProperties = null;
            IFeatureWorkspace pShapeWS = null;
            bool isShapeWS = false;
            try
            {
                ///The shapefile workspace exist
                pWsFact = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();
                isShapeWS = pWsFact.IsWorkspace(dir_Renamed);

                if ((isShapeWS))
                {
                    ConnectionProperties = new PropertySet();
                    ConnectionProperties.SetProperty("DATABASE", dir_Renamed);
                    pShapeWS = (IFeatureWorkspace)pWsFact.Open(ConnectionProperties, 0);

                    IWorkspace2 pWS2 = (IWorkspace2)pShapeWS;
                    ///Check if the shapefile exist
                    if (pWS2.get_NameExists(esriDatasetType.esriDTTable, name))
                        returnValue = pShapeWS.OpenTable(name);
                    else
                        returnValue = null;

                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at OpenTable: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at OpenTable: " + e.Message);
            }
            finally
            {
                if (pWsFact != null)
                    Marshal.FinalReleaseComObject(pWsFact);
                if (ConnectionProperties != null)
                    Marshal.FinalReleaseComObject(ConnectionProperties);
                if (pShapeWS != null)
                    Marshal.FinalReleaseComObject(pShapeWS);
            }

            return returnValue;

        }
        /// <summary>
        /// Checks the input projection of the selected layers. Returns a FALSE if all input layers are not in same projection
        /// </summary>
        /// <param name="pMap"></param>
        /// <param name="m_settings"></param>
        /// <returns></returns>
        public static bool CheckSelectedDataProjection(IMap pMap, SitingToolSettings m_settings)
        {
            bool returnValue = false;

            ILayer pLayer = null;
            ISpatialReference pSpatialReference = null;
            //check the projection of only dem, STREAM & landuse
            short i = 0;
            Dictionary<string,string> pSpatialRefDict = null;
            string pSpatialReferenceName = null;
            try
            {
                for (i = 0; i <= (pMap.LayerCount - 1); i++) {
                    pLayer = pMap.Layer[i];
                    if ((pLayer.Name == m_settings.DEMdata) | (pLayer.Name == m_settings.Landusedata) | (pLayer.Name == m_settings.RasterLanduseData) | (pLayer.Name == m_settings.Roaddata) | (pLayer.Name == m_settings.Soildata) | (pLayer.Name == m_settings.Streamdata) | (pLayer.Name == m_settings.Imperviousdata) | (pLayer.Name == m_settings.WTdata) | (pLayer.Name == m_settings.Landownerdata)) {
                        if ((pLayer != null)) {
                            pSpatialReference = GetSpatialReferenceForLayer(pLayer);

                            if ((pSpatialReference == null)) {
                                pSpatialReferenceName = "Undefined";
                            } else {

                                pSpatialReferenceName = pSpatialReference.Name;
                            }

                            // now Check the Projection...........
                            if (pSpatialRefDict == null) {
                                pSpatialRefDict = new Dictionary<string, string>();
                                pSpatialRefDict.Add(pSpatialReferenceName, pSpatialReferenceName);
                            }
                            if (!pSpatialRefDict.ContainsKey(pSpatialReferenceName)) {
                                MessageBox.Show("Spatial Refenece mismatch between input data. Please correct.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                return returnValue;
                            }
                        }
                    }
                }

                returnValue = true;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at CheckSelectedDataProjection: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at CheckSelectedDataProjection: " + e.Message);
            }
            finally
            {
                if (pLayer != null)
                    Marshal.FinalReleaseComObject(pLayer);
                if (pSpatialReference != null)
                    Marshal.FinalReleaseComObject(pSpatialReference);
            }
            return returnValue;
        }


        /// <summary>
        ///  Checks the input Datasets for specific fields availablility that are used for Analysis.......
        /// </summary>
        /// <param name="CreateJoin"></param>
        /// <param name="pMap"></param>
        /// <param name="m_settings"></param>
        /// <returns></returns>
        public static bool ValidateDatasets_ST(bool CreateJoin, IMap pMap, SitingToolSettings m_settings)
        {
            bool returnValue = false;
            
            ITable pFTable = null;
            IStandaloneTableCollection pStandCol = null;
            IStandaloneTable pStTab = null;
            short iFieldType = 0;

            IDisplayTable pDispTable = null;
            IDisplayTable pDispTable2 = null;
            ITable pReltable = null;
            ITable pTable = null;
            ILayer pLayer = null;
            IFeatureClass pFClass = null;
            IFeatureLayer pFeatLayer = null;
            IQueryFilter filter_Renamed = null;

            try
            {
                ///////////////////////////////////////////////////
                /// Nothing to Validate on Road Data.......

                ///////////////////////////////////////////////////
                /// Nothing to Validate on Stream Data....

                ///////////////////////////////////////////////////
                /// Validate water table Data..................

                m_settings.LoadSettings(m_settings.SitingToolSettingsFile);
                if (!CreateJoin)
                {
                    pFTable = (ITable)GetInputFeatureLayer(m_settings.WTdata, pMap);
                    if ((pFTable != null))
                    {
                        if (pFTable.Fields.FindField("GWdep_ft") == -1)
                        {
                            MessageBox.Show("Water table depth field is missing in water table data (" + m_settings.WTdata + ")." + "\n" + "'GWdep_ft' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }

                    
                        iFieldType = Convert.ToInt16(pFTable.Fields.Field[pFTable.Fields.FindField("GWdep_ft")].Type);
                        if (!(((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSmallInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSingle) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeDouble)))
                        {
                            MessageBox.Show("Field \"GWdep_ft\" should be of type integer, short, double or float in water table dat (" + m_settings.WTdata + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }
                    }

                    // Validate Soil Data................................
                    pFTable = (ITable)GetInputFeatureLayer(m_settings.Soildata, pMap);
                    // Get the Layer from Working folder....
                    if ((pFTable != null))
                    {
                         if (pFTable.Fields.FindField("MUKEY") == -1)
                        {
                            MessageBox.Show("Relational field is missing in soil data (" + m_settings.Soildata + ")." + "\n" + "'MUKEY' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;

                        }

                        iFieldType = Convert.ToInt16(pFTable.Fields.Field[pFTable.Fields.FindField("MUKEY")].Type);

                        if (!((esriFieldType)iFieldType == esriFieldType.esriFieldTypeString))
                        {
                            MessageBox.Show("Field \"MUKEY\" should be of type string in soil data (" + m_settings.Soildata + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }
                    }

                    // Validate Landuse Data........................

                    pFTable = (ITable)GetInputFeatureLayer(m_settings.Landusedata, pMap);
                    if ((pFTable != null))
                    {
                        if (pFTable.Fields.FindField("LU_DESC") == -1)
                        {
                            MessageBox.Show("Land use field is missing in land use data (" + m_settings.Landusedata + ")." + "\n" + "'LU_DESC' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }

                        iFieldType = Convert.ToInt16(pFTable.Fields.Field[pFTable.Fields.FindField("LU_DESC")].Type);
                        if (!((esriFieldType)iFieldType == esriFieldType.esriFieldTypeString))
                        {
                            MessageBox.Show("Field \"LU_DESC\" should be of type string in land use data (" + m_settings.Landusedata + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }
                    }

                    // Validate LandOwner Data........................
                    pFTable = (ITable)GetInputFeatureLayer(m_settings.Landownerdata, pMap);
                    if ((pFTable != null))
                    {
                        if (pFTable.Fields.FindField("Ownership") == -1)
                        {
                            MessageBox.Show("Ownership field is missing in land owner data (" + m_settings.Landownerdata + ")." + "\n" + "'Ownership' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }

                        iFieldType = Convert.ToInt16(pFTable.Fields.Field[pFTable.Fields.FindField("Ownership")].Type);

                        if (!((esriFieldType)iFieldType == esriFieldType.esriFieldTypeString))
                        {
                            MessageBox.Show("Field \"Ownership\" should be of type string in land owner data (" + m_settings.Landownerdata + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return returnValue;
                        }
                    }
                }

                pStandCol = (IStandaloneTableCollection)pMap;
                short i = 0;
                //get table from working folder
                if (CreateJoin)
                {
                    pStTab = (IStandaloneTable)GetDataTable(m_settings.WorkingfolderParent, m_settings.SoilTable);
                }
                else
                {
                    for (i = 0; i <= pStandCol.StandaloneTableCount - 1; i++)
                    {
                        if ((pStandCol.StandaloneTable[i].Name).ToUpper() == (m_settings.SoilTable).ToUpper())
                        {
                            pStTab = pStandCol.StandaloneTable[i];
                            break; 
                        }

                    }
                }


                if ((pStTab != null))
                {


                    if (pStTab.Table.Fields.FindField("MUKEY") == -1)
                    {
                        MessageBox.Show("Relational field is missing in Soil table (" + m_settings.SoilTable + ")." + "\n" + "'MUKEY' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }
                    iFieldType = (short)pStTab.Table.Fields.Field[pStTab.Table.Fields.FindField("MUKEY")].Type;
                    if (!((esriFieldType)iFieldType == esriFieldType.esriFieldTypeString))
                    {
                        MessageBox.Show("Field \"MUKEY\" should be of type string in Soil table (" + m_settings.SoilTable + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }

                    if (pStTab.Table.Fields.FindField("HYDGRP") == -1)
                    {
                        MessageBox.Show("Hydrogroup field is missing in Soil table (" + m_settings.SoilTable + ")." + "\n" + "'HYDGRP' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }
                    iFieldType = (short)pStTab.Table.Fields.Field[pStTab.Table.Fields.FindField("HYDGRP")].Type;
                    if (!((esriFieldType)iFieldType == esriFieldType.esriFieldTypeString))
                    {
                        MessageBox.Show("Field \"HYDGRP\" should be of type string in Soil table (" + m_settings.SoilTable + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }
                }

                pStTab = null;
                if (CreateJoin)
                {
                    pStTab = GetDataTable(m_settings.WorkingfolderParent, m_settings.RasterLanduseTable);
                }
                else
                {
                    for (i = 0; i <= pStandCol.StandaloneTableCount - 1; i++)
                    {
                        if (m_settings.RasterLanduseTable != null)
                        {
                            if ((pStandCol.StandaloneTable[i].Name).ToUpper() == (m_settings.RasterLanduseTable).ToUpper())
                            {
                                pStTab = pStandCol.StandaloneTable[i];
                                break;
                            }
                        }

                    }
                }

                if ((pStTab != null))
                {

                    if (pStTab.Table.Fields.FindField("LUCODE") == -1)
                    {
                        MessageBox.Show("Relational field is missing in land use table (" + m_settings.RasterLanduseTable + ")." + "\n" + "'LUCODE' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }

                    iFieldType = (short)pStTab.Table.Fields.Field[pStTab.Table.Fields.FindField("LUCODE")].Type;
                    if (!(((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSmallInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSingle) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeDouble)))
                    {
                        MessageBox.Show("Field \"LUCODE\" should be of type integer, short, double or float in land use table (" + m_settings.RasterLanduseTable + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }

                    if (pStTab.Table.Fields.FindField("SUITABLE") == -1)
                    {
                        MessageBox.Show("SUITABLE field is missing in land use table (" + m_settings.RasterLanduseTable + ")." + "\n" + "'SUITABLE' field should be available for analysis.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }

                    iFieldType = (short)pStTab.Table.Fields.Field[pStTab.Table.Fields.FindField("SUITABLE")].Type;
                    if (!(((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSmallInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeInteger) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeSingle) | ((esriFieldType)iFieldType == esriFieldType.esriFieldTypeDouble)))
                    {
                        MessageBox.Show("Field \"SUITABLE\" should be of type integer, short, double or float in land use table (" + m_settings.RasterLanduseTable + ").", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return returnValue;
                    }


                    /////////////////////////////////////////////////////
                    /// Now Create Mrlc relation. Exporting  and Joining process is performed here ...........................................
                    ///////////////////////////////////////////////////

                    if (CreateJoin) 
                    {
                        pLayer = GetInputFeatureLayer(m_settings.RasterLanduseData, pMap);
                        // Get the Layer from Working folder....
                        if ((pLayer != null))
                        {
                            if (!Copy_RasterData_toCachefolder(m_settings.RasterLanduseData, m_settings, pMap))
                            {
                                returnValue = false;
                                MessageBox.Show("Land use raster data could not be exported.", BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                return returnValue;
                            }

                            pFClass = RasterUtilities.ConvertRastertoFeature(m_settings.Rasterfolder, m_settings.RasterLanduseData, false, false, "", m_settings, null, pMap);
                            pFeatLayer = new FeatureLayer();
                            pFeatLayer.FeatureClass = pFClass;

                            pDispTable = (IDisplayTable)pFeatLayer;

                            pTable = (ITable)pDispTable.DisplayTable;
                            pDispTable2 = (IDisplayTable)pStTab;
                            filter_Renamed = new QueryFilter();
                            pReltable = pDispTable2.DisplayTable;

                            m_settings.RasterLanduseData = RasterUtilities.Create_Join((IFeatureLayer)pFeatLayer, (ITable)pTable, "GRIDCODE", pReltable, "LUCODE", m_settings);
                        }

                    }
                
                }

                returnValue = true;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at ValidateDatasets_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at ValidateDatasets_ST: " + e.Message);
            }
            finally
            {
                if (pFTable != null)
                    Marshal.FinalReleaseComObject(pFTable);
                if (pStandCol != null)
                    Marshal.FinalReleaseComObject(pStandCol);
                if (pStTab != null)
                    Marshal.FinalReleaseComObject(pStTab);
                if (pDispTable != null)
                    Marshal.FinalReleaseComObject(pDispTable);
                if (pDispTable2 != null)
                    Marshal.FinalReleaseComObject(pDispTable2);
                if (pReltable != null)
                    Marshal.FinalReleaseComObject(pReltable);
                if (pTable != null)
                    Marshal.FinalReleaseComObject(pTable);
                if (pLayer != null)
                    Marshal.FinalReleaseComObject(pLayer);
                if (pFClass != null)
                    Marshal.FinalReleaseComObject(pFClass);
                if (pFeatLayer != null)
                    Marshal.FinalReleaseComObject(pFeatLayer);
                if (filter_Renamed != null)
                    Marshal.FinalReleaseComObject(filter_Renamed);
                if (pFTable != null)
                    Marshal.FinalReleaseComObject(pFTable);
            }

            return returnValue;

        }

        /// <summary>
        /// Checks the input projection of an input layer. Checks the type of input layer (feature/raster) and gets its spatial reference
        /// </summary>
        /// <param name="pLayer"></param>
        /// <returns></returns>
        public static ISpatialReference GetSpatialReferenceForLayer(ILayer pLayer)
        {
            ISpatialReference returnValue = null;

            IFeatureLayer pFeatureLayer = null;
            IRasterLayer pRasterLayer = null;
            IGeoDataset pGeoDataSet = null;
            IRasterProps pRasterProps = null;
            try
            {
                if ((pLayer is IFeatureLayer)) {

                    pFeatureLayer = (IFeatureLayer)pLayer;
                    pGeoDataSet = (IGeoDataset)pFeatureLayer.FeatureClass;
                    returnValue = pGeoDataSet.SpatialReference;
                } else if ((pLayer is IRasterLayer)) {

                    pRasterLayer = (IRasterLayer)pLayer;
                    pRasterProps = (IRasterProps)pRasterLayer.Raster;
                    returnValue = pRasterProps.SpatialReference;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetSpatialReferenceForLayer: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetSpatialReferenceForLayer: " + e.Message);
            }
            if (pFeatureLayer != null)
                Marshal.FinalReleaseComObject(pFeatureLayer);
            if (pRasterLayer != null)
                Marshal.FinalReleaseComObject(pRasterLayer);
            if (pGeoDataSet != null)
                Marshal.FinalReleaseComObject(pGeoDataSet);
            if (pRasterProps != null)
                Marshal.FinalReleaseComObject(pRasterProps);

            return returnValue;
        }


        public static bool Copy_RasterData_toCachefolder(string strRaster, SitingToolSettings m_settings, IMap pMap)
        {

            bool returnValue = true;

            IRasterLayer pRasterLayer = null;
            try
            {
                pRasterLayer = (IRasterLayer)GetInputFeatureLayer(strRaster, pMap);
                if (pRasterLayer != null)
                {
                    Copy_Raster_Data(strRaster, (IDataLayer)pRasterLayer, m_settings.Rasterfolder, pMap);
                }
                else
                {
                    returnValue = false;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Copy_RasterData_toCachefolder: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Copy_RasterData_toCachefolder: " + e.Message);
            }
            finally
            {
                if (pRasterLayer != null)
                    Marshal.FinalReleaseComObject(pRasterLayer);
            }
            return returnValue;
        }

        public static void Copy_Raster_Data(string strRaster, IDataLayer pRasterLayer, string strCopyTo, IMap pMap)
        {
           
            strRaster = strRaster.Trim();

            IDatasetName pDSName = null;
            pDSName = (IDatasetName)pRasterLayer.DataSourceName;

            
            //geoprocessing
            Geoprocessor GP = new Geoprocessor();
            CopyRaster pCopyRaster = new CopyRaster();
            
            try
            {
                pCopyRaster.in_raster = pDSName.WorkspaceName.PathName + "/" + pDSName.Name;
                pCopyRaster.out_rasterdataset = strCopyTo + "/" + strRaster;
                //string[] test = Directory.GetFiles(strCopyTo);
                if (!File.Exists(strCopyTo + "\\" + strRaster + ".aux.xml") & !File.Exists(strCopyTo + "\\" + strRaster + Resource.XMLFileExt) & !File.Exists(strCopyTo + "\\" + strRaster + ".aux"))
                    GP.Execute(pCopyRaster,null);

                RemoveRasterLayer(strRaster, strCopyTo, pMap);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Copy_Raster_Data: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Copy_Raster_Data: " + e.Message);
            }

            if (pRasterLayer != null)
                Marshal.FinalReleaseComObject(pRasterLayer);
            if (pDSName != null)
                Marshal.FinalReleaseComObject(pDSName);
            //if (pCopyRaster != null)
            //     Marshal.FinalReleaseComObject(pCopyRaster);//cannot be released
            //if (GP != null)
            //    Marshal.FinalReleaseComObject(GP);//cannot be released

            return;

        }
        private static void RemoveRasterLayer(string strRaster, string path, IMap pMap)
        {
            IDatasetName pDSName = null;
            IDataLayer pRasterLayer = null;

            short i = 0;
            ILayer pRLayer = null;
            try
            {
                for (i = 0; i <= (pMap.LayerCount - 1); i++)
                {
                    pRLayer = pMap.Layer[i];
                    if ((((pRLayer.Name).ToUpper() == (strRaster).ToUpper()) & (pRLayer is IRasterLayer) & pRLayer.Valid))
                    {
                        pRasterLayer = (IDataLayer)pRLayer;
                        pDSName = (IDatasetName)pRasterLayer.DataSourceName;
                        if (pDSName.WorkspaceName.PathName == path + "\\")
                        {
                            pMap.DeleteLayer(pRLayer);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at RemoveRasterLayer: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at RemoveRasterLayer: " + e.Message);
            }

            if (pRasterLayer != null)
                Marshal.FinalReleaseComObject(pRasterLayer);
            if (pDSName != null)
                Marshal.FinalReleaseComObject(pDSName);
        }



        public static void Delete_Dataset_ST(string strWorkspace, string strDataset)
        {


            //Delete the FeatureClass.....
            IFeatureClass pFClass = null;
            try
            {
                pFClass = OpenShapeFile(strWorkspace, strDataset);
                IDataset pDataset = null;
                if ((pFClass != null)) {
                    pDataset = (IDataset)pFClass;
                    if (pDataset.CanDelete())
                        pDataset.Delete();
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Delete_Dataset_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Delete_Dataset_ST: " + e.Message);
            }
            if (pFClass != null)
                Marshal.FinalReleaseComObject(pFClass);
            return;
        }

        public static void Delete_Table(string strWorkspace, string strDataset)
        {


            //Delete the FeatureClass.....
            ITable pTable = null;
            try
            {
                pTable = OpenTable(strWorkspace, strDataset);
                IDataset pDataset = null;
                if ((pTable != null))
                {
                    pDataset = (IDataset)pTable;
                    if (pDataset.CanDelete())
                        pDataset.Delete();
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Delete_Table: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Delete_Table: " + e.Message);
            }
            if (pTable != null)
                Marshal.FinalReleaseComObject(pTable);
            return;
        }

        public static void RenderUniqueValueFillSymbol_ST(IFeatureLayer pFeatureLayer, string pFieldNames, string pHeading)
        {
            IGeoFeatureLayer pLyr = null;
            pLyr = (IGeoFeatureLayer)pFeatureLayer;

            /////////////////////////////////////////////////////////////////
            /// Seems to be problem deletin/adding fields with result.........
            /// Copy the result Featureclass and Clean the featureClass.......
            /////////////////////////////////////////////////////////////////
            IFeatureClass pFeatClass = null;
            pFeatClass = pFeatureLayer.FeatureClass;

            // Now add the Field.....
            IField pField = null;
            IFieldEdit pFieldEdit = null;
            try
            {
                pField = new Field();
                pFieldEdit = (IFieldEdit)pField;
                pFieldEdit.Name_2 = "BMP_Combin";
                pFieldEdit.AliasName_2 = "BMP_Combin";
                pFieldEdit.Type_2 = esriFieldType.esriFieldTypeString;
                pFieldEdit.Length_2 = 255;
                pFeatClass.AddField(pField);

                string[] strFields = null;
                string strExp = null;
                strFields = pFieldNames.Split(';');
                short iCnt = 0;

                IFeatureCursor pFeatureCursor = null;
                pFeatureCursor = pFeatClass.Search(null, true);
                IFeature pFeature = null;
                pFeature = pFeatureCursor.NextFeature();
                string Output = "";
                while ((pFeature != null)) {
                    strExp = "";
                    for (iCnt = 0; iCnt < strFields.GetLength(0); iCnt++)
                    {
                        strExp = strExp + pFeature.Value[pFeature.Fields.FindField(strFields[iCnt])] + ",";
                    }
                    Output = strExp.Substring(0, strExp.Length - 1); 

                    Output = Output.Replace(" ", "");
                
                    while (Output.Contains(",,"))
                    {
                        Output = Output.Replace(",,", ",");
                    }

                    Output = Output.TrimEnd(',', ' ');
                    Output = Output.TrimStart(',', ' ');

                    pFeature.Value[pFeature.Fields.FindField("BMP_Combin")] = Output;
                    pFeature.Store();
                    pFeature = pFeatureCursor.NextFeature();
                }

                //** Creates a UniqueValuesRenderer and applies it to first layer in the map.
                IFeatureClass pFeatCls = null;
                pFeatCls = pFeatureLayer.FeatureClass;
                IQueryFilter pQueryFilter = null;
                pQueryFilter = new QueryFilter();
                //empty supports: SELECT *
                IFeatureCursor pFeatCursor = null;
                pFeatCursor = pFeatCls.Search(pQueryFilter, false);
                //** Make the color ramp we will use for the symbols in the renderer
                IHsvColor pFromColor = null;
                pFromColor = new HsvColor();
                pFromColor.Hue = 0;
                // red
                pFromColor.Saturation = 100;
                pFromColor.Value = 70;

                IHsvColor pToColor = null;
                pToColor = new HsvColor();
                pToColor.Hue = 255;
                // velvet
                pToColor.Saturation = 100;
                pToColor.Value = 70;

                IAlgorithmicColorRamp rx = null;
                rx = new AlgorithmicColorRamp();
                rx.Algorithm = esriColorRampAlgorithm.esriHSVAlgorithm;
                rx.FromColor = pFromColor;
                rx.ToColor = pToColor;

                //** Make the renderer
                IUniqueValueRenderer pRender = null;
                int n = 0;
                pRender = new UniqueValueRenderer();

                ISimpleFillSymbol symd = null;
                symd = new SimpleFillSymbol();
                symd.Style = esriSimpleFillStyle.esriSFSSolid;
                symd.Outline.Width = 0.4;

                /// These properties should be set prior to adding values
                pRender.FieldCount = 1;
                pRender.Field[0] = "BMP_Combin";
                pRender.FieldDelimiter = ",";
                pRender.DefaultSymbol = (ISymbol)symd;
                pRender.UseDefaultSymbol = true;
                IFeature pFeat = null;
                n = pFeatCls.FeatureCount(pQueryFilter);

                int i = 0;
                bool ValFound = false;
                int uh = 0;
                IFields pFields = null;
                int iField = 0;

                pFields = pFeatCursor.Fields;
                ESRI.ArcGIS.Display.ISimpleFillSymbol symx = null;
                string X = null;
                while (!(i == n)) {
                    symx = new SimpleFillSymbol();
                    symx.Style = esriSimpleFillStyle.esriSFSSolid;
                    symx.Outline.Width = 0.4;
                    pFeat = pFeatCursor.NextFeature();
                    X = "";
                    iField = pFields.FindField("BMP_Combin");
                    X = X + "," + pFeat.Value[iField];
                    //*new Cory*
                    X = X.Substring(1);
                    // Test to see if we've already added this value to the renderer, if not, then add it.
                    ValFound = false;
                    for (uh = 0; uh <= (pRender.ValueCount - 1); uh++) {
                        if (pRender.Value[uh] == X) {
                            break;
                        }
                    }
                    if (!ValFound) {
                        pRender.AddValue(X, pHeading, (ISymbol)symx);
                        pRender.Label[X] = X;
                        pRender.Symbol[X] = (ISymbol)symx;
                    }

                    i = i + 1;
                }
                /// now that we know how many unique values there are
                /// we can size the color ramp and assign the colors.
                rx.Size = pRender.ValueCount + 1;
                bool bOK = true;
                rx.CreateRamp(out bOK);
                IEnumColors RColors = null;
                int ny = 0;
                RColors = rx.Colors;
                RColors.Reset();
                string xv = null;
                ISimpleFillSymbol jsy = null;
                for (ny = 0; ny <= (pRender.ValueCount - 1); ny++) {

                    xv = pRender.Value[ny];

                    if (!string.IsNullOrEmpty(xv)) {

                        jsy = (ISimpleFillSymbol)pRender.Symbol[xv];

                        jsy.Color = RColors.Next();

                        pRender.Symbol[xv] = (ISymbol)jsy;

                    }
                }

                /// If you didn't use a color ramp that was predefined in a style, you need to use "Custom" here, 
                /// otherwise use the name of the color ramp you chose.
                pRender.ColorScheme = "Custom";
                pRender.FieldType[0] = true;

                pLyr.Renderer = (IFeatureRenderer)pRender;

                /// This makes the layer properties symbology tab show
                /// show the correct interface.
                IRendererPropertyPage hx = (IRendererPropertyPage)new UniqueValuePropertyPage();
                pLyr.RendererPropertyPageClassID = hx.ClassID;
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at RenderUniqueValueFillSymbol_ST: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at RenderUniqueValueFillSymbol_ST: " + e.Message);
            }
            finally
            {
                if (pLyr != null)
                    Marshal.FinalReleaseComObject(pLyr );
                if (pFeatClass != null)
                    Marshal.FinalReleaseComObject(pFeatClass);
                if (pField != null)
                    Marshal.FinalReleaseComObject(pField);
                if (pFieldEdit != null)
                    Marshal.FinalReleaseComObject(pFieldEdit);
            }

        }

        public static IWorkspace GetWorkspace(string sPath)
        {
            IWorkspace returnValue = null;

            /// This function returns a shapefile workspace object  given the path.
            /// The path needs to contain shape files
            IWorkspaceFactory pWSF = null;
            try
            {
                pWSF = new ESRI.ArcGIS.DataSourcesFile.ShapefileWorkspaceFactory();
                returnValue = pWSF.OpenFromFile(sPath, 0);
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at GetWorkspace: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at GetWorkspace: " + e.Message);
            }
            return returnValue;

        }

        ///  parse expression in siting criteria frame
        public static string Parse_Expression(string strExp, bool pParse, double cellSize)
        {
            string returnValue = null;


            string strTmp = null;
            int iCnt = 0;
            try
            {
                strTmp = strExp.Substring(iCnt, 1);
                while (!IsNumeric(strTmp))
                {
                    returnValue = returnValue + strTmp;
                    iCnt = iCnt + 1;
                    strTmp = strExp.Substring(iCnt, 1);
                }
            
                if (pParse)
                {
                    returnValue = returnValue + " " + Convert.ToDouble((strExp.Substring(iCnt)).Trim()) / (2.295675E-05 * cellSize * cellSize);
                }
                else
                {
                    returnValue = returnValue + " " + (strExp.Substring(iCnt)).Trim();
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Parse_Expression: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Parse_Expression: " + e.Message);
            }

            return returnValue;

        }



        /// <summary>
        /// parse expression in weight detail frame. before and after >, <, >=, <=, add space  i.e. x>4 to x > 4, otherwise it will return syntax error.
        /// </summary>
        /// <param name="strExp">eg: x > 4</param>
        /// <param name="pParse">if pParse is true, recalculate the grid value</param>
        /// <param name="cellSize"></param>
        /// <returns></returns>
        public static string Parse_Expression2(string strExp, bool pParse, double cellSize)
        {
            string returnValue = null;


            string strTmp = null;
            string strTmp2 = null;
            string strTmpNum = null;
            bool numBegin = false;
            try
            {
                for (int iCnt = 0; iCnt < strExp.Length; iCnt++)
                {

                    strTmp = strExp.Substring(iCnt, 1);
                    if (strTmp == "<" | strTmp == ">" | strTmp == "=")
                    {

                        strTmp2 = strExp.Substring(iCnt + 1, 1);
                        if (strTmp2 == "=")
                        {
                            returnValue = returnValue + strTmp;
                        }
                        else
                        {
                            returnValue = returnValue + strTmp;
                        }

                    }
                    else if (!IsNumeric(strTmp))
                    {
                        if (numBegin == true)
                        {
                            if (pParse)
                            {
                                returnValue = returnValue + " " + Convert.ToDouble(strTmpNum) / (2.295675E-05 * cellSize * cellSize) + strTmp;
                            }
                            else
                            {
                                returnValue = returnValue + " " + strTmpNum + strTmp;
                            }
                        }
                        else
                        {
                            returnValue = returnValue + strTmp;
                        }

                    }

                    if (IsNumeric(strTmp))
                    {
                        numBegin = true;
                        strTmpNum = strTmpNum + strTmp;
                    }
                    else
                    {
                        numBegin = false;
                        strTmpNum = "";
                    }

                }

                //if strTmpNum is not empty
                if (!string.IsNullOrEmpty(strTmpNum))
                {
                    returnValue = returnValue + strTmpNum;
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("Error at Parse_Expression2: " + e.Message, BMPSitingTool.Resource.BMPSitingTitle, MessageBoxButtons.OK, MessageBoxIcon.Information);
                Logger.WriteToLog(BMPSitingTool.SitingToolBar.LOGFILE, "Error at Parse_Expression2: " + e.Message);
            }

            return returnValue;

        }
        public static Boolean IsNumeric (System.Object Expression)
        {
            if(Expression == null || Expression is DateTime)
                return false;

            if(Expression is Int16 || Expression is Int32 || Expression is Int64 || Expression is Decimal || Expression is Single || Expression is Double || Expression is Boolean)
                return true;
   
            try 
            {
                if(Expression is string)
                    Double.Parse(Expression as string);
                else
                    Double.Parse(Expression.ToString());
                    return true;
            } catch {} // just dismiss errors but return false
            return false;
        }

        public static string Generic_Trim(string strSearch, char sFind)
        {
            string returnValue = null;

            strSearch = strSearch.TrimEnd(sFind);
            strSearch = strSearch.TrimStart(sFind);
            returnValue = strSearch;
            return returnValue;

        }
	}
}
