function [ engine_out ] = REVS_modify_engine( engine_in, varargin )
%REVS_modify_engine takes in an engine structure and its displacement(L), apply engine modifiers,
%   and returns the result in a new engine structure.
%
%   [ engine_new ] = REVS_modify_engine( engine, varargin )
%
%   Avaiable engine modifiers can be applied by varargin as follows:
%   Scale engine: 'scale_factor', n
%               where n = NewSize/OrigSize
%   Add DCP: '+DCP';            Remove DCP: '-DCP'
%   Add CCP: '+CCP';            Remove CCP: '-CCP'
%   Add DCP_DVVL: '+DCP_DVVL';  Remove DCP_DVVL: '-DCP_DVVL'
%   Add CCP: '+DCP_CVVL';       Remove CCP: '-DCP_CVVL'
%   Add CCP: '+CCP_DVVL';       Remove CCP: '-CCP_DVVL'
%   Add CCP: '+Camless';        Remove CCP: '-Camless'
%   Add CCP: '+CylDeact';       Remove CCP: '-CylDeact'
%   Add CCP: '+GDI';            Remove CCP: '-GDI'
%   Add CCP: '+HCCI';           Remove CCP: '-HCCI'
%
 %scale_penalty = parse_varargs( varargin, 'no_scale_fuel_adjust', true, 'toggle' );
% upscale_adjust = parse_varargs( varargin, 'upscale_fuel_adjust', false, 'toggle' );


%% Parse varargs
% Check if enough varargin was passed
narginchk( 1, inf );

% Display the varargin variable input
%disp( varargin );

% Parse varargs

scale_penalty = parse_varargs( varargin, 'no_scale_fuel_adjust', true, 'toggle' );
upscale_adjust = parse_varargs( varargin, 'upscale_fuel_adjust', false, 'toggle' );

% Scale engine size: scale_factor = NewSize/OrigSize (scale_factor > 0)
scale_factor = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
%disp('scale_factor:');disp( scale_factor );

% Scale maximum power, e.g. 120->200 kW
scale_max_power = parse_varargs( varargin, 'scale_max_power',[], 'numeric' );
scale_avg_power = parse_varargs( varargin, 'scale_avg_power',[], 'numeric' );
scale_max_torque = parse_varargs( varargin, 'scale_max_torque',[], 'numeric' );
scale_avg_torque = parse_varargs( varargin, 'scale_avg_torque',[], 'numeric' );

averaging_window = parse_varargs( varargin, 'avg_window',linspace(1800, 6000, 25)* convert.rpm2radps, 'numeric' );

% Fill in if only window end points provided
if numel(averaging_window) ==2
	averaging_window = linspace(averaging_window(1), averaging_window(end), 25);
end


% Add DCP
AddDCP = parse_varargs( varargin, '+DCP', false, 'toggle' );
%disp('+DCP:');disp( AddDCP );
% Remove DCP
RmvDCP = parse_varargs( varargin, '-DCP', false, 'toggle' );
%disp('-DCP:');disp( RmvDCP );

% Add CCP
AddCCP = parse_varargs( varargin, '+CCP', false, 'toggle' );
%disp('+CCP:');disp( AddCCP );
% Remove CCP
RmvCCP = parse_varargs( varargin, '-CCP', false, 'toggle' );
%disp('-CCP:');disp( RmvCCP );

% Add DCP_DVVL
AddDCP_DVVL = parse_varargs( varargin, '+DCP_DVVL', false, 'toggle' );
%disp('+DCP_DVVL:');disp( AddDCP_DVVL );
% Remove DCP_DVVL
RmvDCP_DVVL = parse_varargs( varargin, '-DCP_DVVL', false, 'toggle' );
%disp('-DCP_DVVL:');disp( RmvDCP_DVVL );

% Add DCP_CVVL
AddDCP_CVVL = parse_varargs( varargin, '+DCP_CVVL', false, 'toggle' );
%disp('+DCP_CVVL:');disp( AddDCP_CVVL );
% Remove DCP_CVVL
RmvDCP_CVVL = parse_varargs( varargin, '-DCP_CVVL', false, 'toggle' );
%disp('-DCP_CVVL:');disp( RmvDCP_CVVL );

% Add CCP_DVVL
AddCCP_DVVL = parse_varargs( varargin, '+CCP_DVVL', false, 'toggle' );
%disp('+CCP_DVVL:');disp( AddCCP_DVVL );
% Remove CCP_DVVL
RmvCCP_DVVL = parse_varargs( varargin, '-CCP_DVVL', false, 'toggle' );
%disp('-CCP_DVVL:');disp( RmvCCP_DVVL );

% Add Camless
AddCamless = parse_varargs( varargin, '+Camless', false, 'toggle' );
%disp('+Camless:');disp( AddCamless );
% Remove Camless
RmvCamless = parse_varargs( varargin, '-Camless', false, 'toggle' );
%disp('-Camless:');disp( RmvCamless );

% Add CylDeact
AddCylDeact = parse_varargs( varargin, '+CylDeact', false, 'toggle' );
%disp('+CylDeact:');disp( AddCylDeact );
% Remove CylDeact
RmvCylDeact = parse_varargs( varargin, '-CylDeact', false, 'toggle' );
%disp('-CylDeact:');disp( RmvCylDeact );

% Add GDI
AddGDI = parse_varargs( varargin, '+GDI', false, 'toggle' );
%disp('+GDI:');disp( AddGDI );
% Remove GDI
RmvGDI = parse_varargs( varargin, '-GDI', false, 'toggle' );
%disp('-GDI:');disp( RmvGDI );

% Add HCCI
AddHCCI = parse_varargs( varargin, '+HCCI', false, 'toggle' );
%disp('+HCCI:');disp( AddHCCI );
% Remove HCCI
RmvHCCI = parse_varargs( varargin, '-HCCI', false, 'toggle' );
%disp('-HCCI:');disp( RmvHCCI );


%% Copy and create a new engine structure
engine_out = engine_in;


%% 1: Scale torque and fuel map based on Ricardo scaling factors as outlined
% in the PDF about scaling.
ricardo_bsfc_sc_coef = [-0.1058 0.3458 -0.4564 (0.2168-0.000400000000000011)];

engine_torque = interp1( engine_in.full_throttle_speed_radps, engine_in.full_throttle_torque_Nm, averaging_window,'linear','extrap' );
engine_power = engine_torque .* averaging_window;

engine_average_torque = sum( engine_torque)./ length(averaging_window);
engine_average_power = sum( engine_power )./ length(averaging_window);

if ~isempty( scale_max_power )
	scale_factor = scale_max_power / engine_in.max_power_W;
end

if ~isempty( scale_max_torque )
	scale_factor = scale_max_torque / engine_in.max_torque_Nm;
end

if ~isempty( scale_avg_power )
	scale_factor = scale_avg_power / engine_average_power;
end

if ~isempty( scale_avg_torque )
	scale_factor = scale_avg_torque / engine_average_torque;
end

if  scale_factor ~= 1
	
	if (scale_factor<0.5 || scale_factor>2 )
		warning('Engine Scale factor of %f extrapolates scaling curve and may yield erroneous results!', scale_factor);
	end
	
	% Calculate torque
	engine_out.displacement_L					= engine_out.displacement_L * (scale_factor);
	engine_out.fuel_map_torque_Nm				= engine_out.fuel_map_torque_Nm .* (scale_factor);   %for BSFC map
	engine_out.full_throttle_torque_Nm			= engine_out.full_throttle_torque_Nm  .* (scale_factor); %for torque curve
	engine_out.closed_throttle_torque_Nm		= engine_out.closed_throttle_torque_Nm  .* (scale_factor); %for torque curve
	engine_out.naturally_aspirated_torque_Nm	= engine_out.naturally_aspirated_torque_Nm  .* (scale_factor); %for torque curve	
    engine_out.inertia_kgm2						= engine_out.inertia_kgm2 .* (scale_factor);
	

	
%% from REVS1:    
        % Calculate fuel flow rate
     if (scale_factor > 1 && ~upscale_adjust) || ~scale_penalty 
%         bsfc_increase_pcnt = polyval(ricardo_bsfc_sc_coef, 1/scale_factor);
%         engine_out.fuel_map_gps = engine_out.fuel_map_gps .* (scale_factor) ./(1+bsfc_increase_pcnt);
    	engine_out.fuel_map_gps = engine_out.fuel_map_gps .*(scale_factor);
     else
         bsfc_increase_pcnt = polyval(ricardo_bsfc_sc_coef, scale_factor);
         engine_out.fuel_map_gps = engine_out.fuel_map_gps .*(scale_factor) .* (1+bsfc_increase_pcnt);
     end

    
end


%% 2: Add/Remove technology: DCP
if (AddDCP==1 || RmvDCP==1)
	%     './data/DCP_20150601_2.mat'
	map_data = [1626.17355100000,2.72921500000000,6.50000000000000;2365.97196600000,2.69907600000000,6.50000000000000;2435.52566300000,2.15658800000000,6.50000000000000;2157.31087500000,2.32234800000000,6.50000000000000;1632.49661500000,2.33741700000000,6.50000000000000;2011.88041700000,2.69907600000000,6.50000000000000;0,2.01526700000000,6;1470.08547000000,3.00763400000000,6;2484.07148400000,3.00763400000000,6;2651.12665100000,2.01526700000000,6;2515.15151500000,1.83206100000000,6;1975.13597500000,2.01526700000000,6;2541.61810400000,2.67438000000000,6;2602.80691500000,2.34899800000000,6;1984.12004700000,3.01455200000000,6;1722.36791000000,3.01455200000000,6;2228.87529100000,3.01455200000000,6;2126.89393900000,1.93487600000000,6;2347.85353500000,1.86092600000000,6;1613.58780100000,2.02361600000000,6;1800.55361300000,2.02361600000000,6;1470.81390800000,2.52647900000000,6;1477.85547800000,3.51145000000000,5.50000000000000;2484.07148400000,3.51145000000000,5.50000000000000;2740.48174000000,3.00763400000000,5.50000000000000;2825.95182600000,2.03053400000000,5.50000000000000;2484.07148400000,1.63358800000000,5.50000000000000;1986.79098700000,1.75572500000000,5.50000000000000;1477.85547800000,1.75572500000000,5.50000000000000;1357.42035700000,2.03053400000000,5.50000000000000;1357.42035700000,2.91603100000000,5.50000000000000;1708.77039600000,3.51741400000000,5.50000000000000;1977.32129000000,3.50262400000000,5.50000000000000;2228.87529100000,3.51741400000000,5.50000000000000;2623.20318600000,3.25119300000000,5.50000000000000;2789.77272700000,2.46731900000000,5.50000000000000;2650.39821300000,1.80176500000000,5.50000000000000;2232.27467000000,1.68344500000000,5.50000000000000;1729.16666700000,1.75739500000000,5.50000000000000;1473.97047400000,4,5;2985.23698500000,4,5;2985.23698500000,2.01526700000000,5;2526.80652700000,1.43511500000000,5;2060.60606100000,1.48091600000000,5;1815.85081600000,1.52671800000000,5;1477.85547800000,1.51145000000000,5;1225.33022500000,2,5;1225.33022500000,3.00763400000000,5;1334.83877200000,3.45825400000000,5;1219.25990700000,2.52647900000000,5;1345.03690800000,1.75739500000000,5;2317.25913000000,1.46159400000000,5;2820.36713300000,1.80176500000000,5;2657.19697000000,1.60949400000000,5;2990.33605300000,3.38430300000000,5;2990.33605300000,2.67438000000000,5;2691.19075400000,4.00548700000000,5;2398.84421100000,4.00548700000000,5;2065.70512800000,4.00548700000000,5;1807.35237000000,4.00548700000000,5;1225.33022500000,4,4.50000000000000;1477.85547800000,4.48855000000000,4.50000000000000;1982.90598300000,4.97709900000000,4.50000000000000;2452.99145300000,4.50381700000000,4.50000000000000;2985.23698500000,4.48855000000000,4.50000000000000;3121.21212100000,3.98473300000000,4.50000000000000;3241.64724200000,3.02290100000000,4.50000000000000;3245.53224600000,2,4.50000000000000;2985.23698500000,1.49618300000000,4.50000000000000;2557.88655800000,1.22137400000000,4.50000000000000;1982.90598300000,1.23664100000000,4.50000000000000;1477.85547800000,1.25190800000000,4.50000000000000;1108.78010900000,2,4.50000000000000;1108.78010900000,3.06870200000000,4.50000000000000;1103.68104100000,2.51168900000000,4.50000000000000;1239.65617700000,1.74260500000000,4.50000000000000;1732.56604500000,1.25453200000000,4.50000000000000;2279.86596700000,1.23974200000000,4.50000000000000;2786.37334900000,1.37285300000000,4.50000000000000;2728.58391600000,4.50834900000000,4.50000000000000;2235.67404800000,4.74499000000000,4.50000000000000;1742.76418000000,4.73020000000000,4.50000000000000;3116.11305400000,1.74260500000000,4.50000000000000;1171.66860900000,3.54699400000000,4.50000000000000;3181.64714100000,3.49774100000000,4.50000000000000;3244.87777500000,2.45797000000000,4.50000000000000;1000,4.99236600000000,4;1528.36052800000,4.99236600000000,4;1994.56099500000,5.32824400000000,4;2433.56643400000,4.99236600000000,4;2981.35198100000,4.99236600000000,4;3490.28749000000,3.00763400000000,4;3490.28749000000,1.02290100000000,4;1003.88500400000,1.02290100000000,4;1154.67171700000,4.99642200000000,4;1331.43939400000,4.99642200000000,4;1232.85742000000,1.01789100000000,4;1464.01515200000,1.00310100000000,4;1647.58158500000,1.01789100000000,4;1905.93434300000,1.01789100000000,4;2174.48523700000,1.01789100000000,4;2426.03923900000,1.01789100000000,4;2630.00194300000,1.01789100000000,4;2850.96153800000,1.00310100000000,4;3054.92424200000,1.00310100000000,4;3262.28632500000,1.00310100000000,4;3490.04467800000,1.62428400000000,4;3490.04467800000,2.33420800000000,4;3381.26456900000,3.44346400000000,4;3207.89627000000,4.13859700000000,4;2789.77272700000,4.99642200000000,4;2616.40442900000,4.99642200000000,4;2222.07653500000,5.15911300000000,4;1831.14801900000,5.21827300000000,4;1664.57847700000,5.09995200000000,4;987.544151000000,4.31147400000000,4;993.867214000000,3.54294800000000,4;993.867214000000,2.68400700000000,4;993.867214000000,1.77985900000000,4;993.867214000000,4.91423900000000,4;993.867214000000,1.13188600000000,4;993.867214000000,4.98958500000000,4;993.867214000000,1.01133300000000,4;993.867214000000,4.85396300000000,4;993.867214000000,4.91423900000000,4;993.867214000000,2.08124200000000,4;993.867214000000,1.91548100000000,4;1000,5.26717600000000,3.50000000000000;1501.16550100000,5.25190800000000,3.50000000000000;1986.79098700000,5.66412200000000,3.50000000000000;2449.10644900000,5.26717600000000,3.50000000000000;3000.77700100000,5.25190800000000,3.50000000000000;3606.83760700000,3.03816800000000,3.50000000000000;3630.14763000000,2.54961800000000,3.50000000000000;3661.22766100000,2.10687000000000,3.50000000000000;3668.99766900000,1.89313000000000,3.50000000000000;3742.81274300000,1.05343500000000,3.50000000000000;3509.71251000000,0.763359000000000,3.50000000000000;1000,0.748092000000000,3.50000000000000;3143.70876100000,4.91423900000000,3.50000000000000;3377.66210600000,3.99502200000000,3.50000000000000;2897.10928900000,0.755158000000000,3.50000000000000;2214.21844500000,0.755158000000000,3.50000000000000;1645.14274100000,0.755158000000000,3.50000000000000;1151.94379800000,5.26082900000000,3.50000000000000;1328.98957300000,5.26082900000000,3.50000000000000;1177.23605200000,0.740089000000000,3.50000000000000;1423.83552300000,0.740089000000000,3.50000000000000;1000,5.49618300000000,3;1458.43045800000,5.49618300000000,3;1975.13597500000,6.03053400000000,3;2484.07148400000,5.49618300000000,3;2996.89199700000,5.49618300000000,3;3249.41724900000,4.97709900000000,3;3746.69774700000,3.03816800000000,3;3781.66278200000,2.68702300000000,3;3797.20279700000,2.27480900000000,3;3983.68298400000,1.03816800000000,3;3521.36752100000,0.519084000000000,3;992.229992000000,0.519084000000000,3;3504.12337300000,3.94981500000000,3;3889.83023900000,1.64423700000000,3;3763.36897200000,0.785296000000000,3;2972.98605000000,0.514052000000000,3;2384.94115600000,0.514052000000000,3;1904.38834000000,0.514052000000000,3;1449.12777700000,0.514052000000000,3;1158.26686200000,5.48686600000000,3;1328.98957300000,5.48686600000000,3;1739.98869200000,5.77318000000000,3;2245.83376200000,5.72797200000000,3;2751.67883200000,5.48686600000000,3;1000,5.77099200000000,2.50000000000000;1485.62548600000,5.78626000000000,2.50000000000000;1998.44599800000,6.25954200000000,2.50000000000000;2484.07148400000,5.75572500000000,2.50000000000000;2993.00699300000,5.75572500000000,2.50000000000000;3365.96736600000,5.00763400000000,2.50000000000000;3855.47785500000,3.08396900000000,2.50000000000000;4111.88811200000,1.03816800000000,2.50000000000000;3991.45299100000,0.488550000000000,2.50000000000000;3630.14763000000,0.290076000000000,2.50000000000000;996.114996000000,0.259542000000000,2.50000000000000;1240.46668600000,5.75811100000000,2.50000000000000;1746.31175500000,6.01428600000000,2.50000000000000;2245.83376200000,5.98414800000000,2.50000000000000;2751.67883200000,5.75811100000000,2.50000000000000;3561.03094400000,4.23612800000000,2.50000000000000;3940.41474600000,2.39769400000000,2.50000000000000;4022.61457000000,1.73465200000000,2.50000000000000;3769.69203500000,0.408568000000000,2.50000000000000;992.229992000000,6.03053400000000,2;1505.05050500000,6.03053400000000,2;1986.79098700000,6.51908400000000,2;2476.30147600000,6.03053400000000,2;2989.12198900000,6.01526700000000,2;3494.17249400000,4.97709900000000,2;3999.22299900000,3.00763400000000,2;4243.97824400000,0.992366000000000,2;4500.38850000000,-0.0152670000000000,2;1158.26686200000,6.02935500000000,2;1752.63481900000,6.30060000000000,2;2245.83376200000,6.25539200000000,2;2720.06351500000,6.02935500000000,2;3244.87777500000,5.51700500000000,2;3662.19995800000,4.32654300000000,2;3839.24573200000,3.63336300000000,2;4079.52214000000,2.15658800000000,2;4180.69115400000,1.52368400000000,2;4370.38305600000,0.514052000000000,2;4433.61368900000,0.272946000000000,2;3086.80119100000,5.81838700000000,2;2840.20171900000,6.02935500000000,2;4313.47548500000,0.755158000000000,2;4472.46503500000,0.130487000000000,2;992.229992000000,6.25954200000000,1.50000000000000;1497.28049700000,6.25954200000000,1.50000000000000;1979.02097900000,6.76335900000000,1.50000000000000;2437.45143700000,6.53435100000000,1.50000000000000;2985.23698500000,6.51908400000000,1.50000000000000;3513.59751400000,5.40458000000000,1.50000000000000;4115.77311600000,2.99236600000000,1.50000000000000;4212.89821300000,1.92366400000000,1.50000000000000;4376.06837600000,0.992366000000000,1.50000000000000;4616.93861700000,-0.0152670000000000,1.50000000000000;1161.47047400000,6.25357800000000,1.50000000000000;1331.43939400000,6.25357800000000,1.50000000000000;1746.16355900000,6.50501000000000,1.50000000000000;2171.08585900000,6.65291000000000,1.50000000000000;2670.79448300000,6.51980000000000,1.50000000000000;3252.08819000000,6.00214700000000,1.50000000000000;3728.00116600000,4.56751000000000,1.50000000000000;3925.16511300000,3.78363500000000,1.50000000000000;4166.52097900000,2.40815800000000,1.50000000000000;4309.29487200000,1.35806300000000,1.50000000000000;4482.66317000000,0.574189000000000,1.50000000000000;4557.44949500000,0.278387000000000,1.50000000000000;4431.67249400000,0.781250000000000,1.50000000000000;988.344988000000,6.50381700000000,1;1501.16550100000,6.53435100000000,1;1994.56099500000,7.02290100000000,1;2985.23698500000,7.02290100000000,1;3509.71251000000,6,1;4240.09324000000,3.05343500000000,1;4298.36829800000,1.98473300000000,1;4756.79875700000,-0.0305340000000000,1;1158.07109600000,6.51980000000000,1;1375.63131300000,6.51980000000000,1;1746.16355900000,6.78602100000000,1;2249.27156200000,7.02266200000000,1;2670.79448300000,7.03745200000000,1;3173.90248600000,6.65291000000000,1;3377.86519000000,6.25357800000000,1;3660.01359800000,5.33659400000000,1;3836.78127400000,4.64146000000000,1;4044.14335700000,3.84279600000000,1;4275.30108800000,2.54126900000000,1;4411.27622400000,1.46159400000000,1;4526.85508900000,0.899571000000000,1;4669.62898200000,0.337548000000000,1;992.229992000000,6.76335900000000,0.500000000000000;1505.05050500000,6.76335900000000,0.500000000000000;1745.92074600000,7.05343500000000,0.500000000000000;1979.02097900000,7.48091600000000,0.500000000000000;2993.00699300000,7.48091600000000,0.500000000000000;3754.46775400000,6,0.500000000000000;3863.24786300000,5.03816800000000,0.500000000000000;4003.10800300000,4.50381700000000,0.500000000000000;4251.74825200000,3.98473300000000,0.500000000000000;4364.41336400000,3.09923700000000,0.500000000000000;4399.37839900000,2.01526700000000,0.500000000000000;4527.58352800000,1.45038200000000,0.500000000000000;4764.56876500000,0.992366000000000,0.500000000000000;4873.34887300000,-0.0152670000000000,0.500000000000000;1246.78974900000,6.75267400000000,0.500000000000000;2372.29502900000,7.49106100000000,0.500000000000000;2675.80207100000,7.47599200000000,0.500000000000000;3289.13921900000,6.93350300000000,0.500000000000000;3510.44643700000,6.49649800000000,0.500000000000000;4819.32055500000,0.468845000000000,0.500000000000000;4383.02918200000,2.66893800000000,0.500000000000000;1117.27855500000,6.75644100000000,0.500000000000000;1409.62509700000,6.74165100000000,0.500000000000000;1623.78593600000,6.88955200000000,0.500000000000000;1831.14801900000,7.21493300000000,0.500000000000000;2171.08585900000,7.46636500000000,0.500000000000000;3170.50310800000,7.15577300000000,0.500000000000000;3809.58624700000,5.48449400000000,0.500000000000000;3632.81857000000,6.23878800000000,0.500000000000000;4316.09362900000,3.50262400000000,0.500000000000000;4159.72222200000,4.15338700000000,0.500000000000000;4081.53651900000,4.33086800000000,0.500000000000000;4472.46503500000,1.71302500000000,0.500000000000000;4384.08119700000,2.33420800000000,0.500000000000000;4666.22960400000,1.18058200000000,0.500000000000000;4601.64141400000,1.32848300000000,0.500000000000000;4792.00660500000,0.707300000000000,0.500000000000000;4846.39665900000,0.219227000000000,0.500000000000000;986.787322000000,7.32530100000000,0;1284.64983700000,7.31023200000000,0;1437.04461200000,7.29516200000000,0;1818.03154900000,7.89792800000000,0;2192.09145200000,8.01848100000000,0;3023.33567900000,7.98834300000000,0;3480.52000400000,7.18967800000000,0;3972.33950500000,6.13483900000000,0;4062.39096300000,4.91423900000000,0;4484.94011200000,4.19092100000000,0;4574.99157000000,3.06073600000000,0;4665.04302800000,2.03603400000000,0;4983.68664900000,1.22230100000000,0;5115.30031800000,-0.0284370000000000,0;6008.88786300000,-0.0284370000000000,0;6008.88786300000,11.1829990000000,0;986.787322000000,11.1829990000000,0;1645.37357800000,7.42152900000000,0;800,7.10000000000000,0;800,9,0;800,11,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddDCP, RmvDCP, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/DCP.png'));
    title('Ricardo DCP Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm, bsfc_improve_pcnt, [0:0.5:9.5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('Torque (Nm)');
    xlim([800 6000]);
    ylim([0 210/2.4*3.8]);
	%}
end


%% 3: Add/Remove technology: CCP
if (AddCCP==1 || RmvCCP==1)
	
	%     './data/CCP_20150603_2.mat'
	map_data = [1547.39428700000,3.77186600000000,5.50000000000000;1533.83692200000,2.32486700000000,5.50000000000000;1960.89390700000,2.32486700000000,5.50000000000000;2452.34837400000,1.95573500000000,5.50000000000000;2608.25806700000,2.13291800000000,5.50000000000000;2462.51639800000,2.75306100000000,5.50000000000000;1981.22995400000,3.71280500000000,5.50000000000000;1472.82878200000,4.02287600000000,5;1988.00863700000,4.03764100000000,5;2499.79915000000,3.01883600000000,5;2740.44237200000,2.05909200000000,5;2486.24178600000,1.71949000000000,5;1977.84061300000,2.04432700000000,5;1476.21812300000,2.05909200000000,5;1472.82878200000,3.38796800000000,5;1472.82878200000,2.67923400000000,5;1611.79176900000,4.03764100000000,5;1815.15223800000,4.03764100000000,5;2174.42240000000,3.65374400000000,5;2343.88945700000,3.31414200000000,5;2625.20477300000,2.53158100000000,5;2604.86872600000,1.85237800000000,5;2367.61484500000,1.76378600000000,5;2177.81174100000,1.89667400000000,5;1642.29583900000,2.04432700000000,5;1801.59487300000,2.05909200000000,5;1228.79621900000,4.02287600000000,4.50000000000000;1482.99680500000,4.51013100000000,4.50000000000000;1984.61929500000,5.01215100000000,4.50000000000000;2482.85244400000,4.02287600000000,4.50000000000000;2994.64295800000,4.02287600000000,4.50000000000000;2994.64295800000,2.05909200000000,4.50000000000000;2493.02046800000,1.36512300000000,4.50000000000000;1984.61929500000,1.54230700000000,4.50000000000000;1482.99680500000,1.54230700000000,4.50000000000000;1228.79621900000,2.02956100000000,4.50000000000000;1225.40687800000,3.35843800000000,4.50000000000000;1222.01753700000,2.67923400000000,4.50000000000000;1354.20184200000,1.76378600000000,4.50000000000000;1727.02936800000,1.54230700000000,4.50000000000000;2232.04119900000,1.42418400000000,4.50000000000000;2662.48752500000,1.60136800000000,4.50000000000000;2821.78655900000,1.77855100000000,4.50000000000000;2994.64295800000,2.64970400000000,4.50000000000000;2998.03229900000,3.40273400000000,4.50000000000000;2642.15147800000,4.02287600000000,4.50000000000000;2825.17590100000,4.02287600000000,4.50000000000000;2323.55341000000,4.36247800000000,4.50000000000000;2150.69701200000,4.70207900000000,4.50000000000000;1652.46386300000,4.68731400000000,4.50000000000000;1815.15223800000,4.83496700000000,4.50000000000000;994.931680000000,4.03764100000000,4;1482.99680500000,5.01215100000000,4;1981.22995400000,5.32222200000000,4;2252.37724600000,4.98262000000000,4;2479.46310300000,4.52489600000000,4;2991.25361700000,4.51013100000000,4;3177.66738000000,3.91951900000000,4;3245.45420300000,3.10742800000000,4;3245.45420300000,2.00003100000000,4;2984.47493500000,1.04028700000000,4;1466.05009900000,1.05505200000000,4;1005.09970300000,2.00003100000000,4;1171.17741900000,4.39200800000000,4;1323.69777100000,4.71684500000000,4;1662.63188600000,5.14503800000000,4;1832.09894400000,5.23363000000000,4;2167.64371700000,5.11550800000000,4;2638.76213700000,4.52489600000000,4;2825.17590100000,4.52489600000000,4;3092.93385100000,4.22959000000000,4;3211.56079200000,3.50609100000000,4;3248.84354400000,2.53158100000000,4;3160.72067400000,1.67519400000000,4;3082.76582800000,1.36512300000000,4;1350.81250000000,1.27653100000000,4;1194.90280800000,1.60136800000000,4;1716.86134500000,1.04028700000000,4;2089.68887100000,1.04028700000000,4;2425.23364500000,1.04028700000000,4;2747.22105400000,1.04028700000000,4;1001.71036200000,5.02691600000000,3.50000000000000;1191.51346600000,5.10074200000000,3.50000000000000;1496.55417000000,5.26316100000000,3.50000000000000;1981.22995400000,5.66182400000000,3.50000000000000;2469.29508000000,5.02691600000000,3.50000000000000;2987.86427600000,5.02691600000000,3.50000000000000;3347.13443800000,3.94904900000000,3.50000000000000;3492.87610700000,3.03360100000000,3.50000000000000;3492.87610700000,1.02552100000000,3.50000000000000;3055.65109900000,0.804042000000000,3.50000000000000;1466.05009900000,0.774511000000000,3.50000000000000;1001.71036200000,0.995991000000000,3.50000000000000;1750.75475600000,5.48464000000000,3.50000000000000;2167.64371700000,5.44034400000000,3.50000000000000;2306.60670500000,5.24839500000000,3.50000000000000;2638.76213700000,5.02691600000000,3.50000000000000;2808.22919500000,5.02691600000000,3.50000000000000;3109.88055700000,4.68731400000000,3.50000000000000;3238.67552100000,4.30341700000000,3.50000000000000;3418.31060200000,3.49132500000000,3.50000000000000;3496.26544800000,2.39869400000000,3.50000000000000;3499.65478900000,1.68996000000000,3.50000000000000;3340.35575500000,0.951695000000000,3.50000000000000;3191.22474500000,0.848338000000000,3.50000000000000;2801.45051200000,0.774511000000000,3.50000000000000;2252.37724600000,0.774511000000000,3.50000000000000;1832.09894400000,0.774511000000000,3.50000000000000;1320.30843000000,0.848338000000000,3.50000000000000;1164.39873700000,0.922164000000000,3.50000000000000;1005.09970300000,5.35175200000000,3;1415.20998200000,5.46987500000000,3;1984.61929500000,6.04572100000000,3;2472.68442100000,5.35175200000000,3;2991.25361700000,5.35175200000000,3;3275.95827300000,4.64301800000000,3;3516.60149500000,3.94904900000000,3;3645.39645900000,3.15172400000000,3;3713.18328200000,2.50205100000000,3;3730.12998800000,2.14768400000000,3;3997.88793800000,1.04028700000000,3;3391.19587300000,0.656389000000000,3;2974.30691100000,0.553032000000000,3;1550.78362800000,0.553032000000000,3;1462.66075800000,0.553032000000000,3;1005.09970300000,0.685920000000000,3;800,0.800000000000000,3;800,5.30000000000000,3;1164.39873700000,5.39604800000000,3;1289.80436000000,5.42557900000000,3;1581.28769900000,5.60276200000000,3;1750.75475600000,5.80947600000000,3;2154.08635300000,5.80947600000000,3;2313.38538700000,5.57323200000000,3;2635.37279600000,5.35175200000000,3;2821.78655900000,5.35175200000000,3;3136.99528600000,5.05644600000000,3;3397.97455500000,4.30341700000000,3;3580.99897700000,3.50609100000000,3;3821.64219900000,1.74902100000000,3;3913.15441000000,1.39465400000000,3;3852.14626900000,0.936930000000000,3;3645.39645900000,0.789277000000000,3;3187.83540400000,0.582563000000000,3;2723.49566600000,0.553032000000000,3;2428.62298600000,0.553032000000000,3;2130.36096500000,0.553032000000000,3;1828.70960300000,0.553032000000000,3;1299.97238300000,0.582563000000000,3;1137.28400800000,0.641624000000000,3;998.321021000000,5.70611900000000,2.50000000000000;1415.20998200000,5.77994600000000,2.50000000000000;1988.00863700000,6.28196600000000,2.50000000000000;2337.11077500000,5.91283400000000,2.50000000000000;2469.29508000000,5.67658900000000,2.50000000000000;2984.47493500000,5.67658900000000,2.50000000000000;3259.01156800000,5.14503800000000,2.50000000000000;3540.32688300000,4.31818200000000,2.50000000000000;3831.81022200000,2.98930500000000,2.50000000000000;4001.27727900000,2.00003100000000,2.50000000000000;4119.90422000000,1.04028700000000,2.50000000000000;3994.49859700000,0.538267000000000,2.50000000000000;3557.27358900000,0.405379000000000,2.50000000000000;3059.04044000000,0.272491000000000,2.50000000000000;1462.66075800000,0.272491000000000,2.50000000000000;1299.97238300000,0.302022000000000,2.50000000000000;1001.71036200000,0.361083000000000,2.50000000000000;1191.51346600000,5.69135400000000,2.50000000000000;1588.06638100000,5.88330300000000,2.50000000000000;1791.42685000000,6.09001700000000,2.50000000000000;2167.64371700000,6.10478200000000,2.50000000000000;2628.59411400000,5.67658900000000,2.50000000000000;2794.67183000000,5.67658900000000,2.50000000000000;3713.18328200000,3.57991700000000,2.50000000000000;3770.80208100000,0.464440000000000,2.50000000000000;3292.90497900000,0.331553000000000,2.50000000000000;2692.99159600000,0.272491000000000,2.50000000000000;2143.91832900000,0.272491000000000,2.50000000000000;1767.70146200000,0.272491000000000,2.50000000000000;998.321021000000,6.04572100000000,2;1506.72219300000,6.04572100000000,2;1747.36541500000,6.31149600000000,2;1984.61929500000,6.54774100000000,2;2452.34837400000,6.04572100000000,2;2994.64295800000,6.04572100000000,2;3516.60149500000,4.98262000000000,2;4008.05596200000,3.03360100000000,2;4123.29356100000,1.88190800000000,2;4241.92050100000,1.08458300000000,2;4502.89977000000,0.0214810000000000,2;1174.56676100000,6.04572100000000,2;1333.86579500000,6.04572100000000,2;2208.31581100000,6.32626200000000,2;2594.70070200000,6.04572100000000,2;2801.45051200000,6.04572100000000,2;3174.27803900000,5.67658900000000,2;3333.57707300000,5.33698700000000,2;3638.61777600000,4.48060000000000,2;3821.64219900000,3.75710100000000,2;4065.67476100000,2.41345900000000,2;4374.10480600000,0.538267000000000,2;998.321021000000,6.28196600000000,1.50000000000000;1516.89021700000,6.29673100000000,1.50000000000000;1984.61929500000,6.75445500000000,1.50000000000000;2421.84430400000,6.54774100000000,1.50000000000000;2994.64295800000,6.54774100000000,1.50000000000000;3516.60149500000,5.44034400000000,1.50000000000000;4113.12553700000,3.04836700000000,1.50000000000000;4201.24840700000,2.02956100000000,1.50000000000000;4363.93678300000,1.06981700000000,1.50000000000000;4618.13736900000,0.0214810000000000,1.50000000000000;1157.62005500000,6.28196600000000,1.50000000000000;1344.03381800000,6.28196600000000,1.50000000000000;1754.14409700000,6.53297600000000,1.50000000000000;2171.03305900000,6.66586300000000,1.50000000000000;2598.09004300000,6.53297600000000,1.50000000000000;2808.22919500000,6.54774100000000,1.50000000000000;3174.27803900000,6.17860900000000,1.50000000000000;3347.13443800000,5.83900700000000,1.50000000000000;3703.01525800000,4.68731400000000,1.50000000000000;3909.76506800000,3.84569200000000,1.50000000000000;4489.34240500000,0.582563000000000,1.50000000000000;4285.98193600000,1.54230700000000,1.50000000000000;1001.71036200000,6.56250600000000,1;1516.89021700000,6.54774100000000,1;1981.22995400000,7.04976100000000,1;2994.64295800000,7.04976100000000,1;3523.38017700000,5.95712900000000,1;4248.69918300000,3.04836700000000,1;4302.92864200000,2.02956100000000,1;4550.35054600000,0.848338000000000,1;4753.71101500000,0.0214810000000000,1;800,6.50000000000000,1;1164.39873700000,6.53297600000000,1;1357.59118300000,6.53297600000000,1;1669.41056800000,6.72492500000000,1;1842.26696700000,6.88734300000000,1;2198.14778800000,7.03499600000000,1;2526.91387900000,7.03499600000000,1;2821.78655900000,7.03499600000000,1;3174.27803900000,6.68062900000000,1;3364.08114300000,6.31149600000000,1;3675.90052900000,5.33698700000000,1;3869.09297500000,4.56919200000000,1;4069.06410200000,3.78663100000000,1;4438.50228800000,1.37988800000000,1;4652.03078000000,0.434910000000000,1;998.321021000000,6.76922100000000,0.500000000000000;1499.94351100000,6.78398600000000,0.500000000000000;1750.75475600000,7.06452600000000,0.500000000000000;1981.22995400000,7.50748500000000,0.500000000000000;2998.03229900000,7.50748500000000,0.500000000000000;3757.24471700000,6.01619100000000,0.500000000000000;3862.31429200000,5.10074200000000,0.500000000000000;4001.27727900000,4.52489600000000,0.500000000000000;4248.69918300000,4.02287600000000,0.500000000000000;4374.10480600000,3.06313200000000,0.500000000000000;4391.05151200000,2.73829500000000,0.500000000000000;4407.99821700000,2.00003100000000,0.500000000000000;4506.28911100000,1.52754100000000,0.500000000000000;4760.48969700000,1.01075600000000,0.500000000000000;4885.89532000000,0.00671600000000000,0.500000000000000;1157.62005500000,6.75445500000000,0.500000000000000;1337.25513600000,6.75445500000000,0.500000000000000;2171.03305900000,7.49272000000000,0.500000000000000;2520.13519700000,7.49272000000000,0.500000000000000;2784.50380700000,7.50748500000000,0.500000000000000;3177.66738000000,7.16788300000000,0.500000000000000;3428.47862500000,6.66586300000000,0.500000000000000;4126.68290200000,4.27388600000000,0.500000000000000;4818.10849700000,0.523501000000000,0.500000000000000;4635.08407400000,1.29129700000000,0.500000000000000;1000,7.06452600000000,0;1482.99680500000,7.07929200000000,0;1927.00049600000,7.75849500000000,0;3011.58966400000,7.77326000000000,0;3804.69549300000,6.23767000000000,0;3977.55189100000,5.05644600000000,0;4302.92864200000,4.33294700000000,0;4523.23581700000,2.75306100000000,0;4560.51856900000,1.86714300000000,0;4821.49783800000,1.37988800000000,0;4967.23950700000,0.00671600000000000,0;6004.37789900000,0.00671600000000000,0;6007.76724000000,11.1988090000000,0;1000,11.1840430000000,0;800,7.20000000000000,0;800,9,0;800,11,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddCCP, RmvCCP, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/CCP.png'));
    title('Ricardo CCP Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm, bsfc_improve_pcnt, [0:0.5:9.5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('Torque (Nm)');
    ylim([0 210/2.4*3.8]);
	%}
end


%% 4: Add/Remove technology: DCP_DVVL
if (AddDCP_DVVL==1 || RmvDCP_DVVL==1)
	%     './data/DCP_DVVL_20150609_2.mat'
	map_data = [1118.00404600000,-0.0137360000000000,9.50000000000000;1000,0.480769000000000,9.50000000000000;1064.05933900000,0.260989000000000,9.50000000000000;1000,0.961538000000000,9;1232.63654800000,-0.0137360000000000,9;1060.68779500000,0.686813000000000,9;1114.63250200000,0.412088000000000,9;1158.46257600000,0.260989000000000,9;1003.37154400000,1.51098900000000,8.50000000000000;1161.83412000000,1.15384600000000,8.50000000000000;1232.63654800000,0.989011000000000,8.50000000000000;1266.35198900000,0.645604000000000,8.50000000000000;1296.69588700000,0.329670000000000,8.50000000000000;1337.15441700000,0,8.50000000000000;1003.37154400000,2.07417600000000,8;1158.46257600000,2.23901100000000,8;1330.41132800000,2.40384600000000,8;1478.75927200000,2.55494500000000,8;1583.27714100000,2.33516500000000,8;1772.08361400000,1.99175800000000,8;1691.16655400000,2.17033000000000,8;1657.45111300000,1.53846200000000,8;1579.90559700000,1.22252700000000,8;1498.98853700000,0.879121000000000,8;1472.01618300000,0.549451000000000,8;1468.64463900000,0.233516000000000,8;1455.15846300000,-0.0137360000000000,8;1532.70397800000,2.44505500000000,8;1633.85030300000,2.25274700000000,8;1731.62508400000,2.08791200000000,8;1728.25354000000,1.82692300000000,8;1687.79501000000,1.66208800000000,8;1613.62103800000,1.35989000000000,8;1536.07552300000,1.04395600000000,8;1397.84221200000,2.44505500000000,8;1242.75118000000,2.29395600000000,8;1080.91706000000,2.14285700000000,8;800,1.90000000000000,8;1003.37154400000,2.65109900000000,7.50000000000000;1161.83412000000,2.85714300000000,7.50000000000000;1330.41132800000,3.06318700000000,7.50000000000000;1485.50236000000,3.20054900000000,7.50000000000000;1748.48280500000,2.89835200000000,7.50000000000000;1937.28927800000,2.50000000000000,7.50000000000000;2072.15104500000,2.23901100000000,7.50000000000000;2173.29737000000,1.99175800000000,7.50000000000000;2018.20633900000,1.23626400000000,7.50000000000000;1883.34457200000,0.576923000000000,7.50000000000000;1755.22589300000,0,7.50000000000000;800,2.50000000000000,7.50000000000000;1003.37154400000,3.22802200000000,7;1077.54551600000,3.29670300000000,7;1161.83412000000,3.32417600000000,7;1242.75118000000,3.39285700000000,7;1327.03978400000,3.46153800000000,7;1414.69993300000,3.53022000000000,7;1495.61699300000,3.55769200000000,7;1664.19420100000,3.42033000000000,7;1832.77140900000,3.28296700000000,7;2001.34861800000,3.13186800000000,7;2085.63722200000,3.07692300000000,7;2169.92582600000,2.92582400000000,7;2304.78759300000,2.59615400000000,7;2439.64935900000,2.28022000000000,7;2530.68105200000,2.00549500000000,7;2473.36480100000,1.42857100000000,7;2389.07619700000,0.934066000000000,7;2287.92987200000,0.439560000000000,7;2193.52663500000,0,7;800,3,7;1006.74308800000,3.80494500000000,6.50000000000000;1161.83412000000,3.84615400000000,6.50000000000000;1246.12272400000,3.90109900000000,6.50000000000000;1330.41132800000,3.88736300000000,6.50000000000000;1495.61699300000,3.95604400000000,6.50000000000000;1748.48280500000,3.79120900000000,6.50000000000000;1997.97707300000,3.64011000000000,6.50000000000000;2540.79568400000,2.98076900000000,6.50000000000000;2169.92582600000,3.43406600000000,6.50000000000000;2335.13149000000,3.22802200000000,6.50000000000000;2658.79973000000,2.40384600000000,6.50000000000000;2746.45987900000,2.00549500000000,6.50000000000000;2706.00134900000,1.57967000000000,6.50000000000000;2706.00134900000,1.40109900000000,6.50000000000000;2652.05664200000,0.906593000000000,6.50000000000000;2604.85502400000,0.412088000000000,6.50000000000000;2557.65340500000,-0.0137360000000000,6.50000000000000;1003.37154400000,4.28571400000000,6;1495.61699300000,4.35439600000000,6;1246.12272400000,4.32692300000000,6;1991.23398500000,4.17582400000000,6;1755.22589300000,4.24450500000000,6;2500.33715400000,3.58516500000000,6;2173.29737000000,3.95604400000000,6;2341.87457900000,3.76373600000000,6;2797.03304100000,2.99450500000000,6;2672.28590700000,3.24175800000000,6;2945.38098400000,2.01923100000000,6;2874.57855700000,2.43131900000000,6;2904.92245400000,1.40109900000000,6;2861.09238000000,0.906593000000000,6;2834.12002700000,0.576923000000000,6;2776.80377600000,-0.0274730000000000,6;800,4.20000000000000,6;1000,4.76648400000000,5.50000000000000;1495.61699300000,4.72527500000000,5.50000000000000;1246.12272400000,4.72527500000000,5.50000000000000;1981.11935300000,4.65659300000000,5.50000000000000;1751.85434900000,4.67033000000000,5.50000000000000;2510.45178700000,4.06593400000000,5.50000000000000;2995.95414700000,3.10439600000000,5.50000000000000;2166.55428200000,4.42307700000000,5.50000000000000;2335.13149000000,4.24450500000000,5.50000000000000;2675.65745100000,3.76373600000000,5.50000000000000;2847.60620400000,3.44780200000000,5.50000000000000;3093.72892800000,2.11538500000000,5.50000000000000;3063.38503000000,2.60989000000000,5.50000000000000;3107.21510500000,1.78571400000000,5.50000000000000;3110.58664900000,1.60714300000000,5.50000000000000;3090.35738400000,1.42857100000000,5.50000000000000;3097.10047200000,0.947802000000000,5.50000000000000;3043.15576500000,0.604396000000000,5.50000000000000;2989.21105900000,-0.0137360000000000,5.50000000000000;800,4.70000000000000,5.50000000000000;1000,5.09615400000000,5;1158.46257600000,5.09615400000000,5;1330.41132800000,5.05494500000000,5;1495.61699300000,5.06868100000000,5;1664.19420100000,5.06868100000000,5;1832.77140900000,5.05494500000000,5;1997.97707300000,5.06868100000000,5;2169.92582600000,4.89011000000000,5;2419.42009400000,4.65659300000000,5;2668.91436300000,4.36813200000000,5;2840.86311500000,4.16208800000000,5;2992.58260300000,3.96978000000000,5;3093.72892800000,3.47527500000000,5;3164.53135500000,3.13186800000000,5;3228.59069500000,2.47252700000000,5;3296.02157800000,1.84065900000000,5;3296.02157800000,1.31868100000000,5;3316.25084300000,1.07142900000000,5;3275.79231300000,0.645604000000000,5;3225.21915000000,0.315934000000000,5;3188.13216500000,-0.0137360000000000,5;800,5.10000000000000,5;1003.37154400000,5.31593400000000,4.50000000000000;1161.83412000000,5.28846200000000,4.50000000000000;1330.41132800000,5.26098900000000,4.50000000000000;1414.69993300000,5.21978000000000,4.50000000000000;1583.27714100000,5.21978000000000,4.50000000000000;1664.19420100000,5.24725300000000,4.50000000000000;1751.85434900000,5.24725300000000,4.50000000000000;1917.06001300000,5.28846200000000,4.50000000000000;2001.34861800000,5.28846200000000,4.50000000000000;2169.92582600000,5.19230800000000,4.50000000000000;2335.13149000000,5.12362600000000,4.50000000000000;2503.70869900000,5.06868100000000,4.50000000000000;2668.91436300000,4.84890100000000,4.50000000000000;2837.49157100000,4.65659300000000,4.50000000000000;3006.06878000000,4.46428600000000,4.50000000000000;3174.64598800000,3.79120900000000,4.50000000000000;3349.96628500000,2.98076900000000,4.50000000000000;3397.16790300000,2.47252700000000,4.50000000000000;3390.42481500000,2.30769200000000,4.50000000000000;3447.74106500000,1.81318700000000,4.50000000000000;3498.31422800000,1.31868100000000,4.50000000000000;3532.02967000000,0.975275000000000,4.50000000000000;3498.31422800000,0.645604000000000,4.50000000000000;3451.11261000000,0.315934000000000,4.50000000000000;3403.91099100000,-0.0137360000000000,4.50000000000000;1003.37154400000,5.60439600000000,4;1074.17397200000,5.56318700000000,4;1242.75118000000,5.46703300000000,4;1161.83412000000,5.53571400000000,4;1414.69993300000,5.39835200000000,4;1495.61699300000,5.35714300000000,4;1664.19420100000,5.42582400000000,4;1832.77140900000,5.49450500000000,4;1917.06001300000,5.49450500000000,4;1997.97707300000,5.53571400000000,4;2166.55428200000,5.42582400000000,4;2416.04855000000,5.32967000000000,4;2668.91436300000,5.19230800000000,4;2840.86311500000,5.09615400000000,4;2989.21105900000,4.95879100000000,4;3491.57114000000,2.98076900000000,4;3120.70128100000,4.46428600000000,4;3289.27849000000,3.79120900000000,4;3380.31018200000,3.46153800000000,4;3548.88739000000,2.47252700000000,4;3542.14430200000,2.29395600000000,4;3596.08900900000,1.81318700000000,4;3680.37761300000,0.975275000000000,4;3643.29062700000,1.31868100000000,4;3646.66217100000,0.467033000000000,4;3629.80445000000,0.151099000000000,4;3616.31827400000,0,4;1003.37154400000,5.79670300000000,3.50000000000000;1161.83412000000,5.72802200000000,3.50000000000000;1495.61699300000,5.53571400000000,3.50000000000000;1347.26904900000,5.60439600000000,3.50000000000000;1987.86244100000,5.76923100000000,3.50000000000000;1579.90559700000,5.56318700000000,3.50000000000000;1755.22589300000,5.64560400000000,3.50000000000000;2493.59406600000,5.56318700000000,3.50000000000000;2257.58597400000,5.63186800000000,3.50000000000000;2995.95414700000,5.26098900000000,3.50000000000000;2668.91436300000,5.45329700000000,3.50000000000000;2837.49157100000,5.34340700000000,3.50000000000000;3147.67363500000,4.95879100000000,3.50000000000000;3616.31827400000,3.03571400000000,3.50000000000000;3285.90694500000,4.39560400000000,3.50000000000000;3444.36952100000,3.72252700000000,3.50000000000000;3828.72555600000,1.07142900000000,3.50000000000000;3690.49224500000,2.25274700000000,3.50000000000000;3768.03776100000,1.57967000000000,3.50000000000000;3848.95482100000,-0.0137360000000000,3.50000000000000;3852.32636500000,0.439560000000000,3.50000000000000;3828.72555600000,0.535714000000000,3.50000000000000;3838.84018900000,0.755495000000000,3.50000000000000;1003.37154400000,6.03022000000000,3;1077.54551600000,6.03022000000000,3;1161.83412000000,5.96153800000000,3;1327.03978400000,5.81044000000000,3;1414.69993300000,5.75549500000000,3;2001.34861800000,6.01648400000000,3;1498.98853700000,5.64560400000000,3;1708.02427500000,5.82417600000000,3;1832.77140900000,5.92033000000000,3;2995.95414700000,5.50824200000000,3;2254.21443000000,5.90659300000000,3;2584.62575900000,5.71428600000000,3;3262.30613600000,4.97252700000000,3;3144.30209000000,5.20604400000000,3;3734.32232000000,3.10439600000000,3;3430.88334500000,4.29945100000000,3;3592.71746500000,3.65384600000000,3;3815.23938000000,2.48626400000000,3;3896.15644000000,1.79945100000000,3;3997.30276500000,1.04395600000000,3;4034.38975100000,0.480769000000000,3;4071.47673600000,0,3;800,6.10000000000000,3;1984.49089700000,6.26373600000000,2.50000000000000;1498.98853700000,5.86538500000000,2.50000000000000;1232.63654800000,6.09890100000000,2.50000000000000;1091.03169300000,6.11263700000000,2.50000000000000;1003.37154400000,6.20879100000000,2.50000000000000;1330.41132800000,5.98901100000000,2.50000000000000;1620.36412700000,5.93406600000000,2.50000000000000;1748.48280500000,6.03022000000000,2.50000000000000;2507.08024300000,6.08516500000000,2.50000000000000;2193.52663500000,6.16758200000000,2.50000000000000;2345.24612300000,6.11263700000000,2.50000000000000;2995.95414700000,5.75549500000000,2.50000000000000;2722.85906900000,5.90659300000000,2.50000000000000;2864.46392400000,5.82417600000000,2.50000000000000;3383.68172600000,5,2.50000000000000;3124.07282500000,5.50824200000000,2.50000000000000;3258.93459200000,5.21978000000000,2.50000000000000;3869.18408600000,3.04945100000000,2.50000000000000;3508.42886000000,4.45054900000000,2.50000000000000;3636.54753900000,3.94230800000000,2.50000000000000;3747.80849600000,3.50274700000000,2.50000000000000;4125.42144300000,0.961538000000000,2.50000000000000;3943.35805800000,2.43131900000000,2.50000000000000;4007.41739700000,1.93681300000000,2.50000000000000;4071.47673600000,1.40109900000000,2.50000000000000;4290.62710700000,-0.0137360000000000,2.50000000000000;4189.48078200000,0.576923000000000,2.50000000000000;4243.42548900000,0.260989000000000,2.50000000000000;1006.74308800000,6.37362600000000,2;1465.27309500000,6.04395600000000,2;1246.12272400000,6.18131900000000,2;1981.11935300000,6.52472500000000,2;1674.30883300000,6.20879100000000,2;1826.02832100000,6.35989000000000,2;2486.85097800000,6.57967000000000,2;2173.29737000000,6.55219800000000,2;2335.13149000000,6.55219800000000,2;2995.95414700000,6.03022000000000,2;2675.65745100000,6.38736300000000,2;2840.86311500000,6.18131900000000,2;3515.17194900000,4.97252700000000,2;3181.38907600000,5.65934100000000,2;3329.73702000000,5.32967000000000,2;4004.04585300000,3.03571400000000,2;3697.23533400000,4.24450500000000,2;3875.92717500000,3.53022000000000,2;4051.24747100000,2.58241800000000,2;4084.96291300000,2.30769200000000,2;4101.82063400000,2.08791200000000,2;4155.76534100000,1.64835200000000,2;4223.19622400000,1.20879100000000,2;4253.54012100000,1.00274700000000,2;4506.40593400000,-0.0274730000000000,2;4354.68644600000,0.604396000000000,2;4435.60350600000,0.274725000000000,2;1000,6.49725300000000,1.50000000000000;1165.20566400000,6.41483500000000,1.50000000000000;1333.78287300000,6.31868100000000,1.50000000000000;1414.69993300000,6.27747300000000,1.50000000000000;1498.98853700000,6.27747300000000,1.50000000000000;1667.56574500000,6.44230800000000,1.50000000000000;1917.06001300000,6.67582400000000,1.50000000000000;2048.55023600000,6.78571400000000,1.50000000000000;2490.22252200000,7.04670300000000,1.50000000000000;2254.21443000000,6.90934100000000,1.50000000000000;2557.65340500000,7.03296700000000,1.50000000000000;2995.95414700000,6.52472500000000,1.50000000000000;2712.74443700000,6.84065900000000,1.50000000000000;2874.57855700000,6.64835200000000,1.50000000000000;3494.94268400000,5.49450500000000,1.50000000000000;3178.01753200000,6.15384600000000,1.50000000000000;3349.96628500000,5.82417600000000,1.50000000000000;3747.80849600000,4.49175800000000,1.50000000000000;3602.83209700000,5.09615400000000,1.50000000000000;3666.89143600000,4.80769200000000,1.50000000000000;4118.67835500000,3.02197800000000,1.50000000000000;3872.55563000000,4.01098900000000,1.50000000000000;3997.30276500000,3.50274700000000,1.50000000000000;4155.76534100000,2.52747300000000,1.50000000000000;4196.22387100000,2.00549500000000,1.50000000000000;4280.51247500000,1.52472500000000,1.50000000000000;4368.17262300000,1.01648400000000,1.50000000000000;4455.83277100000,0.686813000000000,1.50000000000000;4533.37828700000,0.357143000000000,1.50000000000000;4617.66689100000,-0.0137360000000000,1.50000000000000;1003.37154400000,6.66208800000000,1;1165.20566400000,6.59340700000000,1;1256.23735700000,6.62087900000000,1;1330.41132800000,6.57967000000000,1;1414.69993300000,6.55219800000000,1;1498.98853700000,6.53846200000000,1;1994.60552900000,7.04670300000000,1;1667.56574500000,6.70329700000000,1;1836.14295300000,6.88186800000000,1;2085.63722200000,7.07417600000000,1;2254.21443000000,7.17033000000000,1;2496.96561000000,7.34890100000000,1;2382.33310900000,7.26648400000000,1;3002.69723500000,7.01923100000000,1;2672.28590700000,7.22527500000000,1;2840.86311500000,7.10164800000000,1;3498.31422800000,6.03022000000000,1;3178.01753200000,6.64835200000000,1;3349.96628500000,6.31868100000000,1;3670.26298000000,5.32967000000000,1;3838.84018900000,4.67033000000000,1;4007.41739700000,4.01098900000000,1;4122.04989900000,3.50274700000000,1;4260.28321000000,3.02197800000000,1;4283.88401900000,2.48626400000000,1;4287.25556300000,2.17033000000000,1;4334.45718100000,1.82692300000000,1;4438.97505100000,1.33241800000000,1;4540.12137600000,0.837912000000000,1;4674.98314200000,0.384615000000000,1;4705.32704000000,0.178571000000000,1;4759.27174600000,-0.0137360000000000,1;1003.37154400000,6.84065900000000,0.500000000000000;1080.91706000000,6.81318700000000,0.500000000000000;1161.83412000000,6.81318700000000,0.500000000000000;1330.41132800000,6.77197800000000,0.500000000000000;1418.07147700000,6.74450500000000,0.500000000000000;1498.98853700000,6.77197800000000,0.500000000000000;1667.56574500000,6.95054900000000,0.500000000000000;1991.23398500000,7.51373600000000,0.500000000000000;1832.77140900000,7.21153800000000,0.500000000000000;2493.59406600000,7.69230800000000,0.500000000000000;2169.92582600000,7.54120900000000,0.500000000000000;2335.13149000000,7.60989000000000,0.500000000000000;2995.95414700000,7.50000000000000,0.500000000000000;2672.28590700000,7.60989000000000,0.500000000000000;2840.86311500000,7.54120900000000,0.500000000000000;3761.29467300000,5.98901100000000,0.500000000000000;3262.30613600000,6.97802200000000,0.500000000000000;3596.08900900000,6.31868100000000,0.500000000000000;3427.51180000000,6.64835200000000,0.500000000000000;3815.23938000000,5.49450500000000,0.500000000000000;3875.92717500000,5.01373600000000,0.500000000000000;3963.58732300000,4.64285700000000,0.500000000000000;4081.59136900000,4.32692300000000,0.500000000000000;4165.87997300000,4.16208800000000,0.500000000000000;4253.54012100000,3.99725300000000,0.500000000000000;4020.90357400000,4.47802200000000,0.500000000000000;4297.37019600000,3.69505500000000,0.500000000000000;4334.45718100000,3.32417600000000,0.500000000000000;4374.91571100000,3.06318700000000,0.500000000000000;4371.54416700000,2.82967000000000,0.500000000000000;4388.40188800000,2.50000000000000,0.500000000000000;4381.65880000000,2.32142900000000,0.500000000000000;4415.37424100000,2.22527500000000,0.500000000000000;4408.63115300000,2.00549500000000,0.500000000000000;4469.31894800000,1.71703300000000,0.500000000000000;4499.66284600000,1.51098900000000,0.500000000000000;4755.90020200000,1.00274700000000,0.500000000000000;4600.80917100000,1.33241800000000,0.500000000000000;4671.61159800000,1.19505500000000,0.500000000000000;4803.10182100000,0.659341000000000,0.500000000000000;4840.18880600000,0.357143000000000,0.500000000000000;4877.27579200000,-0.0137360000000000,0.500000000000000;800,6.90000000000000,0.500000000000000;1000,7.07417600000000,0;1003.37154400000,10.8241760000000,0;1003.37154400000,10.0137360000000,0;1006.74308800000,8.68131900000000,0;1583.27714100000,10.8241760000000,0;2729.60215800000,10.8379120000000,0;4000.67430900000,10.8241760000000,0;5211.05866500000,10.8379120000000,0;6003.37154400000,10.8241760000000,0;6000,9.02472500000000,0;6003.37154400000,7.23901100000000,0;6000,5.20604400000000,0;6003.37154400000,3.11813200000000,0;6003.37154400000,1.35989000000000,0;6003.37154400000,-0.0274730000000000,0;5676.33176000000,-0.0274730000000000,0;5265.00337200000,-0.0274730000000000,0;4954.82130800000,-0.0274730000000000,0;4927.84895500000,0.412088000000000,0;4833.44571800000,1.07142900000000,0;4715.44167200000,1.51098900000000,0;4540.12137600000,1.90934100000000,0;4465.94740400000,2.99450500000000,0;4354.68644600000,4.16208800000000,0;4105.19217800000,4.78022000000000,0;3966.95886700000,5.05494500000000,0;3855.69791000000,5.85164800000000,0;3687.12070100000,6.63461500000000,0;3424.14025600000,7.07417600000000,0;3154.41672300000,7.54120900000000,0;2847.60620400000,7.92582400000000,0;2544.16722900000,8.03571400000000,0;2244.09979800000,8.00824200000000,0;1987.86244100000,7.92582400000000,0;1792.31287900000,7.70604400000000,0;1654.07956800000,7.40384600000000,0;1498.98853700000,7.18406600000000,0;1380.98449100000,7.14285700000000,0;1246.12272400000,7.06044000000000,0;1111.26095800000,7.10164800000000,0;1549.56169900000,9.21703300000000,0;2712.74443700000,9.27197800000000,0;3869.18408600000,9.20329700000000,0;5497.63991900000,5.15109900000000,0;5207.68712100000,8.76373600000000,0;4712.07012800000,6.70329700000000,0;5028.99528000000,4.03846200000000,0;5581.92852300000,2.56868100000000,0;4155.76534100000,7.03296700000000,0;800,7.25000000000000,0;800,9,0;800,11,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddDCP_DVVL, RmvDCP_DVVL, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/DCP_DVVL.png'));
    title('Ricardo DCP_DVVL Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm, bsfc_improve_pcnt, [0:0.5:9.5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('Torque (Nm)');
    ylim([0 210/2.4*3.8]);
	%}
end


%% 5: Add/Remove technology: DCP_CVVL
% map_data = [DcpCvvl15p5;DcpCvvl15;DcpCvvl14p5;DcpCvvl14;DcpCvvl13p5;DcpCvvl13;DcpCvvl12p5;DcpCvvl12;DcpCvvl11p5;DcpCvvl11;DcpCvvl10p5;DcpCvvl10;DcpCvvl9p5;DcpCvvl9;DcpCvvl8p5;DcpCvvl8;DcpCvvl7p5;...
% DcpCvvl7;DcpCvvl6p5;DcpCvvl6;DcpCvvl5p5;DcpCvvl5;DcpCvvl4p5;...
% DcpCvvl4;DcpCvvl3p5;DcpCvvl3;DcpCvvl2p5;DcpCvvl2;DcpCvvl1p5;...
% DcpCvvl1;DcpCvvl0p5;DcpCvvl0];
if (AddDCP_CVVL==1 || RmvDCP_CVVL==1)
	%     './data/DCP_CVVL_20150610_2.mat'
	map_data = [1000,0.158730000000000,15.5000000000000;1043.94861400000,0,15.5000000000000;1000,0.291005000000000,15;1108.18120400000,0,15;1064.23259000000,0.132275000000000,15;1003.38066300000,0.423280000000000,14.5000000000000;1060.85192700000,0.264550000000000,14.5000000000000;1131.84584200000,0.105820000000000,14.5000000000000;1179.17511800000,0,14.5000000000000;1000,0.542328000000000,14;1226.50439500000,0.0132280000000000,14;1064.23259000000,0.396825000000000,14;1128.46517900000,0.238095000000000,14;1280.59499700000,0.0132280000000000,13.5000000000000;1212.98174400000,0.198413000000000,13.5000000000000;1000,0.661376000000000,13.5000000000000;1145.36849200000,0.370370000000000,13.5000000000000;1060.85192700000,0.542328000000000,13.5000000000000;1358.35023700000,0,13;1212.98174400000,0.370370000000000,13;1128.46517900000,0.529101000000000,13;1000,0.873016000000000,13;1057.47126400000,0.714286000000000,13;1000,1.03174600000000,12.5000000000000;1108.18120400000,0.780423000000000,12.5000000000000;1216.36240700000,0.502646000000000,12.5000000000000;1324.54361100000,0.251323000000000,12.5000000000000;1419.20216400000,0,12.5000000000000;800,1.60000000000000,12.5000000000000;1003.38066300000,1.34920600000000,12;1162.27180500000,0.978836000000000,12;1270.45300900000,0.648148000000000,12;1378.63421200000,0.330688000000000,12;1483.43475300000,0,12;1000,1.67989400000000,11.5000000000000;1169.03313000000,1.32275100000000,11.5000000000000;1327.92427300000,0.978836000000000,11.5000000000000;1436.10547700000,0.648148000000000,11.5000000000000;1584.85463200000,0.317460000000000,11.5000000000000;1726.84246100000,0.0132280000000000,11.5000000000000;1000,2.02381000000000,11;1973.63083200000,0.0132280000000000,11;1169.03313000000,1.66666700000000,11;1348.20824900000,1.28306900000000,11;1581.47396900000,0.820106000000000,11;1750.50709900000,0.462963000000000,11;1496.95740400000,0.978836000000000,11;1862.06896600000,0.224868000000000,11;1000,2.34127000000000,10.5000000000000;2227.18052700000,0.0132280000000000,10.5000000000000;1246.78837100000,1.97089900000000,10.5000000000000;1415.82150100000,1.64021200000000,10.5000000000000;1581.47396900000,1.30952400000000,10.5000000000000;1750.50709900000,0.978836000000000,10.5000000000000;1916.15956700000,0.648148000000000,10.5000000000000;2081.81203500000,0.317460000000000,10.5000000000000;1003.38066300000,2.65873000000000,10;1158.89114300000,2.43386200000000,10;1331.30493600000,2.20899500000000,10;1496.95740400000,1.97089900000000,10;1665.99053400000,1.64021200000000,10;1831.64300200000,1.30952400000000,10;2081.81203500000,0.820106000000000,10;2467.20757300000,0.0132280000000000,10;2321.83908000000,0.317460000000000,10;2227.18052700000,0.542328000000000,10;1005.54491900000,2.99004200000000,9.50000000000000;1162.50876800000,2.84179400000000,9.50000000000000;1411.60357100000,2.57225200000000,9.50000000000000;1667.52289100000,2.14098500000000,9.50000000000000;1923.44221000000,1.64233300000000,9.50000000000000;2165.71249800000,1.14368100000000,9.50000000000000;2418.21956000000,0.658506000000000,9.50000000000000;2622.95501500000,0.0250820000000000,9.50000000000000;2486.46471200000,0.483303000000000,9.50000000000000;2571.77115100000,0.146376000000000,9.50000000000000;1005.54491900000,3.32696900000000,9;1162.50876800000,3.24610700000000,9;1329.70938900000,3.11133600000000,9;1496.91001100000,2.97656500000000,9;1664.11063300000,2.63963800000000,9;1913.20543700000,2.14098500000000,9;2169.12475600000,1.64233300000000,9;2336.32537800000,1.30540600000000,9;2486.46471200000,0.995433000000000,9;2571.77115100000,0.645028000000000,9;2742.38403100000,0.0116050000000000,9;2646.84081800000,0.362010000000000,9;1002.13266100000,3.66389600000000,8.50000000000000;1162.50876800000,3.55608000000000,8.50000000000000;1411.60357100000,3.36740000000000,8.50000000000000;1493.49775400000,3.31349200000000,8.50000000000000;1664.11063300000,3.09785900000000,8.50000000000000;1756.24158800000,2.97656500000000,8.50000000000000;2001.92413400000,2.47791300000000,8.50000000000000;1916.61769400000,2.63963800000000,8.50000000000000;2169.12475600000,1.97926000000000,8.50000000000000;2094.05508900000,2.19489400000000,8.50000000000000;2332.91312000000,1.64233300000000,8.50000000000000;2503.52600000000,1.31888300000000,8.50000000000000;2882.28659200000,0.0116050000000000,8.50000000000000;2626.36727300000,0.981956000000000,8.50000000000000;2708.26145500000,0.645028000000000,8.50000000000000;2756.03306100000,0.496780000000000,8.50000000000000;2837.92724300000,0.146376000000000,8.50000000000000;1002.13266100000,4.00082400000000,8;1159.09651000000,3.89300700000000,8;1329.70938900000,3.77171300000000,8;1486.67323800000,3.66389600000000,8;1664.11063300000,3.43478600000000,8;1834.72351200000,3.20567500000000,8;1913.20543700000,3.11133600000000,8;1991.68736100000,3.00351900000000,8;2336.32537800000,1.96578300000000,8;2104.29186200000,2.63963800000000,8;2216.89636200000,2.30271000000000,8;2500.11374200000,1.62885600000000,8;2564.94663600000,1.48060800000000,8;2691.20016700000,1.14368100000000,8;2783.33112200000,0.820231000000000,8;2988.06657700000,-0.00187200000000000,8;2953.94400100000,0.159853000000000,8;2861.81304600000,0.537212000000000,8;1002.13266100000,4.27036500000000,7.50000000000000;1165.92102500000,4.16254900000000,7.50000000000000;1496.91001100000,4.00082400000000,7.50000000000000;1329.70938900000,4.06820900000000,7.50000000000000;2001.92413400000,3.47521700000000,7.50000000000000;1664.11063300000,3.82562100000000,7.50000000000000;1831.31125500000,3.66389600000000,7.50000000000000;2169.12475600000,3.13829000000000,7.50000000000000;2244.19442300000,2.97656500000000,7.50000000000000;2322.67634700000,2.63963800000000,7.50000000000000;2407.98278700000,2.31618700000000,7.50000000000000;2503.52600000000,1.97926000000000,7.50000000000000;2626.36727300000,1.64233300000000,7.50000000000000;2756.03306100000,1.31888300000000,7.50000000000000;2872.04981900000,0.981956000000000,7.50000000000000;2957.35625900000,0.645028000000000,7.50000000000000;3022.18915300000,0.375487000000000,7.50000000000000;3131.38139600000,-0.00187200000000000,7.50000000000000;3080.19753200000,0.173330000000000,7.50000000000000;1000,4.51058200000000,7;1162.27180500000,4.43121700000000,7;1331.30493600000,4.40476200000000,7;1496.95740400000,4.33862400000000,7;1665.99053400000,4.19312200000000,7;1831.64300200000,4.10052900000000,7;2000.67613300000,3.96825400000000,7;2169.70926300000,3.63756600000000,7;2335.36173100000,3.30687800000000,7;2487.49154800000,2.97619000000000,7;2572.00811400000,2.64550300000000,7;2659.90534100000,2.31481500000000,7;2737.66058100000,1.97089900000000,7;2825.55780900000,1.64021200000000,7;2913.45503700000,1.30952400000000,7;2991.21027700000,0.992063000000000,7;3075.72684200000,0.648148000000000,7;3156.86274500000,0.317460000000000,7;3248.14063600000,0,7;3207.57268400000,0.158730000000000,7;1000,4.77513200000000,6.50000000000000;1077.75524000000,4.73545000000000,6.50000000000000;1165.65246800000,4.73545000000000,6.50000000000000;1331.30493600000,4.69576700000000,6.50000000000000;1496.95740400000,4.65608500000000,6.50000000000000;1665.99053400000,4.52381000000000,6.50000000000000;1750.50709900000,4.51058200000000,6.50000000000000;2000.67613300000,4.29894200000000,6.50000000000000;1831.64300200000,4.43121700000000,6.50000000000000;2169.70926300000,4.11375700000000,6.50000000000000;2737.66058100000,2.97619000000000,6.50000000000000;2419.87829600000,3.63756600000000,6.50000000000000;2585.53076400000,3.30687800000000,6.50000000000000;2791.75118300000,2.80423300000000,6.50000000000000;2818.79648400000,2.64550300000000,6.50000000000000;2991.21027700000,1.97089900000000,6.50000000000000;2906.69371200000,2.31481500000000,6.50000000000000;3038.53955400000,1.64021200000000,6.50000000000000;3089.24949300000,1.30952400000000,6.50000000000000;3106.15280600000,1.15079400000000,6.50000000000000;3123.05611900000,0.978836000000000,6.50000000000000;3170.38539600000,0.820106000000000,6.50000000000000;3383.36714000000,0,6.50000000000000;3251.52129800000,0.489418000000000,6.50000000000000;3342.79918900000,0.145503000000000,6.50000000000000;1000,5.03968300000000,6;1496.95740400000,5,6;1246.78837100000,5,6;1081.13590300000,5,6;1371.87288700000,5,6;1665.99053400000,4.86772500000000,6;1578.09330600000,4.92063500000000,6;1831.64300200000,4.76190500000000,6;2000.67613300000,4.62963000000000,6;2169.70926300000,4.44444400000000,6;2335.36173100000,4.19312200000000,6;2504.39486100000,3.96825400000000,6;2670.04732900000,3.63756600000000,6;2839.08046000000,3.30687800000000,6;2923.59702500000,3.13492100000000,6;2991.21027700000,2.97619000000000,6;2423.25895900000,4.10052900000000,6;3494.92900600000,-0.0132280000000000,6;3021.63624100000,2.80423300000000,6;3038.53955400000,2.64550300000000,6;3058.82352900000,2.47354500000000,6;3106.15280600000,2.14285700000000,6;3139.95943200000,1.81216900000000,6;3190.66937100000,1.48148100000000,6;3224.47599700000,1.15079400000000,6;3241.37931000000,0.978836000000000,6;3322.51521300000,0.648148000000000,6;3407.03177800000,0.317460000000000,6;800,5,6;1000,5.23809500000000,5.50000000000000;1162.27180500000,5.22486800000000,5.50000000000000;1331.30493600000,5.15873000000000,5.50000000000000;1496.95740400000,5.11904800000000,5.50000000000000;1662.60987200000,5.09259300000000,5.50000000000000;1831.64300200000,5.02645500000000,5.50000000000000;1990.53414500000,5,5.50000000000000;2162.94793800000,4.82804200000000,5.50000000000000;2331.98106800000,4.65608500000000,5.50000000000000;2494.25287400000,4.49735400000000,5.50000000000000;3004.73292800000,3.46560800000000,5.50000000000000;2670.04732900000,4.14021200000000,5.50000000000000;2839.08046000000,3.79629600000000,5.50000000000000;3150.10142000000,2.89682500000000,5.50000000000000;3089.24949300000,3.13492100000000,5.50000000000000;3207.57268400000,2.31481500000000,5.50000000000000;3275.18593600000,1.81216900000000,5.50000000000000;3329.27653800000,1.45502600000000,5.50000000000000;3366.46382700000,1.11111100000000,5.50000000000000;3383.36714000000,0.978836000000000,5.50000000000000;3620.01352300000,-0.0132280000000000,5.50000000000000;3467.88370500000,0.648148000000000,5.50000000000000;3545.63894500000,0.317460000000000,5.50000000000000;1000,5.46296300000000,5;1077.75524000000,5.44973500000000,5;1277.21433400000,5.33068800000000,5;1172.41379300000,5.41005300000000,5;1415.82150100000,5.29100500000000,5;1581.47396900000,5.25132300000000,5;1750.50709900000,5.22486800000000,5;1835.02366500000,5.22486800000000,5;2004.05679500000,5.19841300000000,5;2494.25287400000,5,5;2169.70926300000,5.11904800000000,5;2335.36173100000,5.05291000000000,5;2592.29208900000,4.80158700000000,5;2757.94455700000,4.47089900000000,5;3011.49425300000,3.96825400000000,5;2883.02907400000,4.23280400000000,5;3265.04394900000,2.94973500000000,5;3106.15280600000,3.57142900000000,5;3197.43069600000,3.22751300000000,5;3417.17376600000,1.77248700000000,5;3312.37322500000,2.50000000000000,5;3376.60581500000,2.03703700000000,5;3505.07099400000,0.965608000000000,5;3461.12238000000,1.30952400000000,5;3748.47870200000,0.0132280000000000,5;3582.82623400000,0.648148000000000,5;3670.72346200000,0.330688000000000,5;1000,5.75396800000000,4.50000000000000;1466.53144000000,5.39682500000000,4.50000000000000;1165.65246800000,5.63492100000000,4.50000000000000;1331.30493600000,5.48941800000000,4.50000000000000;1496.95740400000,5.38359800000000,4.50000000000000;1588.23529400000,5.39682500000000,4.50000000000000;1750.50709900000,5.38359800000000,4.50000000000000;1916.15956700000,5.39682500000000,4.50000000000000;2494.25287400000,5.26455000000000,4.50000000000000;2169.70926300000,5.34391500000000,4.50000000000000;2335.36173100000,5.27777800000000,4.50000000000000;2747.80256900000,4.97354500000000,4.50000000000000;2639.62136600000,5.10582000000000,4.50000000000000;2997.97160200000,4.48412700000000,4.50000000000000;2896.55172400000,4.69576700000000,4.50000000000000;3379.98647700000,3.02910100000000,4.50000000000000;3129.81744400000,3.99470900000000,4.50000000000000;3254.90196100000,3.50529100000000,4.50000000000000;3620.01352300000,0.978836000000000,4.50000000000000;3447.59973000000,2.38095200000000,4.50000000000000;3532.11629500000,1.67989400000000,4.50000000000000;3870.18255600000,0,4.50000000000000;3697.76876300000,0.648148000000000,4.50000000000000;3778.90466500000,0.317460000000000,4.50000000000000;800,5.90000000000000,4.50000000000000;1000,6.01851900000000,4;1463.15077800000,5.50264600000000,4;1162.27180500000,5.82010600000000,4;1331.30493600000,5.66137600000000,4;1496.95740400000,5.48941800000000,4;1581.47396900000,5.52910100000000,4;1750.50709900000,5.54232800000000,4;1916.15956700000,5.59523800000000,4;1997.29547000000,5.59523800000000,4;2504.39486100000,5.50264600000000,4;2169.70926300000,5.55555600000000,4;2338.74239400000,5.52910100000000,4;2997.97160200000,4.98677200000000,4;3498.30966900000,3.02910100000000,4;2700.47329300000,5.27777800000000,4;2855.98377300000,5.10582000000000,4;3133.19810700000,4.47089900000000,4;3251.52129800000,3.98148100000000,4;3383.36714000000,3.47883600000000,4;3741.71737700000,1.05820100000000,4;3562.54225800000,2.47354500000000,4;3643.67816100000,1.81216900000000,4;3995.26707200000,0.0132280000000000,4;3836.37593000000,0.648148000000000,4;3924.27315800000,0.317460000000000,4;1003.38066300000,6.11111100000000,3.50000000000000;1415.82150100000,5.68783100000000,3.50000000000000;1165.65246800000,5.96560800000000,3.50000000000000;1294.11764700000,5.83333300000000,3.50000000000000;1496.95740400000,5.64814800000000,3.50000000000000;1655.84854600000,5.68783100000000,3.50000000000000;1835.02366500000,5.75396800000000,3.50000000000000;1993.91480700000,5.79365100000000,3.50000000000000;2504.39486100000,5.75396800000000,3.50000000000000;2250.84516600000,5.75396800000000,3.50000000000000;2997.97160200000,5.25132300000000,3.50000000000000;2754.56389500000,5.48941800000000,3.50000000000000;3150.10142000000,4.94709000000000,3.50000000000000;3613.25219700000,3.04232800000000,3.50000000000000;3281.94726200000,4.41798900000000,3.50000000000000;3440.83840400000,3.74338600000000,3.50000000000000;3856.65990500000,1.08465600000000,3.50000000000000;3697.76876300000,2.42063500000000,3.50000000000000;3782.28532800000,1.74603200000000,3.50000000000000;4120.35158900000,0,3.50000000000000;3978.36375900000,0.582011000000000,3.50000000000000;1000,6.25661400000000,3;1493.57674100000,5.75396800000000,3;1165.65246800000,6.08465600000000,3;1334.68559800000,5.92592600000000,3;1669.37119700000,5.85978800000000,3;1835.02366500000,5.96560800000000,3;1983.77281900000,6.03174600000000,3;2494.25287400000,6.01851900000000,3;2169.70926300000,6.01851900000000,3;2335.36173100000,6.03174600000000,3;2994.59094000000,5.48941800000000,3;3265.04394900000,4.96031700000000,3;3133.19810700000,5.22486800000000,3;2673.42799200000,5.82010600000000,3;2839.08046000000,5.66137600000000,3;3745.09803900000,3.05555600000000,3;3417.17376600000,4.32539700000000,3;3592.96822200000,3.63756600000000,3;4002.02839800000,0.992063000000000,3;3829.61460400000,2.31481500000000,3;3893.84719400000,1.81216900000000,3;3961.46044600000,1.30952400000000,3;4255.57809300000,0,3;4086.54496300000,0.661376000000000,3;4198.10682900000,0.211640000000000,3;1000,6.38888900000000,2.50000000000000;1496.95740400000,5.89947100000000,2.50000000000000;1165.65246800000,6.21693100000000,2.50000000000000;1334.68559800000,6.04497400000000,2.50000000000000;1811.35902600000,6.09788400000000,2.50000000000000;1665.99053400000,5.99206300000000,2.50000000000000;1977.01149400000,6.26984100000000,2.50000000000000;2487.49154800000,6.52116400000000,2.50000000000000;2173.08992600000,6.34920600000000,2.50000000000000;2338.74239400000,6.44179900000000,2.50000000000000;2761.32522000000,6.00529100000000,2.50000000000000;2639.62136600000,6.24338600000000,2.50000000000000;2997.97160200000,5.75396800000000,2.50000000000000;2883.02907400000,5.87301600000000,2.50000000000000;3379.98647700000,5,2.50000000000000;3183.90804600000,5.38359800000000,2.50000000000000;3863.42123100000,3.04232800000000,2.50000000000000;3518.59364400000,4.39153400000000,2.50000000000000;3684.24611200000,3.73015900000000,2.50000000000000;4120.35158900000,0.992063000000000,2.50000000000000;3954.69912100000,2.35449700000000,2.50000000000000;4029.07369800000,1.74603200000000,2.50000000000000;4062.88032500000,1.41534400000000,2.50000000000000;4370.52062200000,0,2.50000000000000;4221.77146700000,0.595238000000000,2.50000000000000;4306.28803200000,0.264550000000000,2.50000000000000;1000,6.46825400000000,2;1081.13590300000,6.41534400000000,2;1250.16903300000,6.24338600000000,2;1500.33806600000,6.04497400000000,2;1415.82150100000,6.08465600000000,2;1665.99053400000,6.21693100000000,2;1838.40432700000,6.38888900000000,2;2000.67613300000,6.54761900000000,2;2494.25287400000,6.99735400000000,2;2173.08992600000,6.71957700000000,2;2335.36173100000,6.87830700000000,2;2673.42799200000,6.64021200000000,2;2839.08046000000,6.32275100000000,2;3494.92900600000,4.98677200000000,2;3092.63015600000,5.82010600000000,2;3325.89587600000,5.31746000000000,2;3995.26707200000,3.00264600000000,2;3660.58147400000,4.32539700000000,2;3829.61460400000,3.66402100000000,2;4012.17038500000,2.83068800000000,2;4116.97092600000,1.83862400000000,2;4062.88032500000,2.34127000000000,2;4248.81676800000,1.00529100000000,2;4198.10682900000,1.34920600000000,2;4502.36646400000,0,2;4363.75929700000,0.515873000000000,2;4414.46923600000,0.357143000000000,2;1000,6.61375700000000,1.50000000000000;1165.65246800000,6.48148100000000,1.50000000000000;1500.33806600000,6.28306900000000,1.50000000000000;1334.68559800000,6.34920600000000,1.50000000000000;2501.01419900000,7.24867700000000,1.50000000000000;1669.37119700000,6.44179900000000,1.50000000000000;1919.54023000000,6.67989400000000,1.50000000000000;2173.08992600000,6.94444400000000,1.50000000000000;2342.12305600000,7.11640200000000,1.50000000000000;2673.42799200000,7.08994700000000,1.50000000000000;3494.92900600000,5.48941800000000,1.50000000000000;2926.97768800000,6.65343900000000,1.50000000000000;3173.76605800000,6.15079400000000,1.50000000000000;4113.59026400000,3.00264600000000,1.50000000000000;4079.78363800000,3.16137600000000,1.50000000000000;3647.05882400000,4.84127000000000,1.50000000000000;3789.04665300000,4.31216900000000,1.50000000000000;3893.84719400000,3.90211600000000,1.50000000000000;3995.26707200000,3.49206300000000,1.50000000000000;3342.79918900000,5.82010600000000,1.50000000000000;4147.39689000000,2.67195800000000,1.50000000000000;4147.39689000000,2.50000000000000,1.50000000000000;4184.58417800000,2.16931200000000,1.50000000000000;4194.72616600000,1.99735400000000,1.50000000000000;4252.19743100000,1.65343900000000,1.50000000000000;4316.43002000000,1.33597900000000,1.50000000000000;4329.95267100000,1.16402100000000,1.50000000000000;4417.84989900000,0.846561000000000,1.50000000000000;4617.30899300000,-0.0132280000000000,1.50000000000000;4532.79242700000,0.343915000000000,1.50000000000000;1003.38066300000,6.74603200000000,1;1081.13590300000,6.71957700000000,1;1250.16903300000,6.61375700000000,1;1469.91210300000,6.53439200000000,1;1331.30493600000,6.58730200000000,1;2487.49154800000,7.47354500000000,1;1669.37119700000,6.71957700000000,1;1916.15956700000,6.94444400000000,1;2173.08992600000,7.18254000000000,1;2335.36173100000,7.35449700000000,1;3001.35226500000,7.01058200000000,1;2673.42799200000,7.31481500000000,1;2842.46112200000,7.14285700000000,1;3511.83231900000,5.96560800000000,1;3180.52738300000,6.65343900000000,1;3349.56051400000,6.32275100000000,1;4258.95875600000,3.01587300000000,1;3670.72346200000,5.34391500000000,1;3832.99526700000,4.65608500000000,1;4002.02839800000,3.99470900000000,1;4177.82285300000,3.33333300000000,1;4279.24273200000,2.51322800000000,1;4286.00405700000,2.18254000000000,1;4306.28803200000,2.01058200000000,1;4566.59905300000,0.701058000000000,1;4397.56592300000,1.52116400000000,1;4495.60513900000,1.01851900000000,1;4749.15483400000,0,1;4671.39959400000,0.357143000000000,1;1003.38066300000,6.85185200000000,0.500000000000000;1165.65246800000,6.83862400000000,0.500000000000000;1496.95740400000,6.77248700000000,0.500000000000000;1334.68559800000,6.77248700000000,0.500000000000000;1987.15348200000,7.48677200000000,0.500000000000000;1669.37119700000,6.94444400000000,0.500000000000000;1831.64300200000,7.20899500000000,0.500000000000000;2497.63353600000,7.75132300000000,0.500000000000000;2173.08992600000,7.57936500000000,0.500000000000000;2335.36173100000,7.67195800000000,0.500000000000000;3001.35226500000,7.46031700000000,0.500000000000000;2673.42799200000,7.64550300000000,0.500000000000000;2839.08046000000,7.53968300000000,0.500000000000000;3755.24002700000,5.99206300000000,0.500000000000000;3261.66328600000,6.98412700000000,0.500000000000000;3508.45165700000,6.48148100000000,0.500000000000000;3812.71129100000,5.48941800000000,0.500000000000000;3866.80189300000,4.98677200000000,0.500000000000000;3947.93779600000,4.65608500000000,0.500000000000000;3995.26707200000,4.49735400000000,0.500000000000000;4245.43610500000,4.00793700000000,0.500000000000000;4083.16430000000,4.32539700000000,0.500000000000000;4164.30020300000,4.16666700000000,0.500000000000000;4864.09736300000,0,0.500000000000000;4286.00405700000,3.66402100000000,0.500000000000000;4333.33333300000,3.33333300000000,0.500000000000000;4370.52062200000,3.04232800000000,0.500000000000000;4367.13995900000,2.84391500000000,0.500000000000000;4387.42393500000,2.67195800000000,0.500000000000000;4380.66261000000,2.34127000000000,0.500000000000000;4400.94658600000,2.16931200000000,0.500000000000000;4400.94658600000,1.99735400000000,0.500000000000000;4502.36646400000,1.52116400000000,0.500000000000000;4749.15483400000,1.01851900000000,0.500000000000000;4620.68965500000,1.26984100000000,0.500000000000000;4789.72278600000,0.687831000000000,0.500000000000000;4833.67140000000,0.357143000000000,0.500000000000000;2588.91142700000,7.68518500000000,0.500000000000000;2754.56389500000,7.60582000000000,0.500000000000000;2926.97768800000,7.50000000000000,0.500000000000000;3092.63015600000,7.28836000000000,0.500000000000000;3346.17985100000,6.81216900000000,0.500000000000000;3596.34888400000,6.32275100000000,0.500000000000000;3427.31575400000,6.65343900000000,0.500000000000000;3677.48478700000,6.15079400000000,0.500000000000000;800,6.80000000000000,0.500000000000000;996.619337000000,7.07672000000000,0;996.619337000000,8.92857100000000,0;989.858012000000,10.7936510000000,0;6006.76132500000,10.7936510000000,0;1496.95740400000,10.7936510000000,0;2250.84516600000,10.7804230000000,0;3373.22515200000,10.7804230000000,0;4759.29682200000,10.5687830000000,0;5279.91886400000,10.8068780000000,0;4465.17917500000,10.8201060000000,0;6006.76132500000,0,0;6027.04530100000,9.27248700000000,0;6006.76132500000,7.42063500000000,0;6003.38066300000,5.48941800000000,0;6006.76132500000,3.45238100000000,0;6023.66463800000,1.74603200000000,0;4935.09127800000,0,0;5185.26031100000,-0.0264550000000000,0;5617.98512500000,-0.0264550000000000,0;800,7,0;800,9,0;800,11,0;1077.20232800000,7.01980400000000,0;1179.57005600000,7.04695000000000,0;1292.17455600000,6.97908400000000,0;1425.25260200000,6.96551100000000,0;1561.74290500000,7.08766900000000,0;1687.99643600000,7.27769300000000,0;1790.36416400000,7.46771700000000,0;1926.85446700000,7.63059400000000,0;2083.81831600000,7.80704500000000,0;2210.07184700000,7.90205700000000,0;2370.44795400000,7.96992200000000,0;2496.70148400000,7.99706900000000,0;2636.60404500000,7.95634900000000,0;2814.04144000000,7.83419100000000,0;2974.41754700000,7.76632500000000,0;3069.96075900000,7.63059400000000,0;3185.97751700000,7.30483900000000,0;3308.81879000000,7.10124200000000,0;3472.60715400000,6.84335300000000,0;3609.09745800000,6.54474400000000,0;3776.29808000000,6.25970900000000,0;3847.95548900000,5.73035700000000,0;3905.96386800000,5.32316300000000,0;3957.14773200000,5.01098100000000,0;4066.33997500000,4.65808000000000,0;4178.94447500000,4.41376400000000,0;4291.54897500000,4.22374000000000,0;4370.03090000000,3.76225400000000,0;4417.80250600000,3.26004800000000,0;4458.74959700000,2.75784200000000,0;4455.33734000000,2.24206300000000,0;4496.28443100000,1.87558900000000,0;4598.65215800000,1.57698000000000,0;4697.60762800000,1.38695700000000,0;4820.44890100000,1.14264000000000,0;4885.28179600000,0.708300000000000,0;4933.05340200000,0.341826000000000,0;1735.76804200000,9.62584400000000,0;3247.39815400000,9.46296600000000,0;4851.15922000000,9.27294200000000,0;5499.48816100000,9.02862600000000,0;1312.64810100000,8.43140900000000,0;1387.71776800000,8.15994600000000,0;1844.96028500000,8.55356700000000,0;1329.70938900000,9.85658700000000,0;2575.18340900000,8.94718700000000,0;2445.51762100000,9.80229400000000,0;3278.10847200000,8.47212800000000,0;3721.70195800000,7.34555900000000,0;4475.81088500000,5.05170000000000,0;4752.20375000000,3.72153400000000,0;5216.27078200000,1.78057700000000,0;5827.06489000000,1.14264000000000,0;5656.45201000000,1.40053000000000,0;5444.89204000000,5.18743200000000,0;4520.17023400000,7.64416700000000,0;4004.91933800000,9.69370900000000,0;4059.51545900000,7.67131400000000,0;5328.87528200000,7.22340000000000,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddDCP_CVVL, RmvDCP_CVVL, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/DCP_CVVL.png'));
    title('Ricardo DCP_CVVL Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:0.5:15.5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0 11]);
	%}
end

%% 6: Add/Remove technology: CCP_DVVL
% map_data = [CcpDvvl9p5;CcpDvvl9;CcpDvvl8p5;CcpDvvl8;CcpDvvl7p5;...
% CcpDvvl7;CcpDvvl6p5;CcpDvvl6;CcpDvvl5p5;CcpDvvl5;CcpDvvl4p5;...
% CcpDvvl4;CcpDvvl3p5;CcpDvvl3;CcpDvvl2p5;CcpDvvl2;CcpDvvl1p5;...
% CcpDvvl1;CcpDvvl0p5;CcpDvvl0];
if (AddCCP_DVVL==1 || RmvCCP_DVVL==1)
	%     './data/CCP_DVVL_20150611_2.mat'
	map_data = [996.635262000000,0.393185000000000,9.50000000000000;1100.94212700000,0,9.50000000000000;1063.93001300000,0.196592000000000,9.50000000000000;996.635262000000,0.655308000000000,9;1225.43741600000,0.0131060000000000,9;1161.50740200000,0.170380000000000,9;1060.56527600000,0.471822000000000,9;1000,0.930537000000000,8.50000000000000;1323.01480500000,0.0131060000000000,8.50000000000000;1114.40107700000,0.615990000000000,8.50000000000000;1215.34320300000,0.314548000000000,8.50000000000000;1000,2.05766700000000,8;1531.62853300000,1.00917400000000,8;1158.14266500000,1.75622500000000,8;1316.28533000000,1.41546500000000,8;1484.52220700000,0.799476000000000,8;1467.69852000000,0.471822000000000,8;1460.96904400000,0.249017000000000,8;1450.87483200000,0,8;1094.21265100000,1.88728700000000,8;1228.80215300000,1.61205800000000,8;1393.67429300000,1.25819100000000,8;1477.79273200000,1.08781100000000,8;800,2.40000000000000,8;1000,2.63433800000000,7.50000000000000;1969.04441500000,1.00917400000000,7.50000000000000;1158.14266500000,2.34600300000000,7.50000000000000;1326.37954200000,2.07077300000000,7.50000000000000;1477.79273200000,1.74311900000000,7.50000000000000;1652.75908500000,1.48099600000000,7.50000000000000;1810.90175000000,1.25819100000000,7.50000000000000;1888.29071300000,0.655308000000000,7.50000000000000;1820.99596200000,0.327654000000000,7.50000000000000;1757.06594900000,0.0262120000000000,7.50000000000000;1000,3.22411500000000,7;1158.14266500000,3.04062900000000,7;1242.26110400000,2.93578000000000,7;2409.82503400000,1.00917400000000,7;1410.49798100000,2.63433800000000,7;1578.73485900000,2.34600300000000,7;1750.33647400000,2.04456100000000,7;1918.57335100000,1.74311900000000,7;2086.81022900000,1.48099600000000,7;2244.95289400000,1.25819100000000,7;2325.70659500000,0.642202000000000,7;2258.41184400000,0.327654000000000,7;2191.11709300000,0.0131060000000000,7;996.635262000000,3.82699900000000,6.50000000000000;1481.15747000000,3.82699900000000,6.50000000000000;1242.26110400000,3.84010500000000,6.50000000000000;1693.13593500000,3.01441700000000,6.50000000000000;1124.49528900000,3.81389300000000,6.50000000000000;1329.74428000000,3.84010500000000,6.50000000000000;1578.73485900000,3.47313200000000,6.50000000000000;2672.27456300000,0.996068000000000,6.50000000000000;1864.73755000000,2.64744400000000,6.50000000000000;2029.60969000000,2.30668400000000,6.50000000000000;2177.65814300000,1.97903000000000,6.50000000000000;2345.89502000000,1.71690700000000,6.50000000000000;2473.75504700000,1.52031500000000,6.50000000000000;2571.33243600000,1.25819100000000,6.50000000000000;2638.62718700000,0.747051000000000,6.50000000000000;2604.97981200000,0.419397000000000,6.50000000000000;2554.50874800000,0,6.50000000000000;800,3.70000000000000,6.50000000000000;996.635262000000,4.22018300000000,6;1070.65948900000,4.23329000000000,6;1238.89636600000,4.31192700000000,6;1481.15747000000,4.37745700000000,6;1336.47375500000,4.31192700000000,6;1985.86810200000,4.16775900000000,6;1662.85329700000,4.29882000000000,6;1827.72543700000,4.23329000000000,6;2069.98654100000,3.97116600000000,6;2086.81022900000,3.85321100000000,6;2086.81022900000,3.64351200000000,6;2117.09286700000,3.32896500000000,6;2117.09286700000,3.14547800000000,6;2144.01076700000,2.98820400000000,6;2497.30821000000,2.14941000000000,6;2288.69448200000,2.64744400000000,6;2399.73082100000,2.38532100000000,6;2867.42934100000,1.00917400000000,6;2618.43876200000,1.82175600000000,6;2722.74562600000,1.49410200000000,6;2810.22880200000,1.21887300000000,6;2776.58142700000,0.0131060000000000,6;2837.14670300000,0.642202000000000,6;2803.49932700000,0.314548000000000,6;1000,4.52162500000000,5.50000000000000;1158.14266500000,4.57405000000000,5.50000000000000;1323.01480500000,4.67889900000000,5.50000000000000;1487.88694500000,4.73132400000000,5.50000000000000;1985.86810200000,4.65268700000000,5.50000000000000;1662.85329700000,4.69200500000000,5.50000000000000;1827.72543700000,4.66579300000000,5.50000000000000;2308.88290700000,3.97116600000000,5.50000000000000;2083.44549100000,4.42988200000000,5.50000000000000;2167.56393000000,4.27260800000000,5.50000000000000;2423.28398400000,3.43381400000000,5.50000000000000;2493.94347200000,3.15858500000000,5.50000000000000;2652.08613700000,2.76540000000000,5.50000000000000;2759.75773900000,2.43774600000000,5.50000000000000;2877.52355300000,2.11009200000000,5.50000000000000;2965.00672900000,1.78243800000000,5.50000000000000;3008.74831800000,1.49410200000000,5.50000000000000;3059.21938100000,1.03538700000000,5.50000000000000;3035.66621800000,1.27129800000000,5.50000000000000;3025.57200500000,0.615990000000000,5.50000000000000;3025.57200500000,0.458716000000000,5.50000000000000;3008.74831800000,0.288336000000000,5.50000000000000;2978.46568000000,0.0131060000000000,5.50000000000000;800,4.50000000000000,5.50000000000000;1000,4.80996100000000,5;1393.67429300000,5.03276500000000,5;1158.14266500000,4.88859800000000,5;1578.73485900000,5.05897800000000,5;1743.60699900000,5.04587200000000,5;1746.97173600000,5.05897800000000,5;1958.95020200000,5.08519000000000,5;3018.84253000000,3.05373500000000,5;2167.56393000000,4.73132400000000,5;2416.55450900000,4.24639600000000,5;2581.42664900000,3.94495400000000,5;2749.66352600000,3.61730000000000,5;2917.90040400000,3.27654000000000,5;3055.85464300000,2.62123200000000,5;3089.50201900000,2.28047200000000,5;3092.86675600000,2.11009200000000,5;3207.26783300000,1.04849300000000,5;3123.14939400000,1.76933200000000,5;3160.16150700000,1.44167800000000,5;3207.26783300000,0.943644000000000,5;3207.26783300000,0.563565000000000,5;3190.44414500000,0.445609000000000,5;3193.80888300000,0.0131060000000000,5;3190.44414500000,0.275229000000000,5;1000,5.15072100000000,4.50000000000000;1158.14266500000,5.16382700000000,4.50000000000000;1326.37954200000,5.19003900000000,4.50000000000000;1491.25168200000,5.22935800000000,4.50000000000000;1662.85329700000,5.26867600000000,4.50000000000000;1831.09017500000,5.26867600000000,4.50000000000000;1989.23284000000,5.29488900000000,4.50000000000000;2164.19919200000,5.13761500000000,4.50000000000000;2332.43607000000,4.90170400000000,4.50000000000000;2500.67294800000,4.57405000000000,4.50000000000000;2584.79138600000,4.46920100000000,4.50000000000000;2985.19515500000,3.98427300000000,4.50000000000000;2749.66352600000,4.27260800000000,4.50000000000000;3156.79677000000,3.14547800000000,4.50000000000000;3055.85464300000,3.64351200000000,4.50000000000000;3119.78465700000,3.31585800000000,4.50000000000000;3325.03364700000,1.48099600000000,4.50000000000000;3207.26783300000,2.64744400000000,4.50000000000000;3277.92732200000,1.97903000000000,4.50000000000000;3321.66891000000,1.31061600000000,4.50000000000000;3375.50471100000,0.825688000000000,4.50000000000000;3372.13997300000,0.655308000000000,4.50000000000000;3392.32839800000,0.498034000000000,4.50000000000000;3388.96366100000,0.327654000000000,4.50000000000000;3422.61103600000,0,4.50000000000000;3409.15208600000,0.157274000000000,4.50000000000000;3237.55047100000,2.30668400000000,4.50000000000000;3338.49259800000,1.15334200000000,4.50000000000000;2837.14670300000,4.16775900000000,4.50000000000000;1000,5.43905600000000,4;1158.14266500000,5.42595000000000,4;1326.37954200000,5.39973800000000,4;1494.61642000000,5.37352600000000,4;1827.72543700000,5.49148100000000,4;1911.84387600000,5.49148100000000,4;1662.85329700000,5.42595000000000,4;1992.59757700000,5.53079900000000,4;2164.19919200000,5.36041900000000,4;2332.43607000000,5.20314500000000,4;2581.42664900000,4.96723500000000,4;2497.30821000000,5.05897800000000,4;3002.01884300000,4.46920100000000,4;2749.66352600000,4.75753600000000,4;2917.90040400000,4.57405000000000,4;3109.69044400000,4.14154700000000,4;3193.80888300000,3.80078600000000,4;3257.73889600000,3.47313200000000,4;3325.03364700000,3.13237200000000,4;3358.68102300000,2.81782400000000,4;3395.69313600000,2.49017000000000,4;3392.32839800000,2.30668400000000,4;3631.22476400000,0.0131060000000000,4;3439.43472400000,1.82175600000000,4;3489.90578700000,1.31061600000000,4;3543.74158800000,0.825688000000000,4;3594.21265100000,0.327654000000000,4;1000,5.77981700000000,3.50000000000000;1158.14266500000,5.70118000000000,3.50000000000000;1323.01480500000,5.59633000000000,3.50000000000000;1491.25168200000,5.53079900000000,3.50000000000000;1985.86810200000,5.75360400000000,3.50000000000000;1662.85329700000,5.62254300000000,3.50000000000000;1831.09017500000,5.68807300000000,3.50000000000000;2507.40242300000,5.38663200000000,3.50000000000000;2167.56393000000,5.62254300000000,3.50000000000000;2329.07133200000,5.49148100000000,3.50000000000000;2991.92463000000,5.00655300000000,3.50000000000000;2665.54508700000,5.25557000000000,3.50000000000000;2837.14670300000,5.12450900000000,3.50000000000000;3348.58681000000,3.94495400000000,3.50000000000000;3116.41991900000,4.63958100000000,3.50000000000000;3234.18573400000,4.29882000000000,3.50000000000000;3533.64737600000,2.77850600000000,3.50000000000000;3419.24629900000,3.47313200000000,3.50000000000000;3483.17631200000,3.09305400000000,3.50000000000000;3560.56527600000,2.49017000000000,3.50000000000000;3600.94212700000,2.14941000000000,3.50000000000000;3644.68371500000,1.82175600000000,3.50000000000000;3695.15477800000,1.49410200000000,3.50000000000000;3745.62584100000,1.17955400000000,3.50000000000000;3786.00269200000,0.825688000000000,3.50000000000000;3823.01480500000,0.314548000000000,3.50000000000000;3856.66218000000,0,3.50000000000000;3809.55585500000,0.563565000000000,3.50000000000000;1000,6.04194000000000,3;1077.38896400000,6.02883400000000,3;1232.16689100000,5.88466600000000,3;1481.15747000000,5.70118000000000,3;1380.21534300000,5.76671000000000,3;1982.50336500000,6.02883400000000,3;1666.21803500000,5.80602900000000,3;1831.09017500000,5.92398400000000,3;2504.03768500000,5.72739200000000,3;2164.19919200000,5.91087800000000,3;2332.43607000000,5.81913500000000,3;2995.28936700000,5.33420700000000,3;2668.90982500000,5.58322400000000,3;2837.14670300000,5.45216300000000,3;3210.63257100000,4.86238500000000,3;3099.59623100000,5.11140200000000,3;3503.36473800000,3.97116600000000,3;3314.93943500000,4.54783700000000,3;3415.88156100000,4.25950200000000,3;3688.42530300000,2.81782400000000,3;3557.20053800000,3.63040600000000,3;3637.95424000000,3.14547800000000,3;3997.98115700000,1.03538700000000,3;3738.89636600000,2.47706400000000,3;3779.27321700000,2.14941000000000,3;3802.82638000000,1.97903000000000,3;3870.12113100000,1.65137600000000,3;3934.05114400000,1.31061600000000,3;4021.53432000000,0.642202000000000,3;4048.45222100000,0.314548000000000,3;4068.64064600000,0,3;800,6.20000000000000,3;1000,6.19921400000000,2.50000000000000;1161.50740200000,6.08125800000000,2.50000000000000;1282.63795400000,6.02883400000000,2.50000000000000;1474.42799500000,5.85845300000000,2.50000000000000;1363.39165500000,5.93709000000000,2.50000000000000;1979.13862700000,6.26474400000000,2.50000000000000;1662.85329700000,5.98951500000000,2.50000000000000;1831.09017500000,6.10747100000000,2.50000000000000;2493.94347200000,6.09436400000000,2.50000000000000;2167.56393000000,6.18610700000000,2.50000000000000;2339.16554500000,6.13368300000000,2.50000000000000;2991.92463000000,5.66186100000000,2.50000000000000;2668.90982500000,5.92398400000000,2.50000000000000;2837.14670300000,5.77981700000000,2.50000000000000;3338.49259800000,4.96723500000000,2.50000000000000;3089.50201900000,5.46526900000000,2.50000000000000;3257.73889600000,5.12450900000000,2.50000000000000;3483.17631200000,4.56094400000000,2.50000000000000;3577.38896400000,4.16775900000000,2.50000000000000;3654.77792700000,3.80078600000000,2.50000000000000;3732.16689100000,3.47313200000000,2.50000000000000;3796.09690400000,3.15858500000000,2.50000000000000;3863.39165500000,2.81782400000000,2.50000000000000;3917.22745600000,2.50327700000000,2.50000000000000;3971.06325700000,2.14941000000000,2.50000000000000;4018.16958300000,1.80865000000000,2.50000000000000;4068.64064600000,1.34993400000000,2.50000000000000;4146.02961000000,0.825688000000000,2.50000000000000;4199.86541000000,0.498034000000000,2.50000000000000;4283.98384900000,0.0131060000000000,2.50000000000000;4240.24226100000,0.235911000000000,2.50000000000000;1000,6.38270000000000,2;1454.23956900000,6.05504600000000,2;1161.50740200000,6.25163800000000,2;1329.74428000000,6.12057700000000,2;1497.98115700000,6.05504600000000,2;1982.50336500000,6.52686800000000,2;1662.85329700000,6.21232000000000,2;1837.81965000000,6.38270000000000,2;2487.21399700000,6.57929200000000,2;2167.56393000000,6.55308000000000,2;2335.80080800000,6.55308000000000,2;2995.28936700000,6.02883400000000,2;2668.90982500000,6.39580600000000,2;2837.14670300000,6.18610700000000,2;3513.45895000000,4.94102200000000,2;3096.23149400000,5.79292300000000,2;3247.64468400000,5.49148100000000,2;3409.15208600000,5.16382700000000,2;3580.75370100000,4.66579300000000,2;3698.51951500000,4.18086500000000,2;3826.37954200000,3.68283100000000,2;3987.88694500000,3.09305400000000,2;3917.22745600000,3.35517700000000,2;4011.44010800000,2.89646100000000,2;4048.45222100000,2.51638300000000,2;4098.92328400000,2.00524200000000,2;4169.58277300000,1.52031500000000,2;4220.05383600000,1.17955400000000,2;4250.33647400000,1.02228000000000,2;4499.32705200000,0.0131060000000000,2;4331.09017500000,0.681520000000000,2;4415.20861400000,0.353866000000000,2;1000,6.51376100000000,1.50000000000000;1242.26110400000,6.39580600000000,1.50000000000000;1161.50740200000,6.42201800000000,1.50000000000000;1454.23956900000,6.27785100000000,1.50000000000000;1326.37954200000,6.33027500000000,1.50000000000000;1497.98115700000,6.29095700000000,1.50000000000000;1410.49798100000,6.29095700000000,1.50000000000000;1982.50336500000,6.76277900000000,1.50000000000000;1662.85329700000,6.44823100000000,1.50000000000000;1831.09017500000,6.61861100000000,1.50000000000000;2497.30821000000,7.06422000000000,1.50000000000000;2167.56393000000,6.84141500000000,1.50000000000000;2332.43607000000,6.95937100000000,1.50000000000000;2672.27456300000,6.88073400000000,1.50000000000000;3002.01884300000,6.48754900000000,1.50000000000000;2840.51144000000,6.68414200000000,1.50000000000000;3493.27052500000,5.49148100000000,1.50000000000000;3173.62045800000,6.15989500000000,1.50000000000000;3341.85733500000,5.81913500000000,1.50000000000000;3614.40107700000,5.00655300000000,1.50000000000000;4109.01749700000,3.05373500000000,1.50000000000000;3742.26110400000,4.50851900000000,1.50000000000000;3870.12113100000,3.99737900000000,1.50000000000000;3997.98115700000,3.49934500000000,1.50000000000000;4146.02961000000,2.66055000000000,1.50000000000000;4149.39434700000,2.51638300000000,1.50000000000000;4196.50067300000,2.00524200000000,1.50000000000000;4233.51278600000,1.84796900000000,1.50000000000000;4246.97173600000,1.67758800000000,1.50000000000000;4314.26648700000,1.33682800000000,1.50000000000000;4327.72543700000,1.19266100000000,1.50000000000000;4415.20861400000,0.851900000000000,1.50000000000000;4499.32705200000,0.524246000000000,1.50000000000000;4566.62180300000,0.196592000000000,1.50000000000000;4617.09286700000,0,1.50000000000000;1003.36473800000,6.68414200000000,1;1161.50740200000,6.63171700000000,1;1242.26110400000,6.57929200000000,1;1329.74428000000,6.57929200000000,1;1410.49798100000,6.55308000000000,1;1494.61642000000,6.53997400000000,1;2053.16285300000,7.07732600000000,1;1666.21803500000,6.72346000000000,1;1841.18438800000,6.88073400000000,1;2493.94347200000,7.36566200000000,1;2251.68236900000,7.18217600000000,1;2998.65410500000,7.01179600000000,1;2668.90982500000,7.20838800000000,1;2837.14670300000,7.11664500000000,1;3503.36473800000,6.00262100000000,1;3176.98519500000,6.65792900000000,1;3345.22207300000,6.33027500000000,1;4250.33647400000,3.01441700000000,1;3580.75370100000,5.66186100000000,1;3708.61372800000,5.16382700000000,1;3829.74428000000,4.66579300000000,1;3950.87483200000,4.16775900000000,1;4085.46433400000,3.68283100000000,1;4169.58277300000,3.34207100000000,1;4250.33647400000,2.83093100000000,1;4263.79542400000,2.67365700000000,1;4263.79542400000,2.50327700000000,1;4280.61911200000,2.33289600000000,1;4280.61911200000,2.17562300000000,1;4297.44279900000,2.00524200000000,1;4331.09017500000,1.84796900000000,1;4398.38492600000,1.52031500000000,1;4499.32705200000,1.00917400000000,1;4583.44549100000,0.681520000000000,1;4667.56393000000,0.353866000000000,1;4755.04710600000,-0.0131060000000000,1;996.635262000000,6.82830900000000,0.500000000000000;1164.87214000000,6.80209700000000,0.500000000000000;1326.37954200000,6.77588500000000,0.500000000000000;1410.49798100000,6.73656600000000,0.500000000000000;1491.25168200000,6.77588500000000,0.500000000000000;1985.86810200000,7.48361700000000,0.500000000000000;1662.85329700000,6.94626500000000,0.500000000000000;1827.72543700000,7.19528200000000,0.500000000000000;2497.30821000000,7.68021000000000,0.500000000000000;2164.19919200000,7.53604200000000,0.500000000000000;2332.43607000000,7.60157300000000,0.500000000000000;2998.65410500000,7.47051100000000,0.500000000000000;2753.02826400000,7.57536000000000,0.500000000000000;2628.53297400000,7.60157300000000,0.500000000000000;2870.79407800000,7.52293600000000,0.500000000000000;3748.99057900000,6.00262100000000,0.500000000000000;3176.98519500000,7.14285700000000,0.500000000000000;3341.85733500000,6.81520300000000,0.500000000000000;3510.09421300000,6.47444300000000,0.500000000000000;3799.46164200000,5.60943600000000,0.500000000000000;3846.56796800000,5.15072100000000,0.500000000000000;3913.86271900000,4.82306700000000,0.500000000000000;3997.98115700000,4.49541300000000,0.500000000000000;4162.85329700000,4.16775900000000,0.500000000000000;4257.06594900000,3.95806000000000,0.500000000000000;4310.90175000000,3.49934500000000,0.500000000000000;4368.10228800000,3.04062900000000,0.500000000000000;4368.10228800000,2.84403700000000,0.500000000000000;4388.29071300000,2.67365700000000,0.500000000000000;4388.29071300000,2.34600300000000,0.500000000000000;4408.47913900000,1.99213600000000,0.500000000000000;4472.40915200000,1.66448200000000,0.500000000000000;4502.69179000000,1.50720800000000,0.500000000000000;4751.68236900000,1.00917400000000,0.500000000000000;4808.88290700000,0.524246000000000,0.500000000000000;4869.44818300000,-0.0131060000000000,0.500000000000000;4586.81022900000,1.33682800000000,0.500000000000000;4670.92866800000,1.17955400000000,0.500000000000000;800,6.80000000000000,0.500000000000000;993.270525000000,7.99475800000000,0;993.270525000000,10.0131060000000,0;993.270525000000,10.8912190000000,0;1709.95962300000,10.8912190000000,0;2544.41453600000,10.8912190000000,0;3452.89367400000,10.9043250000000,0;4384.92597600000,10.9043250000000,0;5313.59354000000,10.9043250000000,0;5754.37415900000,10.8912190000000,0;6010.09421300000,10.9043250000000,0;6016.82368800000,9.39711700000000,0;6013.45895000000,7.81127100000000,0;6016.82368800000,6.12057700000000,0;6013.45895000000,4.19397100000000,0;6013.45895000000,2.51638300000000,0;6013.45895000000,1.14023600000000,0;6013.45895000000,-0.0131060000000000,0;5528.93674300000,-0.0262120000000000,0;5104.97981200000,-0.0131060000000000,0;993.270525000000,7.02490200000000,0;1111.03633900000,6.98558300000000,0;1232.16689100000,6.93315900000000,0;1346.56796800000,6.93315900000000,0;1440.78061900000,6.95937100000000,0;1548.45222100000,7.05111400000000,0;1669.58277300000,7.20838800000000,0;1790.71332400000,7.35255600000000,0;1891.65545100000,7.56225400000000,0;1989.23284000000,7.69331600000000,0;2127.18707900000,7.71952800000000,0;2255.04710600000,7.78505900000000,0;2379.54239600000,7.82437700000000,0;2487.21399700000,7.90301400000000,0;2615.07402400000,7.85059000000000,0;2769.85195200000,7.77195300000000,0;2904.44145400000,7.75884700000000,0;3005.38358000000,7.77195300000000,0;3143.33782000000,7.44429900000000,0;3281.29205900000,7.23460000000000,0;3372.13997300000,7.01179600000000,0;3500,6.78899100000000,0;3614.40107700000,6.53997400000000,0;3745.62584100000,6.30406300000000,0;3823.01480500000,6.15989500000000,0;3860.02691800000,5.62254300000000,0;3930.68640600000,5.12450900000000,0;4004.71063300000,4.80996100000000,0;4105.65275900000,4.53473100000000,0;4223.41857300000,4.33813900000000,0;4314.26648700000,4.14154700000000,0;4388.29071300000,3.46002600000000,0;4358.00807500000,3.78768000000000,0;4425.30282600000,2.98820400000000,0;4465.67967700000,2.67365700000000,0;4442.12651400000,2.38532100000000,0;4482.50336500000,1.92660600000000,0;4580.08075400000,1.63827000000000,0;4694.48183000000,1.37614700000000,0;4802.15343200000,1.16644800000000,0;4855.98923300000,0.694626000000000,0;4896.36608300000,0.262123000000000,0;4919.91924600000,0,0;1497.98115700000,9.67234600000000,0;2483.84926000000,9.48886000000000,0;3792.73216700000,9.50196600000000,0;4738.22341900000,9.34469200000000,0;5438.08882900000,9.12188700000000,0;5485.19515500000,6.47444300000000,0;5535.66621800000,3.09305400000000,0;4956.93135900000,2.43774600000000,0;5623.14939400000,1.36304100000000,0;4731.49394300000,4.11533400000000,0;4324.36070000000,6.02883400000000,0;4024.89905800000,7.79816500000000,0;3102.96096900000,8.68938400000000,0;1925.30282600000,8.83355200000000,0;1366.75639300000,8.12581900000000,0;1201.88425300000,9.12188700000000,0;4755.04710600000,7.96854500000000,0;5081.42664900000,5.81913500000000,0;800,7.40000000000000,0;800,9,0;800,11,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddCCP_DVVL, RmvCCP_DVVL, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/CCP_DVVL.png'));
    title('Ricardo CCP_DVVL Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:0.5:9.5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0 11]);
	%}
end


%% 7: Add/Remove technology: Camless
% map_data = [Camless24;Camless23;Camless22;Camless21;Camless20;...
% Camless19;Camless18;Camless17;Camless16;Camless15;Camless14;...
% Camless13;Camless12;Camless11;Camless10;Camless9;Camless8;...
% Camless7;Camless6;Camless5;Camless4;Camless3;Camless2;Camless1;Camless0];

if (AddCamless==1 || RmvCamless==1)
	%     './data/Camless_20150611_2.mat'
	map_data = [1003.38066300000,0.212500000000000,24;1175.79445600000,0.0125000000000000,24;1077.75524000000,0.100000000000000,24;1003.38066300000,0.425000000000000,23;1361.73089900000,0,23;1081.13590300000,0.325000000000000,23;1158.89114300000,0.237500000000000,23;1250.16903300000,0.137500000000000,23;1000,0.625000000000000,22;1608.51927000000,0,22;1162.27180500000,0.462500000000000,22;1334.68559800000,0.300000000000000,22;1496.95740400000,0.162500000000000,22;1003.38066300000,0.800000000000000,21;1956.72751900000,0,21;1162.27180500000,0.662500000000000,21;1331.30493600000,0.525000000000000,21;1412.44083800000,0.487500000000000,21;1500.33806600000,0.425000000000000,21;1665.99053400000,0.262500000000000,21;1835.02366500000,0.125000000000000,21;1000,1,20;2426.63962100000,0.0250000000000000,20;1162.27180500000,0.925000000000000,20;1331.30493600000,0.812500000000000,20;1412.44083800000,0.762500000000000,20;1500.33806600000,0.737500000000000,20;1747.12643700000,0.537500000000000,20;1665.99053400000,0.600000000000000,20;1584.85463200000,0.662500000000000,20;1835.02366500000,0.462500000000000,20;1916.15956700000,0.400000000000000,20;2000.67613300000,0.375000000000000,20;2085.19269800000,0.300000000000000,20;2254.22582800000,0.162500000000000,20;1003.38066300000,1.21250000000000,19;1162.27180500000,1.15000000000000,19;1327.92427300000,1.08750000000000,19;1500.33806600000,1.02500000000000,19;1662.60987200000,0.887500000000000,19;1750.50709900000,0.862500000000000,19;1831.64300200000,0.787500000000000,19;1916.15956700000,0.725000000000000,19;2000.67613300000,0.700000000000000,19;2166.32860000000,0.625000000000000,19;2338.74239400000,0.587500000000000,19;2507.77552400000,0.525000000000000,19;2588.91142700000,0.462500000000000,19;3430.69641600000,0.0125000000000000,19;2754.56389500000,0.387500000000000,19;2923.59702500000,0.287500000000000,19;3008.11359000000,0.250000000000000,19;3092.63015600000,0.187500000000000,19;3261.66328600000,0.112500000000000,19;3177.14672100000,0.150000000000000,19;1003.38066300000,1.40000000000000,18;1250.16903300000,1.32500000000000,18;1162.27180500000,1.36250000000000,18;1334.68559800000,1.31250000000000,18;1500.33806600000,1.25000000000000,18;1912.77890500000,1.05000000000000,18;1581.47396900000,1.21250000000000,18;1665.99053400000,1.15000000000000,18;1750.50709900000,1.11250000000000,18;1835.02366500000,1.08750000000000,18;2511.15618700000,1.02500000000000,18;2088.57336000000,1.02500000000000,18;2254.22582800000,1.02500000000000,18;2419.87829600000,1.02500000000000,18;2673.42799200000,0.912500000000000,18;2842.46112200000,0.812500000000000,18;3008.11359000000,0.750000000000000,18;3173.76605800000,0.662500000000000,18;3342.79918900000,0.587500000000000,18;3511.83231900000,0.525000000000000,18;3765.38201500000,0.412500000000000,18;3592.96822200000,0.487500000000000,18;3687.62677500000,0.450000000000000,18;3846.51791800000,0.412500000000000,18;3931.03448300000,0.387500000000000,18;4012.17038500000,0.362500000000000,18;4350.23664600000,0.362500000000000,18;4688.30290700000,0.362500000000000,18;5026.36916800000,0.362500000000000,18;5361.05476700000,0.362500000000000,18;5607.84313700000,0.362500000000000,18;6006.76132500000,0.362500000000000,18;6500,0.362500000000000,18;1003.38066300000,1.58750000000000,17;1165.65246800000,1.57500000000000,17;1327.92427300000,1.55000000000000,17;1500.33806600000,1.51250000000000,17;1750.50709900000,1.41250000000000,17;1665.99053400000,1.45000000000000,17;1584.85463200000,1.47500000000000,17;1831.64300200000,1.41250000000000,17;1997.29547000000,1.35000000000000,17;2169.70926300000,1.38750000000000,17;2338.74239400000,1.37500000000000,17;2504.39486100000,1.42500000000000,17;2670.04732900000,1.38750000000000,17;2839.08046000000,1.38750000000000,17;3011.49425300000,1.35000000000000,17;3261.66328600000,1.17500000000000,17;3427.31575400000,1.05000000000000,17;3092.63015600000,1.27500000000000,17;3173.76605800000,1.20000000000000,17;3346.17985100000,1.11250000000000,17;3762.00135200000,1.02500000000000,17;4181.20351600000,1.02500000000000,17;4613.92833000000,1.02500000000000,17;4992.56254200000,1.02500000000000,17;5317.10615300000,1.02500000000000,17;5655.17241400000,1.02500000000000,17;6003.38066300000,1.02500000000000,17;6500,1.02500000000000,17;1003.38066300000,1.80000000000000,16;1077.75524000000,1.81250000000000,16;1162.27180500000,1.77500000000000,16;1327.92427300000,1.77500000000000,16;1496.95740400000,1.75000000000000,16;1578.09330600000,1.75000000000000,16;1665.99053400000,1.72500000000000,16;1835.02366500000,1.71250000000000,16;1916.15956700000,1.68750000000000,16;2085.19269800000,1.72500000000000,16;2166.32860000000,1.71250000000000,16;2338.74239400000,1.77500000000000,16;2419.87829600000,1.77500000000000,16;3004.73292800000,2.01250000000000,16;2585.53076400000,1.83750000000000,16;2764.70588200000,1.91250000000000,16;2923.59702500000,1.97500000000000,16;3173.76605800000,1.88750000000000,16;3258.28262300000,1.83750000000000,16;3427.31575400000,1.72500000000000,16;3511.83231900000,1.68750000000000,16;3596.34888400000,1.75000000000000,16;3846.51791800000,1.91250000000000,16;3721.43340100000,1.82500000000000,16;3934.41514500000,1.98750000000000,16;4015.55104800000,2.02500000000000,16;4356.99797200000,2.02500000000000,16;4803.24543600000,2.02500000000000,16;5205.54428700000,2.02500000000000,16;5617.98512500000,2.02500000000000,16;6003.38066300000,2.02500000000000,16;4150,2.02500000000000,16;4500,2.02500000000000,16;5000,2.02500000000000,16;5400,2.02500000000000,16;6500,2.02500000000000,16;1006.76132500000,2.02500000000000,15;1165.65246800000,2.02500000000000,15;1919.54023000000,2.02500000000000,15;1412.44083800000,2.02500000000000,15;1665.99053400000,2.02500000000000,15;2000.67613300000,2.02500000000000,15;2842.46112200000,2.31250000000000,15;2166.32860000000,2.07500000000000,15;2335.36173100000,2.15000000000000,15;2504.39486100000,2.21250000000000,15;2588.91142700000,2.25000000000000,15;2673.42799200000,2.25000000000000,15;2754.56389500000,2.27500000000000,15;2923.59702500000,2.31250000000000,15;3008.11359000000,2.35000000000000,15;3092.63015600000,2.31250000000000,15;3173.76605800000,2.31250000000000,15;3261.66328600000,2.27500000000000,15;3342.79918900000,2.25000000000000,15;3427.31575400000,2.23750000000000,15;3511.83231900000,2.21250000000000,15;3592.96822200000,2.23750000000000,15;3677.48478700000,2.23750000000000,15;3762.00135200000,2.27500000000000,15;3846.51791800000,2.31250000000000,15;3931.03448300000,2.31250000000000,15;4015.55104800000,2.35000000000000,15;4096.68695100000,2.35000000000000,15;4356.99797200000,2.35000000000000,15;4691.68357000000,2.35000000000000,15;5093.98242100000,2.35000000000000,15;5455.71332000000,2.35000000000000,15;5746.45030400000,2.35000000000000,15;6003.38066300000,2.35000000000000,15;6500,2.35000000000000,15;800,2,15;1003.38066300000,2.37500000000000,14;1077.75524000000,2.35000000000000,14;1162.27180500000,2.38750000000000,14;1246.78837100000,2.37500000000000,14;1327.92427300000,2.37500000000000,14;1412.44083800000,2.41250000000000,14;1500.33806600000,2.41250000000000,14;1993.91480700000,2.51250000000000,14;1584.85463200000,2.43750000000000,14;1665.99053400000,2.43750000000000,14;1750.50709900000,2.46250000000000,14;1919.54023000000,2.50000000000000,14;2088.57336000000,2.55000000000000,14;2169.70926300000,2.53750000000000,14;2254.22582800000,2.57500000000000,14;2338.74239400000,2.57500000000000,14;2419.87829600000,2.61250000000000,14;2504.39486100000,2.60000000000000,14;2585.53076400000,2.61250000000000,14;2673.42799200000,2.63750000000000,14;2754.56389500000,2.63750000000000,14;2842.46112200000,2.63750000000000,14;2923.59702500000,2.68750000000000,14;3008.11359000000,2.68750000000000,14;3092.63015600000,2.68750000000000,14;3173.76605800000,2.65000000000000,14;3261.66328600000,2.63750000000000,14;3342.79918900000,2.65000000000000,14;3427.31575400000,2.61250000000000,14;3511.83231900000,2.61250000000000,14;3596.34888400000,2.61250000000000,14;3927.65382000000,2.68750000000000,14;3758.62069000000,2.63750000000000,14;4275.86206900000,2.68750000000000,14;4620.68965500000,2.68750000000000,14;4985.80121700000,2.68750000000000,14;5381.33874200000,2.68750000000000,14;5712.64367800000,2.68750000000000,14;6000,2.68750000000000,14;1003.38066300000,2.70000000000000,13;1077.75524000000,2.71250000000000,13;1162.27180500000,2.71250000000000,13;1246.78837100000,2.75000000000000,13;1327.92427300000,2.77500000000000,13;1412.44083800000,2.77500000000000,13;1496.95740400000,2.81250000000000,13;1581.47396900000,2.83750000000000,13;1669.37119700000,2.87500000000000,13;1750.50709900000,2.91250000000000,13;1835.02366500000,2.95000000000000,13;1916.15956700000,2.97500000000000,13;2000.67613300000,3.01250000000000,13;2088.57336000000,3.01250000000000,13;2521.29817400000,3.01250000000000,13;3035.15889100000,3.01250000000000,13;3579.44557100000,3.01250000000000,13;4228.53279200000,3.01250000000000,13;4624.07031800000,3.01250000000000,13;5104.12440800000,3.01250000000000,13;5617.98512500000,3.01250000000000,13;6000,3.01250000000000,13;2169.70926300000,3.01250000000000,13;2338.74239400000,3.01250000000000,13;2757.94455700000,3.01250000000000,13;1003.38066300000,3.03750000000000,12;1081.13590300000,3.05000000000000,12;1165.65246800000,3.05000000000000,12;1246.78837100000,3.08750000000000,12;1331.30493600000,3.11250000000000,12;1409.06017600000,3.11250000000000,12;1500.33806600000,3.13750000000000,12;1997.29547000000,3.35000000000000,12;1665.99053400000,3.21250000000000,12;1835.02366500000,3.27500000000000,12;2436.78160900000,3.35000000000000,12;2933.73901300000,3.35000000000000,12;3423.93509100000,3.35000000000000,12;3863.42123100000,3.35000000000000,12;4444.89519900000,3.35000000000000,12;5029.74983100000,3.35000000000000,12;5526.70723500000,3.35000000000000,12;6003.38066300000,3.35000000000000,12;2085.19269800000,3.35000000000000,12;2250.84516600000,3.35000000000000,12;2676.80865400000,3.35000000000000,12;2200,3.35000000000000,12;2600,3.35000000000000,12;3600,3.35000000000000,12;4000,3.35000000000000,12;4600,3.35000000000000,12;4800,3.35000000000000,12;5300,3.35000000000000,12;5800,3.35000000000000,12;6500,3.35000000000000,12;1006.76132500000,3.37500000000000,11;1081.13590300000,3.37500000000000,11;1162.27180500000,3.37500000000000,11;1246.78837100000,3.41250000000000,11;1331.30493600000,3.41250000000000,11;1412.44083800000,3.43750000000000,11;1500.33806600000,3.43750000000000,11;1581.47396900000,3.46250000000000,11;1665.99053400000,3.53750000000000,11;1750.50709900000,3.56250000000000,11;2000.67613300000,3.67500000000000,11;1919.54023000000,3.63750000000000,11;1835.02366500000,3.60000000000000,11;2423.25895900000,3.67500000000000,11;2926.97768800000,3.67500000000000,11;3430.69641600000,3.67500000000000,11;3931.03448300000,3.67500000000000,11;4434.75321200000,3.67500000000000,11;4935.09127800000,3.67500000000000,11;5357.67410400000,3.67500000000000,11;5702.50169000000,3.67500000000000,11;6010.14198800000,3.67500000000000,11;2085.19269800000,3.67500000000000,11;2179.85125100000,3.67500000000000,11;2680.18931700000,3.67500000000000,11;3183.90804600000,3.67500000000000,11;3667.34279900000,3.67500000000000,11;4181.20351600000,3.67500000000000,11;4684.92224500000,3.67500000000000,11;5151.45368500000,3.67500000000000,11;5523.32657200000,3.67500000000000,11;5868.15415800000,3.67500000000000,11;1000,3.66250000000000,10;1077.75524000000,3.67500000000000,10;1162.27180500000,3.71250000000000,10;1246.78837100000,3.71250000000000,10;1331.30493600000,3.71250000000000,10;1412.44083800000,3.75000000000000,10;1500.33806600000,3.75000000000000,10;1578.09330600000,3.80000000000000,10;1665.99053400000,3.83750000000000,10;1750.50709900000,3.86250000000000,10;2004.05679500000,4.01250000000000,10;1919.54023000000,3.97500000000000,10;1835.02366500000,3.93750000000000,10;2426.63962100000,4.01250000000000,10;2933.73901300000,4.01250000000000,10;3549.01960800000,4.01250000000000,10;4093.30628800000,4.01250000000000,10;4583.50236600000,4.01250000000000,10;5039.89181900000,4.01250000000000,10;5553.75253500000,4.01250000000000,10;6006.76132500000,4.01250000000000,10;2095.33468600000,4.01250000000000,10;2254.22582800000,4.01250000000000,10;2676.80865400000,4.01250000000000,10;3265.04394900000,4.01250000000000,10;3849.89858000000,4.01250000000000,10;4346.85598400000,4.01250000000000,10;4837.05206200000,4.01250000000000,10;5313.72549000000,4.01250000000000,10;5810.68289400000,4.01250000000000,10;6500,4.01250000000000,10;800,3.70000000000000,10;1000,4.01250000000000,9;1162.27180500000,4,9;1246.78837100000,4.01250000000000,9;1500.33806600000,4.01250000000000,9;1361.73089900000,4.01250000000000,9;1581.47396900000,4.05000000000000,9;1990.53414500000,4.23750000000000,9;1747.12643700000,4.13750000000000,9;1835.02366500000,4.17500000000000,9;1916.15956700000,4.20000000000000,9;2081.81203500000,4.23750000000000,9;2173.08992600000,4.23750000000000,9;2254.22582800000,4.20000000000000,9;2335.36173100000,4.18750000000000,9;2504.39486100000,4.20000000000000,9;2419.87829600000,4.20000000000000,9;2754.56389500000,4.20000000000000,9;3008.11359000000,4.20000000000000,9;3261.66328600000,4.20000000000000,9;3508.45165700000,4.20000000000000,9;3772.14334000000,4.20000000000000,9;4100.06761300000,4.20000000000000,9;4431.37254900000,4.20000000000000,9;4826.91007400000,4.20000000000000,9;5158.21501000000,4.20000000000000,9;5523.32657200000,4.20000000000000,9;5824.20554400000,4.20000000000000,9;6003.38066300000,4.20000000000000,9;1006.76132500000,4.35000000000000,8;1162.27180500000,4.33750000000000,8;1327.92427300000,4.33750000000000,8;1496.95740400000,4.33750000000000,8;1578.09330600000,4.37500000000000,8;1665.99053400000,4.37500000000000,8;1990.53414500000,4.47500000000000,8;1835.02366500000,4.43750000000000,8;1750.50709900000,4.40000000000000,8;2088.57336000000,4.46250000000000,8;2169.70926300000,4.43750000000000,8;2338.74239400000,4.43750000000000,8;2419.87829600000,4.41250000000000,8;2504.39486100000,4.41250000000000,8;2676.80865400000,4.41250000000000,8;2923.59702500000,4.41250000000000,8;3430.69641600000,4.41250000000000,8;3931.03448300000,4.41250000000000,8;4434.75321200000,4.41250000000000,8;4935.09127800000,4.41250000000000,8;5448.95199500000,4.41250000000000,8;6006.76132500000,4.41250000000000,8;1003.38066300000,4.67500000000000,7;1081.13590300000,4.67500000000000,7;1246.78837100000,4.67500000000000,7;1415.82150100000,4.67500000000000,7;1669.37119700000,4.67500000000000,7;1922.92089200000,4.67500000000000,7;1753.88776200000,4.67500000000000,7;2088.57336000000,4.67500000000000,7;2169.70926300000,4.62500000000000,7;2257.60649100000,4.62500000000000,7;2338.74239400000,4.62500000000000,7;2419.87829600000,4.58750000000000,7;2507.77552400000,4.60000000000000,7;2673.42799200000,4.60000000000000,7;2923.59702500000,4.60000000000000,7;3427.31575400000,4.60000000000000,7;3931.03448300000,4.60000000000000,7;4438.13387400000,4.60000000000000,7;4938.47194100000,4.60000000000000,7;5438.81000700000,4.60000000000000,7;6003.38066300000,4.60000000000000,7;5776.87626800000,4.60000000000000,7;1006.76132500000,5.01250000000000,6;1081.13590300000,5,6;1165.65246800000,5,6;1246.78837100000,5,6;1331.30493600000,5,6;1412.44083800000,5,6;1500.33806600000,5,6;1584.85463200000,4.96250000000000,6;1669.37119700000,4.96250000000000,6;1750.50709900000,4.92500000000000,6;1835.02366500000,4.92500000000000,6;1919.54023000000,4.90000000000000,6;2004.05679500000,4.90000000000000,6;2088.57336000000,4.86250000000000,6;2169.70926300000,4.86250000000000,6;2254.22582800000,4.82500000000000,6;2338.74239400000,4.82500000000000,6;2419.87829600000,4.80000000000000,6;2507.77552400000,4.80000000000000,6;2588.91142700000,4.80000000000000,6;2676.80865400000,4.80000000000000,6;2926.97768800000,4.80000000000000,6;3427.31575400000,4.80000000000000,6;3934.41514500000,4.80000000000000,6;4438.13387400000,4.80000000000000,6;4938.47194100000,4.80000000000000,6;5445.57133200000,4.80000000000000,6;5776.87626800000,4.80000000000000,6;6006.76132500000,4.80000000000000,6;1003.38066300000,5.47500000000000,5;1162.27180500000,5.43750000000000,5;1331.30493600000,5.32500000000000,5;1500.33806600000,5.26250000000000,5;1588.23529400000,5.26250000000000,5;1669.37119700000,5.22500000000000,5;1753.88776200000,5.22500000000000,5;1835.02366500000,5.22500000000000,5;1919.54023000000,5.18750000000000,5;2004.05679500000,5.20000000000000,5;2169.70926300000,5.12500000000000,5;2338.74239400000,5.06250000000000,5;2419.87829600000,5.03750000000000,5;2507.77552400000,5,5;2676.80865400000,5,5;2926.97768800000,5,5;3427.31575400000,5,5;3931.03448300000,5,5;4438.13387400000,5,5;4941.85260300000,5,5;5445.57133200000,5,5;5776.87626800000,5,5;6006.76132500000,5,5;1000,6.02500000000000,4;1081.13590300000,5.90000000000000,4;1246.78837100000,5.73750000000000,4;1415.82150100000,5.56250000000000,4;1500.33806600000,5.50000000000000,4;1584.85463200000,5.52500000000000,4;1750.50709900000,5.56250000000000,4;1669.37119700000,5.52500000000000,4;1922.92089200000,5.60000000000000,4;1835.02366500000,5.56250000000000,4;2004.05679500000,5.58750000000000,4;2088.57336000000,5.55000000000000,4;2173.08992600000,5.55000000000000,4;2257.60649100000,5.52500000000000,4;2342.12305600000,5.52500000000000,4;2423.25895900000,5.48750000000000,4;2507.77552400000,5.48750000000000,4;2588.91142700000,5.46250000000000,4;2676.80865400000,5.42500000000000,4;2757.94455700000,5.40000000000000,4;2842.46112200000,5.40000000000000,4;2926.97768800000,5.36250000000000,4;3011.49425300000,5.32500000000000,4;3092.63015600000,5.32500000000000,4;3177.14672100000,5.30000000000000,4;3265.04394900000,5.28750000000000,4;3346.17985100000,5.30000000000000,4;3427.31575400000,5.26250000000000,4;3515.21298200000,5.26250000000000,4;3596.34888400000,5.26250000000000,4;3680.86545000000,5.22500000000000,4;3765.38201500000,5.22500000000000,4;3849.89858000000,5.22500000000000,4;3934.41514500000,5.20000000000000,4;4103.44827600000,5.20000000000000,4;4272.48140600000,5.20000000000000,4;4438.13387400000,5.20000000000000,4;4941.85260300000,5.20000000000000,4;5445.57133200000,5.20000000000000,4;5780.25693000000,5.20000000000000,4;6010.14198800000,5.20000000000000,4;6500,5.20000000000000,4;800,6.20000000000000,4;1000,6.26250000000000,3;1496.95740400000,5.76250000000000,3;1165.65246800000,6.10000000000000,3;1334.68559800000,5.92500000000000,3;1665.99053400000,5.85000000000000,3;1835.02366500000,5.95000000000000,3;2004.05679500000,6.02500000000000,3;2088.57336000000,6.02500000000000,3;2260.98715300000,6.02500000000000,3;2423.25895900000,6.02500000000000,3;2511.15618700000,6.02500000000000,3;2676.80865400000,5.88750000000000,3;2845.84178500000,5.76250000000000,3;2923.59702500000,5.68750000000000,3;3014.87491500000,5.66250000000000,3;3096.01081800000,5.62500000000000,3;3177.14672100000,5.58750000000000,3;3265.04394900000,5.53750000000000,3;3346.17985100000,5.56250000000000,3;3430.69641600000,5.52500000000000,3;3515.21298200000,5.48750000000000,3;3596.34888400000,5.46250000000000,3;3684.24611200000,5.46250000000000,3;3765.38201500000,5.43750000000000,3;3849.89858000000,5.43750000000000,3;3931.03448300000,5.40000000000000,3;4022.31237300000,5.40000000000000,3;4181.20351600000,5.40000000000000,3;4438.13387400000,5.40000000000000,3;4948.61392800000,5.40000000000000,3;5448.95199500000,5.40000000000000,3;5780.25693000000,5.40000000000000,3;6016.90331300000,5.40000000000000,3;1000,6.50000000000000,2;1483.43475300000,6.02500000000000,2;1165.65246800000,6.32500000000000,2;1334.68559800000,6.15000000000000,2;1503.71872900000,6.06250000000000,2;1665.99053400000,6.21250000000000,2;1838.40432700000,6.40000000000000,2;2000.67613300000,6.55000000000000,2;2173.08992600000,6.71250000000000,2;2342.12305600000,6.87500000000000,2;2494.25287400000,7.02500000000000,2;2676.80865400000,6.65000000000000,2;2842.46112200000,6.32500000000000,2;3011.49425300000,6.02500000000000,2;3092.63015600000,5.95000000000000,2;3173.76605800000,5.91250000000000,2;3261.66328600000,5.88750000000000,2;3346.17985100000,5.81250000000000,2;3430.69641600000,5.78750000000000,2;3515.21298200000,5.76250000000000,2;3599.72954700000,5.72500000000000,2;3684.24611200000,5.70000000000000,2;3765.38201500000,5.66250000000000,2;3849.89858000000,5.66250000000000,2;3931.03448300000,5.62500000000000,2;4018.93171100000,5.58750000000000,2;4103.44827600000,5.60000000000000,2;4184.58417800000,5.60000000000000,2;4434.75321200000,5.60000000000000,2;4945.23326600000,5.60000000000000,2;5448.95199500000,5.60000000000000,2;5783.63759300000,5.60000000000000,2;6013.52265000000,5.60000000000000,2;6500,5.60000000000000,2;4250,5.60000000000000,2;4700,5.60000000000000,2;5200,5.60000000000000,2;5600,5.60000000000000,2;5900,5.60000000000000,2;6200,5.60000000000000,2;2425.04407500000,6.94995600000000,2;2588.83243900000,6.82349200000000,2;2510.35051500000,6.98789500000000,2;2547.88534800000,6.92466300000000,2;2465.99116600000,6.97524900000000,2;2425.04407500000,6.94995600000000,2;2592.24469700000,6.82349200000000,2;1003.38066300000,6.75000000000000,1;1412.44083800000,6.55000000000000,1;1473.29276500000,6.53750000000000,1;1165.65246800000,6.68750000000000,1;1334.68559800000,6.58750000000000,1;1250.16903300000,6.61250000000000,1;1669.37119700000,6.72500000000000,1;1838.40432700000,6.88750000000000,1;2004.05679500000,7.02500000000000,1;2169.70926300000,7.17500000000000,1;2338.74239400000,7.35000000000000,1;2501.01419900000,7.48750000000000,1;2676.80865400000,7.31250000000000,1;2842.46112200000,7.15000000000000,1;2930.35835000000,7.08750000000000,1;3011.49425300000,7,1;3180.52738300000,6.66250000000000,1;3342.79918900000,6.32500000000000,1;3511.83231900000,6.02500000000000,1;3680.86545000000,5.92500000000000,1;3849.89858000000,5.85000000000000,1;3931.03448300000,5.81250000000000,1;4018.93171100000,5.78750000000000,1;4106.82893800000,5.78750000000000,1;4431.37254900000,5.78750000000000,1;4948.61392800000,5.78750000000000,1;5448.95199500000,5.78750000000000,1;5790.39891800000,5.78750000000000,1;6013.52265000000,5.78750000000000,1;6500,5.78750000000000,1;2592.24469700000,7.38046300000000,1;2756.03306100000,7.25458900000000,1;3097.25882000000,6.81403000000000,1;3261.04718400000,6.48675700000000,1;3595.44842800000,5.98326000000000,1;3431.66006300000,6.15948400000000,1;2425.04407500000,7.40563800000000,1;1517.38355700000,6.52451900000000,1;1582.21645100000,6.60004400000000,1;2251.01893800000,7.25458900000000,1;4200,5.78750000000000,1;4300,5.78750000000000,1;4500,5.78750000000000,1;4600,5.78750000000000,1;4700,5.78750000000000,1;4800,5.78750000000000,1;5000,5.78750000000000,1;5100,5.78750000000000,1;5200,5.78750000000000,1;5300,5.78750000000000,1;5500,5.78750000000000,1;5600,5.78750000000000,1;5700,5.78750000000000,1;5900,5.78750000000000,1;6100,5.78750000000000,1;6200,5.78750000000000,1;6300,5.78750000000000,1;6400,5.78750000000000,1;1000,7.80000000000000,0;1000,8.92500000000000,0;993.238675000000,10.1750000000000,0;1524.00270500000,10.1750000000000,0;2227.18052700000,10.1875000000000,0;2920.21636200000,10.1750000000000,0;3535.49695700000,10.2000000000000,0;4275.86206900000,10.1875000000000,0;5060.17579400000,10.2125000000000,0;5682.21771500000,10.1875000000000,0;6013.52265000000,10.1875000000000,0;6013.52265000000,8.78750000000000,0;6010.14198800000,7.43750000000000,0;6016.90331300000,6.27500000000000,0;996.619337000000,7.13750000000000,0;1473.29276500000,9.08750000000000,0;2355.64570700000,9.06250000000000,0;3177.14672100000,9.03750000000000,0;3927.65382000000,9.03750000000000,0;4654.49628100000,9.02500000000000,0;5432.04868200000,8.98750000000000,0;2757.94455700000,8.40000000000000,0;1949.96619300000,8.16250000000000,0;1439.48613900000,7.80000000000000,0;800,7.20000000000000,0;800,9,0;800,11,0;6500,6,0;6500,9,0;6500,11,0;2670.72662100000,7.56927400000000,0;1169.33328300000,7.06577800000000,0;1343.35842000000,6.96507900000000,0;1476.43646600000,7.01542800000000,0;1664.11063300000,7.19165200000000,0;1838.13577000000,7.38046300000000,0;1995.09961900000,7.56927400000000,0;2158.88798300000,7.78326000000000,0;2332.91312000000,7.95948400000000,0;2493.28922700000,8.07277100000000,0;2684.37565200000,7.87137200000000,0;2837.92724300000,7.69514900000000,0;3011.95238000000,7.54410000000000,0;3182.56525900000,7.19165200000000,0;3353.17813900000,6.86437900000000,0;3510.14198800000,6.53710700000000,0;3680.75486700000,6.41123300000000,0;3847.95548900000,6.37347000000000,0;4018.56836800000,6.27277100000000,0;4233.54059600000,6.27277100000000,0;4424.62702100000,6.27277100000000,0;4683.95859800000,6.27277100000000,0;4936.46565900000,6.27277100000000,0;5243.56884200000,6.27277100000000,0;5475.60235800000,6.27277100000000,0;5687.16232900000,6.27277100000000,0;5898.72229900000,6.27277100000000,0;6000,6.27277100000000,0;6500,6.27277100000000,0;4018.56836800000,6.10000000000000,0;4233.54059600000,6.10000000000000,0;4424.62702100000,6.10000000000000,0;4683.95859800000,6.10000000000000,0;4936.46565900000,6.10000000000000,0;5243.56884200000,6.10000000000000,0;5475.60235800000,6.10000000000000,0;5687.16232900000,6.10000000000000,0;5898.72229900000,6.10000000000000,0;6000,6.10000000000000,0;6500,6.10000000000000,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddCamless, RmvCamless, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/Camless.png'));
    title('Ricardo Camless Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:1:24] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0 11]);
	%}
end


%% 8: Add/Remove technology: CylDeact
% map_data = [CylDeact50;CylDeact48;CylDeact46;CylDeact44;CylDeact42;...
% CylDeact40;CylDeact38;CylDeact36;CylDeact34;CylDeact32;CylDeact30;...
% CylDeact28;CylDeact26;CylDeact24;CylDeact22;CylDeact20;CylDeact18;...
% CylDeact16;CylDeact14;CylDeact12;CylDeact10;CylDeact8;CylDeact6;...
% CylDeact4;CylDeact2;CylDeact0];
if (AddCylDeact==1 || RmvCylDeact==1)
	%     './data/CylDeact_20150612_2.mat'
	map_data = [802.576891000000,0.335366000000000,50;860.764755000000,0.335366000000000,50;923.108894000000,0.335366000000000,50;1000,0.335366000000000,50;1035.32834600000,0.317073000000000,50;1070.65669200000,0.310976000000000,50;1103.90689900000,0.304878000000000,50;1157.93848700000,0.304878000000000,50;1218.20448900000,0.304878000000000,50;1280.54862800000,0.304878000000000,50;1334.58021600000,0.304878000000000,50;1378.22111400000,0.304878000000000,50;1399.00249400000,0.304878000000000,50;1446.79966700000,0.280488000000000,50;1523.69077300000,0.256098000000000,50;1577.72236100000,0.250000000000000,50;1627.59767200000,0.237805000000000,50;1654.61346600000,0.237805000000000,50;1654.61346600000,0.146341000000000,50;1654.61346600000,0.0609760000000000,50;1654.61346600000,0,50;802.576891000000,0.395000000000000,48;856.608479000000,0.395000000000000,48;921.030756000000,0.395000000000000,48;1000,0.395000000000000,48;1041.56276000000,0.371951000000000,48;1087.28179600000,0.359756000000000,48;1155.86034900000,0.347561000000000,48;1118.45386500000,0.359756000000000,48;1216.12635100000,0.347561000000000,48;1284.70490400000,0.347561000000000,48;1334.58021600000,0.347561000000000,48;1405.23690800000,0.347561000000000,48;1428.09642600000,0.323171000000000,48;1461.34663300000,0.323171000000000,48;1480.04987500000,0.317073000000000,48;1517.45635900000,0.310976000000000,48;1532.00332500000,0.304878000000000,48;1592.26932700000,0.304878000000000,48;1608.89443100000,0.298780000000000,48;1635.91022400000,0.292683000000000,48;1650.45719000000,0.280488000000000,48;1692.01995000000,0.280488000000000,48;1708.64505400000,0.274390000000000,48;1791.77057400000,0.274390000000000,48;1754.36409000000,0.268293000000000,48;1563.17539500000,0.298780000000000,48;1820.86450500000,0.262195000000000,48;1833.33333300000,0.250000000000000,48;1864.50540300000,0.250000000000000,48;1879.05236900000,0.237805000000000,48;1937.24023300000,0.237805000000000,48;1937.24023300000,0.164634000000000,48;1937.24023300000,0.0670730000000000,48;1937.24023300000,0,48;804.655029000000,0.439024000000000,46;856.608479000000,0.439024000000000,46;925.187032000000,0.439024000000000,46;1002.07813800000,0.439024000000000,46;1035.32834600000,0.420732000000000,46;1060.26600200000,0.402439000000000,46;1105.98503700000,0.402439000000000,46;1184.95428100000,0.402439000000000,46;1249.37655900000,0.402439000000000,46;1309.64256000000,0.402439000000000,46;1365.75228600000,0.402439000000000,46;1396.92435600000,0.396341000000000,46;1428.09642600000,0.378049000000000,46;1480.04987500000,0.371951000000000,46;1529.92518700000,0.365854000000000,46;1602.66001700000,0.353659000000000,46;1637.98836200000,0.341463000000000,46;1704.48877800000,0.329268000000000,46;1800.08312600000,0.317073000000000,46;1825.02078100000,0.304878000000000,46;1876.97423100000,0.292683000000000,46;1928.92768100000,0.286585000000000,46;2003.74064800000,0.274390000000000,46;2055.69409800000,0.262195000000000,46;2402.74314200000,0.256098000000000,46;2446.38404000000,0.243902000000000,46;2138.81961800000,0.256098000000000,46;2224.02327500000,0.256098000000000,46;2309.22693300000,0.256098000000000,46;2473.39983400000,0.237805000000000,46;2525.35328300000,0.237805000000000,46;2575.22859500000,0.237805000000000,46;2575.22859500000,0.134146000000000,46;2575.22859500000,0.0487800000000000,46;2575.22859500000,0,46;800.498753000000,0.493902000000000,44;991.687448000000,0.493902000000000,44;858.686617000000,0.493902000000000,44;925.187032000000,0.493902000000000,44;1025.32730700000,0.472660000000000,44;1048.96612600000,0.460408000000000,44;1396.27493800000,0.454282000000000,44;1101.69887800000,0.454282000000000,44;1159.88674100000,0.454282000000000,44;1219.89297600000,0.454282000000000,44;1278.08084000000,0.454282000000000,44;1330.81359100000,0.454282000000000,44;1365.36263500000,0.454282000000000,44;1427.18724000000,0.442031000000000,44;1467.19139700000,0.417528000000000,44;1503.55881100000,0.405276000000000,44;1539.92622600000,0.399150000000000,44;1609.02431400000,0.393025000000000,44;1654.48358300000,0.380773000000000,44;1703.57959300000,0.380773000000000,44;1798.13487100000,0.380773000000000,44;1829.04717400000,0.368521000000000,44;1878.14318400000,0.356270000000000,44;1932.69430600000,0.344018000000000,44;2003.61076500000,0.325641000000000,44;2058.16188700000,0.319515000000000,44;2079.98233600000,0.313389000000000,44;2132.71508700000,0.313389000000000,44;2198.17643400000,0.313389000000000,44;2263.63778100000,0.313389000000000,44;2341.82772200000,0.313389000000000,44;2403.65232800000,0.313389000000000,44;2452.74833700000,0.295012000000000,44;2470.93204500000,0.288886000000000,44;2601.85473800000,0.288886000000000,44;2558.21384000000,0.288886000000000,44;2518.20968400000,0.288886000000000,44;2636.40378200000,0.282760000000000,44;2650.95074800000,0.264383000000000,44;2705.50187000000,0.264383000000000,44;2678.22630900000,0.264383000000000,44;2725.50394800000,0.258257000000000,44;2780.05507100000,0.258257000000000,44;2805.51226100000,0.258257000000000,44;2805.51226100000,0.135741000000000,44;2805.51226100000,0.0622320000000000,44;2805.51226100000,0.000974000000000000,44;802.576891000000,0.572485000000000,42;852.452203000000,0.554065000000000,42;921.030756000000,0.554065000000000,42;993.765586000000,0.554065000000000,42;958.437240000000,0.554065000000000,42;1027.01579400000,0.535645000000000,42;1060.26600200000,0.529505000000000,42;1074.81296800000,0.523365000000000,42;1120.53200300000,0.517225000000000,42;1137.15710700000,0.511085000000000,42;1401.08063200000,0.511085000000000,42;1336.65835400000,0.511085000000000,42;1272.23607600000,0.511085000000000,42;1199.50124700000,0.511085000000000,42;1450.95594300000,0.492665000000000,42;1477.97173700000,0.486525000000000,42;1513.30008300000,0.468105000000000,42;1554.86284300000,0.455825000000000,42;1604.73815500000,0.449685000000000,42;1650.45719000000,0.437405000000000,42;1706.56691600000,0.431265000000000,42;1762.67664200000,0.431265000000000,42;1798.00498800000,0.431265000000000,42;1825.02078100000,0.412844000000000,42;1876.97423100000,0.400564000000000,42;1931.00581900000,0.394424000000000,42;2003.74064800000,0.382144000000000,42;2053.61596000000,0.376004000000000,42;2070.24106400000,0.363724000000000,42;2163.75727300000,0.363724000000000,42;2234.41396500000,0.363724000000000,42;2327.93017500000,0.363724000000000,42;2431.83707400000,0.363724000000000,42;2531.58769700000,0.363724000000000,42;2600.16625100000,0.357584000000000,42;2635.49459700000,0.345304000000000,42;2660.43225300000,0.333024000000000,42;2712.38570200000,0.314604000000000,42;2764.33915200000,0.314604000000000,42;2803.82377400000,0.320744000000000,42;2812.13632600000,0.247064000000000,42;2812.13632600000,0.118124000000000,42;2812.13632600000,0.00146400000000000,42;803.486076000000,0.644182000000000,40;836.216750000000,0.638056000000000,40;852.582086000000,0.625805000000000,40;898.041355000000,0.631930000000000,40;943.500623000000,0.625805000000000,40;994.415004000000,0.619679000000000,40;1012.59871200000,0.607427000000000,40;1027.14567700000,0.601301000000000,40;1052.60286800000,0.595176000000000,40;1078.06005800000,0.589050000000000,40;1118.06421400000,0.582924000000000,40;1145.33977600000,0.570672000000000,40;1190.79904400000,0.570672000000000,40;1243.53179600000,0.576798000000000,40;1296.26454700000,0.570672000000000,40;1345.36055700000,0.570672000000000,40;1392.63819600000,0.570672000000000,40;1421.73212800000,0.564547000000000,40;1476.28325000000,0.552295000000000,40;1543.56296800000,0.540043000000000,40;1603.56920200000,0.527792000000000,40;1679.94077300000,0.503289000000000,40;1729.03678300000,0.491037000000000,40;1801.77161300000,0.478786000000000,40;1821.77369100000,0.466534000000000,40;1872.68807100000,0.454282000000000,40;1907.23711600000,0.442031000000000,40;1959.96986700000,0.442031000000000,40;2001.79239400000,0.442031000000000,40;2029.06795500000,0.429779000000000,40;2056.34351600000,0.417528000000000,40;2121.80486300000,0.417528000000000,40;2198.17643400000,0.423653000000000,40;2281.82148800000,0.417528000000000,40;2354.55631800000,0.417528000000000,40;2394.56047400000,0.417528000000000,40;2412.74418100000,0.405276000000000,40;2478.20552800000,0.405276000000000,40;2545.48524500000,0.405276000000000,40;2600.03636700000,0.405276000000000,40;2614.58333300000,0.393025000000000,40;2710.95698300000,0.386899000000000,40;2650.95074800000,0.393025000000000,40;2767.32647500000,0.380773000000000,40;2805.51226100000,0.374647000000000,40;2814.60411500000,0.282760000000000,40;2821.87759800000,0.233754000000000,40;2821.87759800000,0.117364000000000,40;2821.87759800000,-0.00515200000000000,40;804.655029000000,0.719845000000000,38;829.592685000000,0.713705000000000,38;856.608479000000,0.707565000000000,38;873.233583000000,0.701425000000000,38;910.640067000000,0.701425000000000,38;941.812136000000,0.689145000000000,38;975.062344000000,0.689145000000000,38;1002.07813800000,0.676865000000000,38;1029.09393200000,0.664585000000000,38;1058.18786400000,0.658445000000000,38;1078.96924400000,0.652305000000000,38;1105.98503700000,0.646165000000000,38;1133.00083100000,0.640025000000000,38;1209.89193700000,0.640025000000000,38;1284.70490400000,0.646165000000000,38;1355.36159600000,0.646165000000000,38;1409.39318400000,0.640025000000000,38;1421.86201200000,0.627745000000000,38;1444.72153000000,0.627745000000000,38;1471.73732300000,0.621605000000000,38;1507.06566900000,0.609325000000000,38;1542.39401500000,0.597045000000000,38;1586.03491300000,0.597045000000000,38;1604.73815500000,0.590905000000000,38;1629.67581000000,0.584765000000000,38;1644.22277600000,0.572485000000000,38;1689.94181200000,0.572485000000000,38;1712.80133000000,0.560205000000000,38;1785.53616000000,0.566345000000000,38;1804.23940100000,0.554065000000000,38;1831.25519500000,0.541785000000000,38;1868.66167900000,0.523365000000000,38;1908.14630100000,0.511085000000000,38;1949.70906100000,0.504945000000000,38;2014.13133800000,0.492665000000000,38;2057.77223600000,0.486525000000000,38;2120.11637600000,0.486525000000000,38;2174.14796300000,0.486525000000000,38;2205.32003300000,0.480385000000000,38;2280.13300100000,0.480385000000000,38;2363.25852000000,0.480385000000000,38;2411.05569400000,0.468105000000000,38;2494.18121400000,0.468105000000000,38;2568.99418100000,0.468105000000000,38;2604.32252700000,0.449685000000000,38;2637.57273500000,0.443545000000000,38;2681.21363300000,0.437405000000000,38;2710.30756400000,0.431265000000000,38;2770.57356600000,0.431265000000000,38;2803.82377400000,0.431265000000000,38;2818.37074000000,0.333024000000000,38;2828.76143000000,0.234784000000000,38;2828.76143000000,0.111984000000000,38;2828.76143000000,-0.00467600000000000,38;803.486076000000,0.809578000000000,36;839.853491000000,0.803452000000000,36;881.676018000000,0.791201000000000,36;923.498545000000,0.778949000000000,36;963.502702000000,0.766698000000000,36;988.959892000000,0.760572000000000,36;1012.59871200000,0.748320000000000,36;1050.78449700000,0.736069000000000,36;1087.15191200000,0.729943000000000,36;1114.42747300000,0.729943000000000,36;1150.79488800000,0.723817000000000,36;1212.61949300000,0.723817000000000,36;1267.17061500000,0.717691000000000,36;1328.99522000000,0.717691000000000,36;1385.36471300000,0.717691000000000,36;1412.64027400000,0.717691000000000,36;1439.91583500000,0.705440000000000,36;1476.28325000000,0.693188000000000,36;1519.92414800000,0.693188000000000,36;1558.10993300000,0.680937000000000,36;1581.74875300000,0.680937000000000,36;1614.47942600000,0.662559000000000,36;1654.48358300000,0.656433000000000,36;1703.57959300000,0.644182000000000,36;1738.12863700000,0.631930000000000,36;1787.22464700000,0.631930000000000,36;1810.86346600000,0.619679000000000,36;1839.95739800000,0.607427000000000,36;1887.23503700000,0.595176000000000,36;1941.78616000000,0.582924000000000,36;1989.06379900000,0.582924000000000,36;2010.88424800000,0.570672000000000,36;2038.15980900000,0.570672000000000,36;2069.07211100000,0.564547000000000,36;2134.53345800000,0.564547000000000,36;2210.90502900000,0.564547000000000,36;2280.00311700000,0.564547000000000,36;2343.64609300000,0.558421000000000,36;2394.56047400000,0.558421000000000,36;2416.38092300000,0.546169000000000,36;2443.65648400000,0.546169000000000,36;2469.11367400000,0.533918000000000,36;2527.30153800000,0.533918000000000,36;2587.30777200000,0.533918000000000,36;2618.22007500000,0.527792000000000,36;2650.95074800000,0.515540000000000,36;2683.68142100000,0.515540000000000,36;2721.86720700000,0.509415000000000,36;2763.68973400000,0.509415000000000,36;2803.69389000000,0.509415000000000,36;2816.42248500000,0.405276000000000,36;2841.87967600000,0.233754000000000,36;2843.69804700000,0.325641000000000,36;2843.69804700000,0.123490000000000,36;2843.69804700000,0.000974000000000000,36;801.667706000000,0.913717000000000,34;818.033042000000,0.901465000000000,34;845.308603000000,0.889214000000000,34;872.584165000000,0.876962000000000,34;896.222984000000,0.864710000000000,34;918.043433000000,0.852459000000000,34;943.500623000000,0.846333000000000,34;974.412926000000,0.834081000000000,34;1008.96197000000,0.821830000000000,34;1041.69264300000,0.809578000000000,34;1090.78865300000,0.809578000000000,34;1118.06421400000,0.803452000000000,34;1178.07044900000,0.797327000000000,34;1243.53179600000,0.803452000000000,34;1305.35640100000,0.803452000000000,34;1352.63404000000,0.803452000000000,34;1418.09538700000,0.797327000000000,34;1434.46072300000,0.791201000000000,34;1472.64650900000,0.791201000000000,34;1489.01184500000,0.778949000000000,34;1523.56088900000,0.778949000000000,34;1541.74459700000,0.760572000000000,34;1592.65897800000,0.760572000000000,34;1605.38757300000,0.754446000000000,34;1639.93661700000,0.742195000000000,34;1670.84891900000,0.729943000000000,34;1709.03470500000,0.717691000000000,34;1752.67560300000,0.711566000000000,34;1785.40627600000,0.711566000000000,34;1810.86346600000,0.693188000000000,34;1847.23088100000,0.693188000000000,34;1872.68807100000,0.687062000000000,34;1887.23503700000,0.674811000000000,34;1918.14734000000,0.674811000000000,34;1936.33104700000,0.668685000000000,34;1996.33728200000,0.668685000000000,34;1965.42497900000,0.668685000000000,34;2010.88424800000,0.656433000000000,34;2047.25166300000,0.656433000000000,34;2063.61699900000,0.644182000000000,34;2169.08250200000,0.644182000000000,34;2310.91542000000,0.644182000000000,34;2398.19721500000,0.644182000000000,34;2241.81733200000,0.644182000000000,34;2118.16812100000,0.644182000000000,34;2414.56255200000,0.631930000000000,34;2481.84226900000,0.631930000000000,34;2552.75872800000,0.631930000000000,34;2589.12614300000,0.631930000000000,34;2603.67310900000,0.619679000000000,34;2630.94867000000,0.619679000000000,34;2649.13237700000,0.607427000000000,34;2692.77327500000,0.607427000000000,34;2716.41209500000,0.601301000000000,34;2756.41625100000,0.601301000000000,34;2803.69389000000,0.601301000000000,34;2818.24085600000,0.460408000000000,34;2836.42456400000,0.325641000000000,34;2850.97153000000,0.239880000000000,34;2850.97153000000,0.105112000000000,34;2850.97153000000,-0.00515200000000000,34;803.486076000000,1.00560400000000,32;841.671862000000,0.993352000000000,32;888.949501000000,0.962723000000000,32;950.774106000000,0.919842000000000,32;1025.32730700000,0.901465000000000,32;1090.78865300000,0.889214000000000,32;1143.52140500000,0.876962000000000,32;1250.80527800000,0.876962000000000,32;1407.18516200000,0.876962000000000,32;1328.99522000000,0.876962000000000,32;1476.28325000000,0.852459000000000,32;1512.65066500000,0.852459000000000,32;1572.65689900000,0.840207000000000,32;1616.29779700000,0.834081000000000,32;1650.84684100000,0.815704000000000,32;1694.48773900000,0.809578000000000,32;1752.67560300000,0.797327000000000,32;1790.86138800000,0.791201000000000,32;1821.77369100000,0.791201000000000,32;1863.59621800000,0.785075000000000,32;1905.41874500000,0.778949000000000,32;1939.96778900000,0.766698000000000,32;1992.70054000000,0.760572000000000,32;2030.88632600000,0.748320000000000,32;2089.07419000000,0.742195000000000,32;2134.53345800000,0.736069000000000,32;2216.36014100000,0.729943000000000,32;2298.18682500000,0.729943000000000,32;2387.28699100000,0.729943000000000,32;2429.10951800000,0.729943000000000,32;2450.92996700000,0.729943000000000,32;2476.38715700000,0.723817000000000,32;2583.67103100000,0.723817000000000,32;2521.84642600000,0.723817000000000,32;2610.94659200000,0.705440000000000,32;2636.40378200000,0.711566000000000,32;2658.22423100000,0.693188000000000,32;2696.41001700000,0.699314000000000,32;2729.14069000000,0.693188000000000,32;2801.87552000000,0.687062000000000,32;2767.32647500000,0.687062000000000,32;2818.24085600000,0.558421000000000,32;2832.78782200000,0.411402000000000,32;2858.24501200000,0.233754000000000,32;2858.24501200000,0.123490000000000,32;2858.24501200000,-0.00515200000000000,32;2845.51641700000,0.325641000000000,32;799.849335000000,1.12811900000000,30;852.582086000000,1.09136500000000,30;907.133209000000,1.06686100000000,30;950.774106000000,1.04848400000000,30;996.233375000000,1.03010700000000,30;1036.23753100000,1.01172900000000,30;1074.42331700000,1.00560400000000,30;1112.60910200000,0.987226000000000,30;1154.43162900000,0.968849000000000,30;1208.98275100000,0.962723000000000,30;1256.26039100000,0.962723000000000,30;1303.53803000000,0.968849000000000,30;1339.90544500000,0.968849000000000,30;1368.99937700000,0.987226000000000,30;1412.64027400000,0.987226000000000,30;1443.55257700000,0.981100000000000,30;1481.73836200000,0.962723000000000,30;1521.74251900000,0.950471000000000,30;1556.29156300000,0.932094000000000,30;1596.29571900000,0.932094000000000,30;1627.20802200000,0.925968000000000,30;1665.39380700000,0.919842000000000,30;1707.21633400000,0.907591000000000,30;1754.49397300000,0.883088000000000,30;1830.86554400000,0.889214000000000,30;1912.69222800000,0.889214000000000,30;1999.97402300000,0.889214000000000,30;2032.70469700000,0.883088000000000,30;2065.43537000000,0.870836000000000,30;2090.89256000000,0.870836000000000,30;2116.34975100000,0.858585000000000,30;2170.90087300000,0.858585000000000,30;2230.90710700000,0.858585000000000,30;2278.18474600000,0.858585000000000,30;2334.55423900000,0.858585000000000,30;2354.55631800000,0.870836000000000,30;2385.46862000000,0.870836000000000,30;2414.56255200000,0.883088000000000,30;2465.47693300000,0.883088000000000,30;2521.84642600000,0.883088000000000,30;2589.12614300000,0.883088000000000,30;2607.30985000000,0.870836000000000,30;2629.13029900000,0.870836000000000,30;2650.95074800000,0.858585000000000,30;2683.68142100000,0.858585000000000,30;2714.59372400000,0.852459000000000,30;2754.59788000000,0.852459000000000,30;2792.78366600000,0.852459000000000,30;2803.69389000000,0.852459000000000,30;2810.96737300000,0.766698000000000,30;2812.78574400000,0.705440000000000,30;2821.87759800000,0.613553000000000,30;2832.78782200000,0.521666000000000,30;2845.51641700000,0.411402000000000,30;2854.60827100000,0.331767000000000,30;2867.33686600000,0.239880000000000,30;2867.33686600000,0.111238000000000,30;2867.33686600000,-0.00515200000000000,30;801.667706000000,1.25676100000000,28;845.308603000000,1.24450900000000,28;901.678096000000,1.22000600000000,28;950.774106000000,1.20775500000000,28;987.141521000000,1.18937700000000,28;1036.23753100000,1.16487400000000,28;1083.51517000000,1.14649700000000,28;1123.51932700000,1.12811900000000,28;1158.06837100000,1.10361600000000,28;1190.79904400000,1.09749000000000,28;1236.25831300000,1.09136500000000,28;1283.53595200000,1.09749000000000,28;1323.54010800000,1.09136500000000,28;1356.27078100000,1.11586800000000,28;1392.63819600000,1.10974200000000,28;1436.27909400000,1.09749000000000,28;1479.91999200000,1.07911300000000,28;1516.28740600000,1.07298700000000,28;1549.01808000000,1.06686100000000,28;1576.29364100000,1.06073600000000,28;1621.75290900000,1.04235800000000,28;1663.57543600000,1.03010700000000,28;1719.94492900000,1.03010700000000,28;1783.58790500000,1.00560400000000,28;1852.68599300000,1.01172900000000,28;1925.42082300000,1.02398100000000,28;1983.60868700000,1.03010700000000,28;2016.33936000000,1.03623200000000,28;2056.34351600000,1.02398100000000,28;2103.62115500000,1.01172900000000,28;2150.89879500000,0.999478000000000,28;2218.17851200000,0.999478000000000,28;2281.82148800000,1.01172900000000,28;2323.64401500000,1.02398100000000,28;2356.37468800000,1.03623200000000,28;2400.01558600000,1.04848400000000,28;2461.84019100000,1.04235800000000,28;2532.75665000000,1.04235800000000,28;2581.85266000000,1.04235800000000,28;2630.94867000000,1.03010700000000,28;2694.59164600000,1.03010700000000,28;2758.23462200000,1.03623200000000,28;2798.23877800000,1.03623200000000,28;2810.96737300000,0.907591000000000,28;2825.51433900000,0.723817000000000,28;2841.87967600000,0.533918000000000,28;2858.24501200000,0.405276000000000,28;2876.42872000000,0.239880000000000,28;2876.42872000000,0.117364000000000,28;2874.61034900000,0.000974000000000000,28;803.486076000000,1.44666000000000,26;854.400457000000,1.40378000000000,26;910.769950000000,1.36089900000000,26;965.321072000000,1.31801900000000,26;1027.14567700000,1.28739000000000,26;1085.33354100000,1.26288700000000,26;1141.70303400000,1.23225800000000,26;1196.25415600000,1.22000600000000,26;1239.89505400000,1.22000600000000,26;1285.35432300000,1.22000600000000,26;1318.08499600000,1.23225800000000,26;1370.81774700000,1.23838400000000,26;1409.00353300000,1.24450900000000,26;1449.00768900000,1.22613200000000,26;1496.28532800000,1.22000600000000,26;1550.83645100000,1.20775500000000,26;1592.65897800000,1.20775500000000,26;1621.75290900000,1.18937700000000,26;1649.02847000000,1.17712600000000,26;1679.94077300000,1.17712600000000,26;1703.57959300000,1.17100000000000,26;1758.13071500000,1.17100000000000,26;1794.49813000000,1.15874800000000,26;1827.22880300000,1.17100000000000,26;1876.32481300000,1.17712600000000,26;1912.69222800000,1.18937700000000,26;1949.05964300000,1.20162900000000,26;1989.06379900000,1.22000600000000,26;2003.61076500000,1.23225800000000,26;2039.97818000000,1.22000600000000,26;2085.43744800000,1.20775500000000,26;2119.98649200000,1.19550300000000,26;2169.08250200000,1.18325100000000,26;2207.26828800000,1.17100000000000,26;2261.81941000000,1.18937700000000,26;2312.73379100000,1.21388000000000,26;2358.19305900000,1.23225800000000,26;2401.83395700000,1.25063500000000,26;2481.84226900000,1.24450900000000,26;2569.12406500000,1.24450900000000,26;2658.22423100000,1.25063500000000,26;2703.68350000000,1.26288700000000,26;2747.32439700000,1.26288700000000,26;2774.59995800000,1.27513800000000,26;2805.51226100000,1.27513800000000,26;2810.96737300000,1.11586800000000,26;2818.24085600000,0.962723000000000,26;2830.96945100000,0.834081000000000,26;2838.24293400000,0.693188000000000,26;2858.24501200000,0.497163000000000,26;2874.61034900000,0.344018000000000,26;2883.70220300000,0.239880000000000,26;2883.70220300000,0.0928610000000000,26;2883.70220300000,0.000974000000000000,26;803.486076000000,1.59980500000000,24;854.400457000000,1.56305000000000,24;923.498545000000,1.50179300000000,24;996.233375000000,1.44053500000000,24;1076.24168700000,1.40378000000000,24;1150.79488800000,1.35477400000000,24;1238.07668300000,1.33639600000000,24;1321.72173700000,1.34864800000000,24;1396.27493800000,1.37315100000000,24;1503.55881100000,1.37927700000000,24;1598.11409000000,1.37927700000000,24;1672.66729000000,1.35477400000000,24;1743.58374900000,1.34252200000000,24;1821.77369100000,1.34252200000000,24;1879.96155400000,1.37315100000000,24;1945.42290100000,1.43440900000000,24;2003.61076500000,1.48954100000000,24;2047.25166300000,1.47116400000000,24;2107.25789700000,1.46503800000000,24;2176.35598500000,1.46503800000000,24;2260.00103900000,1.45891200000000,24;2314.55216100000,1.47728900000000,24;2390.92373200000,1.50791800000000,24;2454.56670800000,1.50791800000000,24;2520.02805500000,1.50791800000000,24;2580.03428900000,1.50791800000000,24;2630.94867000000,1.50791800000000,24;2700.04675800000,1.53242200000000,24;2750.96113900000,1.55692500000000,24;2801.87552000000,1.58142800000000,24;2810.96737300000,1.36089900000000,24;2821.87759800000,1.18325100000000,24;2834.60619300000,0.938220000000000,24;2850.97153000000,0.717691000000000,24;2870.97360800000,0.466534000000000,24;2892.79405700000,0.258257000000000,24;2892.79405700000,0.123490000000000,24;2892.79405700000,-0.00515200000000000,24;2880.06546100000,0.356270000000000,24;801.667706000000,1.74069800000000,22;847.126974000000,1.71619500000000,22;910.769950000000,1.66106300000000,22;983.504780000000,1.61818300000000,22;1061.69472200000,1.58755400000000,22;1118.06421400000,1.53854700000000,22;1185.34393200000,1.51404400000000,22;1250.80527800000,1.51404400000000,22;1316.26662500000,1.53854700000000,22;1396.27493800000,1.56917600000000,22;1450.82606000000,1.55692500000000,22;1498.10369900000,1.54467300000000,22;1574.47527000000,1.55079900000000,22;1672.66729000000,1.55079900000000,22;1781.76953400000,1.55079900000000,22;1861.77784700000,1.61818300000000,22;1916.32896900000,1.67331500000000,22;1958.15149600000,1.71619500000000,22;2005.42913500000,1.76520200000000,22;2052.70677500000,1.76520200000000,22;2092.71093100000,1.77132700000000,22;2134.53345800000,1.78970500000000,22;2169.08250200000,1.78970500000000,22;2214.54177100000,1.80195600000000,22;2272.72963400000,1.82033400000000,22;2330.91749800000,1.83871100000000,22;2383.65024900000,1.85096300000000,22;2421.83603500000,1.84483700000000,22;2472.75041600000,1.83258500000000,22;2527.30153800000,1.82645900000000,22;2576.39754800000,1.83258500000000,22;2643.67726500000,1.82033400000000,22;2694.59164600000,1.82645900000000,22;2725.50394800000,1.85096300000000,22;2765.50810500000,1.86321400000000,22;2800.05714900000,1.88159200000000,22;2812.78574400000,1.69781800000000,22;2820.05922700000,1.57530200000000,22;2830.96945100000,1.39765400000000,22;2834.60619300000,1.23838400000000,22;2845.51641700000,1.10361600000000,22;2849.15315900000,0.944346000000000,22;2860.06338300000,0.748320000000000,22;2870.97360800000,0.582924000000000,22;2878.24709100000,0.466534000000000,22;2890.97568600000,0.362396000000000,22;2903.70428100000,0.246006000000000,22;2903.70428100000,0.0806090000000000,22;2903.70428100000,-0.00515200000000000,22;800,1.75000000000000,22;803.486076000000,1.93059800000000,20;876.220906000000,1.94284900000000,20;968.957814000000,1.96735300000000,20;1050.78449700000,1.89384300000000,20;1112.60910200000,1.82645900000000,20;1199.89089800000,1.75295000000000,20;1152.61325900000,1.78970500000000,20;925.316916000000,1.95510100000000,20;1243.53179600000,1.76520200000000,20;1387.18308400000,1.87546600000000,20;1283.53595200000,1.77745300000000,20;1332.63196200000,1.82645900000000,20;1539.92622600000,1.80808200000000,20;1438.09746500000,1.84483700000000,20;1485.37510400000,1.82033400000000,20;1623.57128000000,1.80195600000000,20;1579.93038200000,1.80195600000000,20;1676.30403200000,1.83871100000000,20;1736.31026600000,1.88771700000000,20;1799.95324200000,1.92447200000000,20;1854.50436400000,1.97960400000000,20;1907.23711600000,2.04086200000000,20;1967.24335000000,2.09599400000000,20;2005.42913500000,2.13887500000000,20;2059.98025800000,2.13274900000000,20;2114.53138000000,2.12662300000000,20;2181.81109700000,2.15725200000000,20;2267.27452200000,2.15725200000000,20;2340.00935200000,2.15112600000000,20;2385.46862000000,2.16337800000000,20;2438.20137200000,2.13887500000000,20;2496.38923500000,2.10824600000000,20;2561.85058200000,2.10212000000000,20;2625.49355800000,2.10824600000000,20;2678.22630900000,2.10824600000000,20;2729.14069000000,2.12662300000000,20;2803.69389000000,2.17563000000000,20;2816.42248500000,1.96735300000000,20;2827.33271000000,1.77132700000000,20;2834.60619300000,1.56917600000000,20;2845.51641700000,1.31801900000000,20;2860.06338300000,1.04235800000000,20;2876.42872000000,0.821830000000000,20;2887.33894400000,0.552295000000000,20;2905.52265200000,0.276634000000000,20;2905.52265200000,0.172496000000000,20;2905.52265200000,-0.00515200000000000,20;2896.43079800000,0.429779000000000,20;801.667706000000,2.18175500000000,18;856.218828000000,2.18788100000000,18;903.496467000000,2.20013300000000,18;954.410848000000,2.20013300000000,18;999.870116000000,2.21238400000000,18;1038.05590200000,2.20013300000000,18;1098.06213600000,2.16337800000000,18;1167.16022400000,2.13887500000000,18;1203.52763900000,2.12662300000000,18;1243.53179600000,2.13887500000000,18;1290.80943500000,2.16337800000000,18;1341.72381500000,2.18175500000000,18;1396.27493800000,2.20625800000000,18;1423.55049900000,2.20625800000000,18;1459.91791400000,2.20625800000000,18;1512.65066500000,2.19400700000000,18;1558.10993300000,2.18788100000000,18;1607.20594300000,2.18788100000000,18;1659.93869500000,2.21851000000000,18;1725.40004200000,2.26751600000000,18;1805.40835400000,2.32877400000000,18;1854.50436400000,2.34102600000000,18;1927.23919400000,2.35940300000000,18;1976.33520400000,2.37165500000000,18;2029.06795500000,2.40841000000000,18;2094.52930200000,2.44516400000000,18;2136.35182900000,2.46966700000000,18;2181.81109700000,2.50029600000000,18;2241.81733200000,2.50642200000000,18;2267.27452200000,2.50029600000000,18;2332.73586900000,2.50029600000000,18;2403.65232800000,2.50029600000000,18;2449.11159600000,2.48191900000000,18;2492.75249400000,2.46966700000000,18;2538.21176200000,2.45741600000000,18;2587.30777200000,2.44516400000000,18;2658.22423100000,2.44516400000000,18;2738.23254400000,2.44516400000000,18;2761.87136300000,2.45741600000000,18;2801.87552000000,2.46354200000000,18;2816.42248500000,2.29814500000000,18;2825.51433900000,2.12662300000000,18;2834.60619300000,1.96735300000000,18;2845.51641700000,1.78357900000000,18;2852.78990000000,1.59367900000000,18;2861.88175400000,1.42215700000000,18;2870.97360800000,1.21388000000000,18;2872.79197800000,1.04848400000000,18;2880.06546100000,0.895339000000000,18;2887.33894400000,0.803452000000000,18;2900.06753900000,0.570672000000000,18;2905.52265200000,0.423653000000000,18;2918.25124700000,0.307263000000000,18;2923.70635900000,0.246006000000000,18;2923.70635900000,0.0989870000000000,18;2923.70635900000,-0.0112780000000000,18;801.667706000000,2.43903900000000,16;863.492311000000,2.42066100000000,16;930.772028000000,2.40841000000000,16;978.049667000000,2.39003200000000,16;1036.23753100000,2.37165500000000,16;1092.60702400000,2.37165500000000,16;1127.15606800000,2.35940300000000,16;1174.43370700000,2.35940300000000,16;1219.89297600000,2.35940300000000,16;1276.26246900000,2.40841000000000,16;1348.99729800000,2.45741600000000,16;1398.09330800000,2.50029600000000,16;1490.83021600000,2.49417100000000,16;1561.74667500000,2.49417100000000,16;1621.75290900000,2.49417100000000,16;1672.66729000000,2.53092500000000,16;1730.85515400000,2.56768000000000,16;1783.58790500000,2.60443500000000,16;1839.95739800000,2.62281200000000,16;1881.77992500000,2.64119000000000,16;1939.96778900000,2.65956700000000,16;2014.52098900000,2.69632200000000,16;2078.16396500000,2.75145400000000,16;2127.25997500000,2.79433400000000,16;2170.90087300000,2.81883800000000,16;2199.99480500000,2.84946600000000,16;2276.36637600000,2.84946600000000,16;2343.64609300000,2.84946600000000,16;2403.65232800000,2.85559200000000,16;2445.47485500000,2.82496300000000,16;2516.39131300000,2.81271200000000,16;2556.39547000000,2.80046000000000,16;2600.03636700000,2.78820900000000,16;2638.22215300000,2.80046000000000,16;2687.31816300000,2.81271200000000,16;2729.14069000000,2.81883800000000,16;2758.23462200000,2.82496300000000,16;2785.51018300000,2.83721500000000,16;2801.87552000000,2.84946600000000,16;2810.96737300000,2.69632200000000,16;2823.69596800000,2.49417100000000,16;2834.60619300000,2.31039700000000,16;2845.51641700000,2.08374300000000,16;2854.60827100000,1.91222100000000,16;2863.70012500000,1.74069800000000,16;2869.15523700000,1.55692500000000,16;2874.61034900000,1.34252200000000,16;2885.52057400000,1.10361600000000,16;2892.79405700000,0.895339000000000,16;2905.52265200000,0.693188000000000,16;2914.61450500000,0.497163000000000,16;2925.52473000000,0.313389000000000,16;2929.16147100000,0.258257000000000,16;2929.16147100000,0.123490000000000,16;2929.16147100000,-0.00515200000000000,16;800,2.40000000000000,16;803.486076000000,2.61668600000000,14;850.763716000000,2.60443500000000,14;919.861804000000,2.59218300000000,14;976.231297000000,2.59218300000000,14;1028.96404800000,2.57993200000000,14;1087.15191200000,2.56768000000000,14;1145.33977600000,2.56768000000000,14;1203.52763900000,2.56768000000000,14;1261.71550300000,2.62281200000000,14;1334.45033300000,2.70244700000000,14;1401.73005000000,2.77595700000000,14;1501.74044100000,2.77595700000000,14;1601.75083100000,2.78208300000000,14;1656.30195300000,2.80658600000000,14;1732.67352500000,2.84946600000000,14;1818.13694900000,2.89234700000000,14;1878.14318400000,2.92910200000000,14;1918.14734000000,2.95973100000000,14;1979.97194500000,2.99036000000000,14;2007.24750600000,3.03324000000000,14;2047.25166300000,3.03936600000000,14;2096.34767200000,3.07612100000000,14;2150.89879500000,3.11287500000000,14;2183.62946800000,3.11900100000000,14;2234.54384900000,3.16800800000000,14;2281.82148800000,3.20476200000000,14;2338.19098100000,3.25989400000000,14;2416.38092300000,3.25989400000000,14;2476.38715700000,3.25989400000000,14;2558.21384000000,3.25989400000000,14;2607.30985000000,3.23539100000000,14;2674.58956800000,3.20476200000000,14;2720.04883600000,3.18638500000000,14;2761.87136300000,3.17413300000000,14;2803.69389000000,3.16188200000000,14;2823.69596800000,2.92297600000000,14;2830.96945100000,2.78820900000000,14;2840.06130500000,2.60443500000000,14;2845.51641700000,2.43291300000000,14;2858.24501200000,2.26139100000000,14;2867.33686600000,2.07149100000000,14;2874.61034900000,1.92447200000000,14;2881.88383200000,1.78357900000000,14;2887.33894400000,1.65493700000000,14;2890.97568600000,1.48954100000000,14;2894.61242700000,1.33027000000000,14;2900.06753900000,1.18325100000000,14;2903.70428100000,1.04235800000000,14;2905.52265200000,0.876962000000000,14;2916.43287600000,0.729943000000000,14;2920.06961800000,0.540043000000000,14;2927.34310100000,0.417528000000000,14;2932.79821300000,0.344018000000000,14;2938.25332500000,0.258257000000000,14;2938.25332500000,0.117364000000000,14;2938.25332500000,-0.00515200000000000,14;803.486076000000,2.76370500000000,12;854.400457000000,2.77595700000000,12;918.043433000000,2.81271200000000,12;963.502702000000,2.83108900000000,12;999.870116000000,2.86171800000000,12;1043.51101400000,2.84334100000000,12;1098.06213600000,2.81883800000000,12;1161.70511200000,2.80046000000000,12;1196.25415600000,2.78820900000000,12;1250.80527800000,2.83721500000000,12;1303.53803000000,2.89847300000000,12;1352.63404000000,2.95360500000000,12;1368.99937700000,2.95973100000000,12;1399.91167900000,3.00261100000000,12;1423.55049900000,2.99648500000000,12;1443.55257700000,2.99036000000000,12;1469.00976700000,2.97810800000000,12;1532.65274300000,2.97810800000000,12;1616.29779700000,2.97810800000000,12;1669.03054900000,3.02711400000000,12;1723.58167100000,3.05774300000000,12;1772.67768100000,3.08837200000000,12;1821.77369100000,3.11287500000000,12;1859.95947600000,3.14963000000000,12;1898.14526200000,3.18638500000000,12;1932.69430600000,3.21701400000000,12;1979.97194500000,3.25989400000000,12;2038.15980900000,3.25989400000000,12;2112.71300900000,3.25989400000000,12;2192.72132200000,3.25989400000000,12;2229.08873600000,3.27214600000000,12;2294.55008300000,3.27214600000000,12;2350.91957600000,3.29052300000000,12;2436.38300100000,3.28439800000000,12;2518.20968400000,3.28439800000000,12;2605.49148000000,3.28439800000000,12;2687.31816300000,3.28439800000000,12;2749.14276800000,3.28439800000000,12;2803.69389000000,3.28439800000000,12;2823.69596800000,3.26602000000000,12;2832.78782200000,3.13737900000000,12;2849.15315900000,2.91685000000000,12;2860.06338300000,2.66569300000000,12;2867.33686600000,2.50642200000000,12;2872.79197800000,2.37778100000000,12;2881.88383200000,2.20625800000000,12;2889.15731500000,2.01635900000000,12;2892.79405700000,1.83871100000000,12;2898.24916900000,1.66718900000000,12;2903.70428100000,1.49566700000000,12;2909.15939300000,1.28739000000000,12;2916.43287600000,1.09749000000000,12;2921.88798800000,0.870836000000000,12;2927.34310100000,0.656433000000000,12;2934.61658400000,0.515540000000000,12;2941.89006700000,0.405276000000000,12;2947.34517900000,0.282760000000000,12;2950.98192000000,0.239880000000000,12;2950.98192000000,0.0928610000000000,12;2950.98192000000,-0.0112780000000000,12;2649.13237700000,3.28439800000000,12;800,2.75000000000000,12;801.667706000000,2.91072400000000,10;850.763716000000,2.94135300000000,10;894.404613000000,2.97198200000000,10;943.500623000000,3.01486300000000,10;1001.68848700000,3.06999500000000,10;1054.42123900000,3.03936600000000,10;1116.24584400000,3.00261100000000,10;1168.97859500000,2.97198200000000,10;1199.89089800000,2.95360500000000,10;1227.16645900000,2.99036000000000,10;1250.80527800000,3.02711400000000,10;1308.99314200000,3.08224600000000,10;1358.08915200000,3.13125300000000,10;1401.73005000000,3.17413300000000,10;1429.00561100000,3.16800800000000,10;1476.28325000000,3.15575600000000,10;1523.56088900000,3.14963000000000,10;1625.38965100000,3.15575600000000,10;1576.29364100000,3.15575600000000,10;1672.66729000000,3.18638500000000,10;1712.67144600000,3.21088800000000,10;1778.13279300000,3.25989400000000,10;1839.95739800000,3.26602000000000,10;1907.23711600000,3.27214600000000,10;1976.33520400000,3.28439800000000,10;2016.33936000000,3.29664900000000,10;2087.25581900000,3.30277500000000,10;2161.80901900000,3.30277500000000,10;2219.99688300000,3.30277500000000,10;2289.09497100000,3.30277500000000,10;2370.92165400000,3.32115200000000,10;2440.01974200000,3.32115200000000,10;2507.29946000000,3.32115200000000,10;2581.85266000000,3.32115200000000,10;2641.85889400000,3.32115200000000,10;2720.04883600000,3.32115200000000,10;2783.69181200000,3.32115200000000,10;2803.69389000000,3.32115200000000,10;2830.96945100000,3.27827200000000,10;2850.97153000000,3.26602000000000,10;2865.51849500000,3.07612100000000,10;2876.42872000000,2.86784400000000,10;2887.33894400000,2.60443500000000,10;2894.61242700000,2.31652300000000,10;2907.34102200000,2.01023300000000,10;2910.97776400000,1.83258500000000,10;2918.25124700000,1.63656000000000,10;2925.52473000000,1.39765400000000,10;2930.97984200000,1.14649700000000,10;2936.43495400000,0.870836000000000,10;2938.25332500000,0.674811000000000,10;2947.34517900000,0.435905000000000,10;2958.25540300000,0.264383000000000,10;2958.25540300000,0.117364000000000,10;2958.25540300000,0.00710000000000000,10;805.304447000000,3.05161800000000,8;858.037199000000,3.11287500000000,8;905.314838000000,3.16188200000000,8;943.500623000000,3.21088800000000,8;983.504780000000,3.23000000000000,8;1021.69056500000,3.23602000000000,8;1068.96820400000,3.22314000000000,8;1127.15606800000,3.17413300000000,8;1198.07252700000,3.13125300000000,8;1238.07668300000,3.16188200000000,8;1281.71758100000,3.21088800000000,8;1328.99522000000,3.26602000000000,8;1381.72797200000,3.27214600000000,8;1416.27701600000,3.27827200000000,8;1479.91999200000,3.28439800000000,8;1549.01808000000,3.28439800000000,8;1612.66105600000,3.28439800000000,8;1687.21425600000,3.28439800000000,8;1721.76330000000,3.29052300000000,8;1778.13279300000,3.29664900000000,8;1818.13694900000,3.30890100000000,8;1879.96155400000,3.30890100000000,8;1910.87385700000,3.31502700000000,8;1954.51475500000,3.32115200000000,8;1981.79031600000,3.33340400000000,8;2027.24958400000,3.34565500000000,8;2099.98441400000,3.34565500000000,8;2176.35598500000,3.35178100000000,8;2214.54177100000,3.35790700000000,8;2280.00311700000,3.35790700000000,8;2372.74002500000,3.35790700000000,8;2489.11575200000,3.35790700000000,8;2627.31192900000,3.35790700000000,8;2720.04883600000,3.35790700000000,8;2803.69389000000,3.35790700000000,8;2556.39547000000,3.35790700000000,8;2827.33271000000,3.32727800000000,8;2872.79197800000,3.26602000000000,8;2881.88383200000,3.26602000000000,8;2889.15731500000,3.11287500000000,8;2898.24916900000,2.92297600000000,8;2903.70428100000,2.73920200000000,8;2912.79613500000,2.57993200000000,8;2916.43287600000,2.39003200000000,8;2920.06961800000,2.19400700000000,8;2927.34310100000,1.98573000000000,8;2930.97984200000,1.82645900000000,8;2930.97984200000,1.59367900000000,8;2936.43495400000,1.42215700000000,8;2943.70843700000,1.21388000000000,8;2945.52680800000,1.02398100000000,8;2950.98192000000,0.821830000000000,8;2949.16354900000,0.680937000000000,8;2954.61866200000,0.552295000000000,8;2960.07377400000,0.429779000000000,8;2963.71051500000,0.319515000000000,8;2965.52888600000,0.270509000000000,8;2965.52888600000,0.147993000000000,8;2965.52888600000,-0.0112780000000000,8;1036.23753100000,3.24764300000000,8;965.321072000000,3.24151700000000,8;800,3.05000000000000,8;1001.03421400000,3.28000000000000,8;805.304447000000,3.22314000000000,6;830.761638000000,3.26602000000000,6;859.855569000000,3.26602000000000,6;890.767872000000,3.27827200000000,6;925.316916000000,3.29052300000000,6;963.502702000000,3.29664900000000,6;999.870116000000,3.32115200000000,6;1019.87219500000,3.31502700000000,6;1045.32938500000,3.29664900000000,6;1079.87842900000,3.29052300000000,6;1101.69887800000,3.29052300000000,6;1134.42955100000,3.27214600000000,6;1174.43370700000,3.27214600000000,6;1219.89297600000,3.27827200000000,6;1258.07876100000,3.29052300000000,6;1285.35432300000,3.29052300000000,6;1310.81151300000,3.30890100000000,6;1350.81566900000,3.30890100000000,6;1383.54634200000,3.32115200000000,6;1414.45864500000,3.33340400000000,6;1481.73836200000,3.33953000000000,6;1587.20386500000,3.33953000000000,6;1536.28948500000,3.33953000000000,6;1447.18931800000,3.33953000000000,6;1632.66313400000,3.33953000000000,6;1696.30611000000,3.33953000000000,6;1725.40004200000,3.33953000000000,6;1756.31234400000,3.34565500000000,6;1787.22464700000,3.34565500000000,6;1807.22672500000,3.35790700000000,6;1849.04925200000,3.35790700000000,6;1890.87177900000,3.35790700000000,6;1927.23919400000,3.36403300000000,6;1965.42497900000,3.37015900000000,6;2009.06587700000,3.38241000000000,6;2070.89048200000,3.38241000000000,6;2125.44160400000,3.38241000000000,6;2181.81109700000,3.38241000000000,6;2209.08665800000,3.38853600000000,6;2283.63985900000,3.38853600000000,6;2323.64401500000,3.39466200000000,6;2438.20137200000,3.40078800000000,6;2372.74002500000,3.40078800000000,6;2490.93412300000,3.40078800000000,6;2556.39547000000,3.40078800000000,6;2627.31192900000,3.40078800000000,6;2707.32024100000,3.40078800000000,6;2758.23462200000,3.40078800000000,6;2803.69389000000,3.40078800000000,6;2830.96945100000,3.35178100000000,6;2865.51849500000,3.32115200000000,6;2892.79405700000,3.28439800000000,6;2912.79613500000,3.26602000000000,6;2920.06961800000,3.09449800000000,6;2925.52473000000,2.88622100000000,6;2930.97984200000,2.70244700000000,6;2940.07169600000,2.42678700000000,6;2940.07169600000,2.22463600000000,6;2945.52680800000,1.99185600000000,6;2947.34517900000,1.82645900000000,6;2952.80029100000,1.71619500000000,6;2952.80029100000,1.53242200000000,6;2956.43703200000,1.37315100000000,6;2958.25540300000,1.21388000000000,6;2958.25540300000,0.876962000000000,6;2963.71051500000,0.699314000000000,6;2969.16562800000,0.497163000000000,6;2970.98399800000,0.319515000000000,6;2978.25748100000,0.276634000000000,6;2978.25748100000,0.141867000000000,6;2978.25748100000,0.000974000000000000,6;800,3.20000000000000,6;805.304447000000,3.32115200000000,4;838.035121000000,3.32115200000000,4;865.310682000000,3.33340400000000,4;907.133209000000,3.34565500000000,4;950.774106000000,3.35790700000000,4;983.504780000000,3.36403300000000,4;1012.59871200000,3.37015900000000,4;1045.32938500000,3.35178100000000,4;1090.78865300000,3.35790700000000,4;1125.33769700000,3.34565500000000,4;1178.07044900000,3.34565500000000,4;1228.98483000000,3.34565500000000,4;1276.26246900000,3.35790700000000,4;1328.99522000000,3.36403300000000,4;1379.90960100000,3.37628400000000,4;1410.82190400000,3.39466200000000,4;1469.00976700000,3.39466200000000,4;1529.01600200000,3.39466200000000,4;1599.93246100000,3.38853600000000,4;1652.66521200000,3.38853600000000,4;1701.76122200000,3.38853600000000,4;1761.76745600000,3.38853600000000,4;1809.04509600000,3.40691300000000,4;1867.23295900000,3.40691300000000,4;1934.51267700000,3.40691300000000,4;1958.15149600000,3.41303900000000,4;2009.06587700000,3.41916500000000,4;2063.61699900000,3.41916500000000,4;2119.98649200000,3.41916500000000,4;2189.08458000000,3.41916500000000,4;2269.09289300000,3.41916500000000,4;2338.19098100000,3.41916500000000,4;2423.65440600000,3.41916500000000,4;2512.75457200000,3.41916500000000,4;2587.30777200000,3.41916500000000,4;2654.58749000000,3.41916500000000,4;2723.68557800000,3.41916500000000,4;2803.69389000000,3.41916500000000,4;2838.24293400000,3.37628400000000,4;2874.61034900000,3.33953000000000,4;2914.61450500000,3.28439800000000,4;2927.34310100000,3.27214600000000,4;2943.70843700000,3.26602000000000,4;2947.34517900000,3.10062400000000,4;2952.80029100000,2.92910200000000,4;2956.43703200000,2.68407000000000,4;2958.25540300000,2.45129000000000,4;2958.25540300000,2.31039700000000,4;2961.89214500000,2.20625800000000,4;2965.52888600000,2.04698800000000,4;2967.34725700000,1.90609500000000,4;2967.34725700000,1.69781800000000,4;2969.16562800000,1.55692500000000,4;2972.80236900000,1.43440900000000,4;2972.80236900000,1.23838400000000,4;2972.80236900000,1.07298700000000,4;2976.43911100000,0.968849000000000,4;2978.25748100000,0.772824000000000,4;2978.25748100000,0.625805000000000,4;2981.89422300000,0.527792000000000,4;2981.89422300000,0.356270000000000,4;2987.34933500000,0.331767000000000,4;2987.34933500000,0.147993000000000,4;2987.34933500000,0.000974000000000000,4;803.486076000000,3.41303900000000,2;839.853491000000,3.40691300000000,2;887.131131000000,3.41303900000000,2;918.043433000000,3.41303900000000,2;948.955736000000,3.41916500000000,2;992.596633000000,3.41916500000000,2;1032.60079000000,3.41916500000000,2;1098.06213600000,3.41916500000000,2;1158.06837100000,3.41916500000000,2;1236.25831300000,3.41916500000000,2;1316.26662500000,3.41916500000000,2;1352.63404000000,3.43141700000000,2;1405.36679100000,3.44366800000000,2;1449.00768900000,3.43754200000000,2;1512.65066500000,3.44366800000000,2;1561.74667500000,3.44366800000000,2;1616.29779700000,3.44979400000000,2;1709.03470500000,3.44366800000000,2;1778.13279300000,3.44979400000000,2;1865.41458900000,3.44979400000000,2;1930.87593500000,3.44979400000000,2;2014.52098900000,3.46204600000000,2;2056.34351600000,3.46204600000000,2;2125.44160400000,3.46204600000000,2;2209.08665800000,3.46204600000000,2;2267.27452200000,3.46204600000000,2;2334.55423900000,3.46204600000000,2;2410.92581000000,3.46204600000000,2;2498.20760600000,3.46204600000000,2;2594.58125500000,3.46204600000000,2;2678.22630900000,3.46204600000000,2;2738.23254400000,3.46204600000000,2;2803.69389000000,3.46204600000000,2;2836.42456400000,3.40691300000000,2;2872.79197800000,3.37628400000000,2;2907.34102200000,3.32727800000000,2;2958.25540300000,3.27214600000000,2;2970.98399800000,3.26602000000000,2;2976.43911100000,3.09449800000000,2;2976.43911100000,2.78820900000000,2;2983.71259400000,2.56155400000000,2;2981.89422300000,2.23076200000000,2;2983.71259400000,2.40841000000000,2;2976.43911100000,2.94747900000000,2;2981.89422300000,2.04086200000000,2;2983.71259400000,1.89996900000000,2;2983.71259400000,1.68556600000000,2;2983.71259400000,1.50179300000000,2;2983.71259400000,1.29964100000000,2;2987.34933500000,1.23225800000000,2;2987.34933500000,1.03623200000000,2;2990.98607600000,0.840207000000000,2;2990.98607600000,0.650308000000000,2;2990.98607600000,0.442031000000000,2;2998.25955900000,0.337892000000000,2;2998.25955900000,0.160244000000000,2;2998.25955900000,-0.00515200000000000,2;800,3.40000000000000,2;3038.26371600000,-0.00515200000000000,0;3305.56421400000,-0.0112780000000000,0;3647.41791400000,-0.0112780000000000,0;3816.52639200000,-0.0112780000000000,0;3816.52639200000,1.50791800000000,0;3820.16313400000,3.43754200000000,0;3816.52639200000,5.07312800000000,0;3387.39089800000,5.07312800000000,0;2920.06961800000,5.07312800000000,0;2432.74625900000,5.08538000000000,0;1961.78823800000,5.07925400000000,0;1565.38341600000,5.07925400000000,0;1167.16022400000,5.07925400000000,0;799.849335000000,5.07925400000000,0;798.030964000000,4.40541700000000,0;799.849335000000,3.59068700000000,0;1107.15399000000,4.05012100000000,0;1470.82813800000,4.06237300000000,0;1803.58998300000,4.06849900000000,0;2145.44368200000,4.07462500000000,0;2505.48108900000,4.10525400000000,0;2874.61034900000,4.10525400000000,0;3207.37219500000,3.71320300000000,0;3225.55590200000,2.80046000000000,0;3220.10079000000,2.01023300000000,0;3238.28449700000,1.22000600000000,0;3247.37635100000,0.521666000000000,0;3240.10286800000,-0.00515200000000000,0;798.030964000000,3.47429700000000,0;834.398379000000,3.49267400000000,0;872.584165000000,3.50492600000000,0;908.951579000000,3.52330300000000,0;963.502702000000,3.52942900000000,0;1019.87219500000,3.52942900000000,0;1065.33146300000,3.52942900000000,0;1112.60910200000,3.53555500000000,0;1158.06837100000,3.54780700000000,0;1203.52763900000,3.57231000000000,0;1250.80527800000,3.56005800000000,0;1318.08499600000,3.56618400000000,0;1372.63611800000,3.60293900000000,0;1425.36886900000,3.60293900000000,0;1487.19347500000,3.60293900000000,0;1543.56296800000,3.60293900000000,0;1603.56920200000,3.62744200000000,0;1659.93869500000,3.62744200000000,0;1730.85515400000,3.62744200000000,0;1792.67975900000,3.62744200000000,0;1856.32273500000,3.65807100000000,0;1927.23919400000,3.67032200000000,0;2049.07003300000,3.65807100000000,0;2056.34351600000,3.65807100000000,0;1990.88217000000,3.66419700000000,0;2147.26205300000,3.65807100000000,0;2239.99896100000,3.65807100000000,0;2309.09704900000,3.66419700000000,0;2392.74210300000,3.66419700000000,0;2452.74833700000,3.65807100000000,0;2523.66479600000,3.65807100000000,0;2605.49148000000,3.65194500000000,0;2674.58956800000,3.65194500000000,0;2758.23462200000,3.65194500000000,0;2809.14900200000,3.65194500000000,0;2869.15523700000,3.60293900000000,0;2941.89006700000,3.53555500000000,0;2992.80444700000,3.46204600000000,0;3036.44534500000,3.37628400000000,0;3036.44534500000,3.14963000000000,0;3036.44534500000,2.86171800000000,0;3036.44534500000,2.62281200000000,0;3043.71882800000,2.23076200000000,0;3038.26371600000,2.41453500000000,0;3050.99231100000,1.97347800000000,0;3060.08416500000,1.67331500000000,0;3063.72090600000,1.42215700000000,0;3067.35764800000,1.17100000000000,0;3074.63113100000,0.987226000000000,0;3076.44950100000,0.760572000000000,0;3081.90461300000,0.589050000000000,0;3085.54135500000,0.399150000000000,0;3085.54135500000,0.203125000000000,0;3087.35972600000,-0.0112780000000000,0;799.849335000000,3.99498900000000,0;1878.14318400000,3.56005800000000,0;1949.05964300000,3.56005800000000,0;2009.06587700000,3.56618400000000,0;2074.52722400000,3.57231000000000,0;2130.89671700000,3.57231000000000,0;2187.26620900000,3.57231000000000,0;2243.63570200000,3.57231000000000,0;2294.55008300000,3.57231000000000,0;2361.82980000000,3.56618400000000,0;2416.38092300000,3.56618400000000,0;2470.93204500000,3.56618400000000,0;2529.11990900000,3.56005800000000,0;2587.30777200000,3.56005800000000,0;2643.67726500000,3.56005800000000,0;2701.86512900000,3.56005800000000,0;2747.32439700000,3.56005800000000,0;2801.87552000000,3.56005800000000,0;2856.42664200000,3.51105200000000,0;2914.61450500000,3.43754200000000,0;2970.98399800000,3.34565500000000,0;1819.95532000000,3.49267400000000,0;1883.59829600000,3.49880000000000,0;1958.15149600000,3.49880000000000,0;2021.79447200000,3.51717800000000,0;2074.52722400000,3.51717800000000,0;2138.17020000000,3.51717800000000,0;2203.63154600000,3.51717800000000,0;2267.27452200000,3.51105200000000,0;2321.82564400000,3.51105200000000,0;2372.74002500000,3.51105200000000,0;2418.19929300000,3.51105200000000,0;2480.02389900000,3.51105200000000,0;2536.39339200000,3.51105200000000,0;2581.85266000000,3.51105200000000,0;2632.76704100000,3.51105200000000,0;2670.95282600000,3.50492600000000,0;2716.41209500000,3.50492600000000,0;2781.87344100000,3.51105200000000,0;2801.87552000000,3.51105200000000,0;2829.15108100000,3.48042300000000,0;2860.06338300000,3.44366800000000,0;2890.97568600000,3.41916500000000,0;2923.70635900000,3.38241000000000,0;2958.25540300000,3.34565500000000,0;1764.54599500000,3.50006400000000,0;1714.99114400000,3.50006400000000,0;1647.08264400000,3.50006400000000,0;1603.03388700000,3.50006400000000,0;1538.79611700000,3.50623300000000,0;1472.72298200000,3.50623300000000,0;1437.85105000000,3.51240100000000,0;1386.46083400000,3.50623300000000,0;1342.41207700000,3.50006400000000,0;1298.36332100000,3.48772700000000,0;1261.65602300000,3.48772700000000,0;1221.27799700000,3.49389500000000,0;1180.89997000000,3.48772700000000,0;1138.68657800000,3.48772700000000,0;1101.97928100000,3.48772700000000,0;1067.10734800000,3.49389500000000,0;1023.05859200000,3.50623300000000,0;991.857389000000,3.49389500000000,0;958.820822000000,3.50006400000000,0;923.948889000000,3.50623300000000,0;852.369660000000,3.50623300000000,0;815.662363000000,3.50006400000000,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddCylDeact, RmvCylDeact, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/CylDeact.png'));
    title('Ricardo CylDeact Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:2:50] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0 5]);
    xlim([800 3500]);
	%}
end


%% 9: Add/Remove technology: GDI
% map_data = [GDI5;GDI4p5;GDI4;GDI3p5;GDI3;GDI2p5;...
% GDI2;GDI1p5;GDI1;GDI0p5;GDI0];

if (AddGDI==1 || RmvGDI==1)
	%     './data/GDI_20150617_3.mat'
	map_data = [2000.76161500000,6.02555900000000,5;2000.76161500000,5.78274800000000,5;2000.76161500000,5.57827500000000,5;2000.76161500000,5.36102200000000,5;2000.76161500000,5.07987200000000,5;2000.76161500000,5.23322700000000,5;1496.95354200000,6.02555900000000,4.50000000000000;1634.04417400000,6.15335500000000,4.50000000000000;1723.15308500000,6.23003200000000,4.50000000000000;1808.83473000000,6.30670900000000,4.50000000000000;1884.23457700000,6.39616600000000,4.50000000000000;2000.76161500000,6.51118200000000,4.50000000000000;2124.14318400000,6.42172500000000,4.50000000000000;2292.07920800000,6.29393000000000,4.50000000000000;2418.88804300000,6.23003200000000,4.50000000000000;2545.69687700000,6.12779600000000,4.50000000000000;2627.95125700000,6.06389800000000,4.50000000000000;2713.63290200000,6.01278000000000,4.50000000000000;2795.88728100000,5.82108600000000,4.50000000000000;2881.56892600000,5.64217300000000,4.50000000000000;2963.82330500000,5.45047900000000,4.50000000000000;3004.95049500000,5.36102200000000,4.50000000000000;3056.35948200000,5.06709300000000,4.50000000000000;3056.35948200000,4.77316300000000,4.50000000000000;3063.21401400000,4.50479200000000,4.50000000000000;3066.64128000000,4.24920100000000,4.50000000000000;3066.64128000000,3.96805100000000,4.50000000000000;3046.07768500000,3.69968100000000,4.50000000000000;3022.08682400000,3.35463300000000,4.50000000000000;3011.80502700000,3.12460100000000,4.50000000000000;2998.09596300000,3.03514400000000,4.50000000000000;2912.41431800000,3.07348200000000,4.50000000000000;2826.73267300000,3.12460100000000,4.50000000000000;2703.35110400000,3.17571900000000,4.50000000000000;2576.54227000000,3.23961700000000,4.50000000000000;2453.16070100000,3.30351400000000,4.50000000000000;2343.48819500000,3.34185300000000,4.50000000000000;2244.09748700000,3.40575100000000,4.50000000000000;2158.41584200000,3.45686900000000,4.50000000000000;2079.58872800000,3.48242800000000,4.50000000000000;1993.90708300000,3.52076700000000,4.50000000000000;2213.25209400000,6.37060700000000,4.50000000000000;1911.65270400000,3.63578300000000,4.50000000000000;1825.97105900000,3.76357800000000,4.50000000000000;1743.71667900000,3.85303500000000,4.50000000000000;1661.46230000000,3.96805100000000,4.50000000000000;1623.76237600000,4.01916900000000,4.50000000000000;1592.91698400000,4.27476000000000,4.50000000000000;1562.07159200000,4.50479200000000,4.50000000000000;1534.65346500000,4.74760400000000,4.50000000000000;1510.66260500000,4.97763600000000,4.50000000000000;1500.38080700000,5.10543100000000,4.50000000000000;1500.38080700000,5.33546300000000,4.50000000000000;1500.38080700000,5.56549500000000,4.50000000000000;1500.38080700000,5.79552700000000,4.50000000000000;1551.78979400000,6.06389800000000,4.50000000000000;996.572734000000,6.01278000000000,4;996.572734000000,5.70607000000000,4;996.572734000000,5.33546300000000,4;996.572734000000,4.97763600000000,4;996.572734000000,4.58147000000000,4;996.572734000000,4.26198100000000,4;996.572734000000,4.00639000000000,4;1082.25437900000,6.11501600000000,4;1167.93602400000,6.17891400000000,4;1250.19040400000,6.26837100000000,4;1335.87204900000,6.34504800000000,4;1459.25361800000,6.46006400000000,4;1544.93526300000,6.54952100000000,4;1627.18964200000,6.62619800000000,4;1712.87128700000,6.71565500000000,4;1795.12566600000,6.79233200000000,4;1877.38004600000,6.88178900000000,4;1959.63442500000,6.95846600000000,4;2004.18888000000,7.00958500000000,4;2083.01599400000,7.00958500000000,4;2168.69763900000,7.00958500000000,4;2254.37928400000,7.00958500000000,4;2340.06092900000,7.00958500000000,4;2422.31530800000,7.00958500000000,4;2511.42421900000,7.00958500000000,4;2600.53313000000,7.00958500000000,4;2672.50571200000,7.00958500000000,4;2758.18735700000,7.00958500000000,4;2840.44173600000,7.00958500000000,4;2919.26885000000,7.00958500000000,4;3004.95049500000,7.00958500000000,4;3049.50495000000,6.76677300000000,4;3087.20487400000,6.53674100000000,4;3135.18659600000,6.30670900000000,4;3176.31378500000,6.07667700000000,4;3210.58644300000,5.83386600000000,4;3248.28636700000,5.59105400000000,4;3282.55902500000,5.36102200000000,4;3316.83168300000,5.13099000000000,4;3333.96801200000,5.04153400000000,4;3340.82254400000,4.82428100000000,4;3340.82254400000,4.65814700000000,4;3351.10434100000,4.49201300000000,4;3351.10434100000,4.30031900000000,4;3357.95887300000,4.17252400000000,4;3357.95887300000,3.94249200000000,4;3337.39527800000,3.69968100000000,4;3320.25894900000,3.48242800000000,4;3313.40441700000,3.23961700000000,4;3303.12262000000,3.02236400000000,4;3207.15917700000,2.88178900000000,4;3121.47753200000,2.74121400000000,4;3025.51409000000,2.56230000000000,4;2915.84158400000,2.56230000000000,4;2830.15993900000,2.58785900000000,4;2747.90556000000,2.58785900000000,4;2662.22391500000,2.61341900000000,4;2538.84234600000,2.63897800000000,4;2412.03351100000,2.66453700000000,4;2288.65194200000,2.69009600000000,4;2202.97029700000,2.71565500000000,4;2079.58872800000,2.74121400000000,4;1993.90708300000,2.76677300000000,4;1911.65270400000,2.84345000000000,4;1825.97105900000,2.90734800000000,4;1740.28941400000,2.97124600000000,4;1702.58949000000,2.99680500000000,4;1661.46230000000,3.04792300000000,4;1575.78065500000,3.16293900000000,4;1493.52627600000,3.27795500000000,4;1411.27189600000,3.40575100000000,4;1325.59025100000,3.54632600000000,4;1202.20868200000,3.71246000000000,4;1116.52703700000,3.85303500000000,4;1030.84539200000,3.95527200000000,4;1000,6.83067100000000,3.50000000000000;1085.68164500000,6.92012800000000,3.50000000000000;1167.93602400000,6.99680500000000,3.50000000000000;1253.61766900000,7.04792300000000,3.50000000000000;1335.87204900000,7.08626200000000,3.50000000000000;1418.12642800000,7.15016000000000,3.50000000000000;1503.80807300000,7.20127800000000,3.50000000000000;1586.06245200000,7.25239600000000,3.50000000000000;1668.31683200000,7.30351400000000,3.50000000000000;1750.57121100000,7.34185300000000,3.50000000000000;1836.25285600000,7.40575100000000,3.50000000000000;1918.50723500000,7.45686900000000,3.50000000000000;2004.18888000000,7.50798700000000,3.50000000000000;2100.15232300000,7.50798700000000,3.50000000000000;2168.69763900000,7.50798700000000,3.50000000000000;2254.37928400000,7.50798700000000,3.50000000000000;2336.63366300000,7.50798700000000,3.50000000000000;2422.31530800000,7.50798700000000,3.50000000000000;2507.99695400000,7.50798700000000,3.50000000000000;2597.10586400000,7.50798700000000,3.50000000000000;2717.06016800000,7.50798700000000,3.50000000000000;2802.74181300000,7.50798700000000,3.50000000000000;2884.99619200000,7.50798700000000,3.50000000000000;2953.54150800000,7.50798700000000,3.50000000000000;3004.95049500000,7.50798700000000,3.50000000000000;3087.20487400000,7.38019200000000,3.50000000000000;3169.45925400000,7.25239600000000,3.50000000000000;3248.28636700000,7.11182100000000,3.50000000000000;3296.26808800000,7.04792300000000,3.50000000000000;3330.54074600000,6.99680500000000,3.50000000000000;3364.81340400000,6.74121400000000,3.50000000000000;3402.51332800000,6.49840300000000,3.50000000000000;3440.21325200000,6.25559100000000,3.50000000000000;3488.19497300000,5.91054300000000,3.50000000000000;3525.89489700000,5.66773200000000,3.50000000000000;3560.16755500000,5.43770000000000,3.50000000000000;3594.44021300000,5.19488800000000,3.50000000000000;3618.43107400000,5.04153400000000,3.50000000000000;3625.28560500000,4.73482400000000,3.50000000000000;3638.99466900000,4.37699700000000,3.50000000000000;3642.42193500000,4.15974400000000,3.50000000000000;3645.84920000000,3.96805100000000,3.50000000000000;3632.14013700000,3.68690100000000,3.50000000000000;3615.00380800000,3.45686900000000,3.50000000000000;3601.29474500000,3.21405800000000,3.50000000000000;3591.01294700000,3.02236400000000,3.50000000000000;3488.19497300000,2.85623000000000,3.50000000000000;3436.78598600000,2.76677300000000,3.50000000000000;3385.37699900000,2.69009600000000,3.50000000000000;3347.67707500000,2.61341900000000,3.50000000000000;3282.55902500000,2.52396200000000,3.50000000000000;3214.01370900000,2.42172500000000,3.50000000000000;3111.19573500000,2.25559100000000,3.50000000000000;3025.51409000000,2.11501600000000,3.50000000000000;2905.55978700000,2.11501600000000,3.50000000000000;2778.75095200000,2.11501600000000,3.50000000000000;2737.62376200000,2.12779600000000,3.50000000000000;2837.01447100000,2.11501600000000,3.50000000000000;2655.36938300000,2.12779600000000,3.50000000000000;2559.40594100000,2.12779600000000,3.50000000000000;2466.86976400000,2.12779600000000,3.50000000000000;2367.47905600000,2.12779600000000,3.50000000000000;2278.37014500000,2.12779600000000,3.50000000000000;2254.37928400000,2.12779600000000,3.50000000000000;2178.97943600000,2.14057500000000,3.50000000000000;2103.57958900000,2.14057500000000,3.50000000000000;2007.61614600000,2.14057500000000,3.50000000000000;1932.21629900000,2.19169300000000,3.50000000000000;1856.81645100000,2.23003200000000,3.50000000000000;1774.56207200000,2.29393000000000,3.50000000000000;1678.59862900000,2.37060700000000,3.50000000000000;1613.48057900000,2.40894600000000,3.50000000000000;1558.64432600000,2.44728400000000,3.50000000000000;1486.67174400000,2.49840300000000,3.50000000000000;1428.40822500000,2.53674100000000,3.50000000000000;1370.14470700000,2.57508000000000,3.50000000000000;1305.02665700000,2.62619800000000,3.50000000000000;1233.05407500000,2.67731600000000,3.50000000000000;1161.08149300000,2.72843500000000,3.50000000000000;1102.81797400000,2.76677300000000,3.50000000000000;1054.83625300000,2.80511200000000,3.50000000000000;1013.70906300000,2.83067100000000,3.50000000000000;965.727342000000,3.02236400000000,3.50000000000000;941.736481000000,3.25239600000000,3.50000000000000;921.172887000000,3.49520800000000,3.50000000000000;900.609292000000,3.72524000000000,3.50000000000000;876.618431000000,3.98083100000000,3.50000000000000;876.618431000000,4.21086300000000,3.50000000000000;873.191165000000,4.50479200000000,3.50000000000000;873.191165000000,4.79872200000000,3.50000000000000;873.191165000000,5.01597400000000,3.50000000000000;873.191165000000,5.24600600000000,3.50000000000000;880.045697000000,5.46325900000000,3.50000000000000;880.045697000000,5.74440900000000,3.50000000000000;890.327494000000,5.85942500000000,3.50000000000000;890.327494000000,6.02555900000000,3.50000000000000;917.745621000000,6.20447300000000,3.50000000000000;952.018279000000,6.44728400000000,3.50000000000000;972.581874000000,6.61341900000000,3.50000000000000;907.463823000000,6.99680500000000,3;952.018279000000,7.07348200000000,3;1000,7.15016000000000,3;1082.25437900000,7.22683700000000,3;1167.93602400000,7.30351400000000,3;1250.19040400000,7.38019200000000,3;1291.31759300000,7.41853000000000,3;1376.99923800000,7.48242800000000,3;1459.25361800000,7.55910500000000,3;1544.93526300000,7.62300300000000,3;1630.61690800000,7.69968100000000,3;1712.87128700000,7.76357800000000,3;1798.55293200000,7.84025600000000,3;1880.80731200000,7.90415300000000,3;1963.06169100000,7.98083100000000,3;2004.18888000000,8.00639000000000,3;2083.01599400000,8.00639000000000,3;2168.69763900000,8.00639000000000,3;2254.37928400000,8.00639000000000,3;2374.33358700000,8.00639000000000,3;2490.86062500000,8.00639000000000,3;2586.82406700000,8.00639000000000,3;2672.50571200000,8.00639000000000,3;2751.33282600000,8.00639000000000,3;2837.01447100000,8.00639000000000,3;2926.12338200000,8.00639000000000,3;3004.95049500000,8.00639000000000,3;3083.77760900000,7.89137400000000,3;3172.88651900000,7.75079900000000,3;3255.14089900000,7.63578300000000,3;3340.82254400000,7.48242800000000,3;3419.64965700000,7.38019200000000,3;3505.33130200000,7.23961700000000,3;3591.01294700000,7.11182100000000,3;3669.84006100000,6.97124600000000,3;3690.40365600000,6.74121400000000,3;3714.39451600000,6.49840300000000,3;3741.81264300000,6.26837100000000,3;3745.23990900000,6.15335500000000,3;3765.80350300000,6.03833900000000,3;3776.08530100000,5.91054300000000,3;3806.93069300000,5.68051100000000,3;3841.20335100000,5.45047900000000,3;3851.48514900000,5.32268400000000,3;3865.19421200000,5.20766800000000,3;3882.33054100000,5.09265200000000,3;3892.61233800000,4.96485600000000,3;3902.89413600000,4.84984000000000,3;3906.32140100000,4.73482400000000,3;3906.32140100000,4.61980800000000,3;3916.60319900000,4.49201300000000,3;3923.45773000000,4.33865800000000,3;3923.45773000000,4.27476000000000,3;3930.31226200000,4.14696500000000,3;3940.59405900000,4.03194900000000,3;3930.31226200000,3.90415300000000,3;3930.31226200000,3.80191700000000,3;3916.60319900000,3.55910500000000,3;3902.89413600000,3.31629400000000,3;3889.18507200000,3.21405800000000,3;3892.61233800000,3.08626200000000,3;3865.19421200000,2.99680500000000,3;3779.51256700000,2.85623000000000,3;3697.25818700000,2.71565500000000,3;3618.43107400000,2.57508000000000,3;3532.74942900000,2.42172500000000,3;3447.06778400000,2.28115000000000,3;3361.38613900000,2.12779600000000,3;3282.55902500000,2.01278000000000,3;3196.87738000000,1.92332300000000,3;3111.19573500000,1.84664500000000,3;3032.36862100000,1.75718800000000,3;2987.81416600000,1.73162900000000,3;2946.68697600000,1.73162900000000,3;2902.13252100000,1.74440900000000,3;2823.30540700000,1.74440900000000,3;2778.75095200000,1.74440900000000,3;2747.90556000000,1.76996800000000,3;2614.24219300000,1.76996800000000,3;2528.56054800000,1.76996800000000,3;2487.43335900000,1.79552700000000,3;2401.75171400000,1.79552700000000,3;2319.49733400000,1.80830700000000,3;2278.37014500000,1.82108600000000,3;2192.68850000000,1.82108600000000,3;2113.86138600000,1.83386600000000,3;2069.30693100000,1.84664500000000,3;1997.33434900000,1.84664500000000,3;1945.92536200000,1.84664500000000,3;1901.37090600000,1.87220400000000,3;1819.11652700000,1.87220400000000,3;1777.98933700000,1.89776400000000,3;1654.60776800000,1.89776400000000,3;1712.87128700000,1.89776400000000,3;1610.05331300000,1.91054300000000,3;1524.37166800000,1.91054300000000,3;1483.24447800000,1.92332300000000,3;1400.99009900000,1.92332300000000,3;1359.86290900000,1.93610200000000,3;1315.30845400000,1.96166100000000,3;1233.05407500000,1.96166100000000,3;1191.92688500000,1.98722000000000,3;1106.24524000000,1.98722000000000,3;1027.41812600000,2.01278000000000,3;1000,2,3;965.727342000000,2.10223600000000,3;941.736481000000,2.28115000000000,3;900.609292000000,2.53674100000000,3;852.627570000000,2.75399400000000,3;808.073115000000,2.98402600000000,3;790.936786000000,3.11182100000000,3;784.082254000000,3.22683700000000,3;780.654989000000,3.35463300000000,3;773.800457000000,3.46964900000000,3;766.945925000000,3.58466500000000,3;749.809596000000,3.82747600000000,3;742.955065000000,3.94249200000000,3;742.955065000000,4.17252400000000,3;742.955065000000,4.41533500000000,3;742.955065000000,4.64536700000000,3;742.955065000000,4.87539900000000,3;742.955065000000,5.11821100000000,3;749.809596000000,5.36102200000000,3;756.664128000000,5.59105400000000,3;766.945925000000,5.82108600000000,3;784.082254000000,6.06389800000000,3;814.927647000000,6.29393000000000,3;842.345773000000,6.52396200000000,3;876.618431000000,6.75399400000000,3;890.327494000000,6.88178900000000,3;806.839940000000,7.02258800000000,2.50000000000000;889.863052000000,7.15127700000000,2.50000000000000;931.374608000000,7.25422800000000,2.50000000000000;1004.01983000000,7.37004800000000,2.50000000000000;1083.58364600000,7.43439300000000,2.50000000000000;1170.06605400000,7.51160700000000,2.50000000000000;1253.08916600000,7.57595100000000,2.50000000000000;1332.65298100000,7.64029600000000,2.50000000000000;1415.67609300000,7.71750900000000,2.50000000000000;1502.15850100000,7.79472300000000,2.50000000000000;1586,7.87193600000000,2.50000000000000;1668.20472500000,7.93628100000000,2.50000000000000;1751.22783700000,8.00062600000000,2.50000000000000;1837.71024500000,8.07783900000000,2.50000000000000;1917.27406000000,8.16792200000000,2.50000000000000;2003.75646800000,8.23226600000000,2.50000000000000;2086.77958000000,8.25800400000000,2.50000000000000;2045.26802400000,8.24226600000000,2.50000000000000;2169.80269200000,8.28374200000000,2.50000000000000;2252.82580400000,8.31000000000000,2.50000000000000;2335.84891600000,8.33521700000000,2.50000000000000;2418.87202700000,8.36095500000000,2.50000000000000;2505.35443600000,8.38669300000000,2.50000000000000;2588.37754700000,8.40466930000000,2.50000000000000;2671.40065900000,8.41243100000000,2.50000000000000;2754.42377100000,8.43816900000000,2.50000000000000;2837.44688300000,8.45103800000000,2.50000000000000;2920.46999500000,8.47677600000000,2.50000000000000;3006.95240300000,8.50251300000000,2.50000000000000;3048.46395900000,8.46392200000000,2.50000000000000;3089.97551400000,8.45090100000000,2.50000000000000;3172.99862600000,8.38579800000000,2.50000000000000;3252.56244200000,8.33371500000000,2.50000000000000;3339.04485000000,8.25559100000000,2.50000000000000;3425.52725800000,8.21652900000000,2.50000000000000;3505.09107300000,8.15142600000000,2.50000000000000;3591.57348200000,8.09934300000000,2.50000000000000;3633.08503700000,8.04726000000000,2.50000000000000;3674.59659300000,8.02121900000000,2.50000000000000;3723.02674200000,8.00819800000000,2.50000000000000;3747.24181600000,7.89101200000000,2.50000000000000;3812.96844600000,7.65664000000000,2.50000000000000;3882.15437300000,7.42226800000000,2.50000000000000;3951.34029900000,7.17487500000000,2.50000000000000;4017.06693000000,6.94050300000000,2.50000000000000;4117.38652300000,6.71915100000000,2.50000000000000;4058.57848500000,6.86237900000000,2.50000000000000;4183.11315300000,6.54988200000000,2.50000000000000;4266.13626500000,6.34155200000000,2.50000000000000;4349.15937700000,6.12020000000000,2.50000000000000;4394.13022900000,6.02905500000000,2.50000000000000;4414.88600700000,5.92489000000000,2.50000000000000;4466.77545200000,5.67749700000000,2.50000000000000;4518.66489700000,5.44312500000000,2.50000000000000;4560.17645300000,5.19573200000000,2.50000000000000;4587.85082300000,5.09156700000000,2.50000000000000;4608.60660100000,4.85719500000000,2.50000000000000;4625.90308300000,4.62282200000000,2.50000000000000;4643.19956400000,4.38845000000000,2.50000000000000;4660.49604600000,4.15407800000000,2.50000000000000;4667.41463900000,4.02387100000000,2.50000000000000;4632.82167500000,3.80252000000000,2.50000000000000;4591.31012000000,3.55512700000000,2.50000000000000;4549.79856400000,3.33377500000000,2.50000000000000;4532.50208200000,3.20356900000000,2.50000000000000;4511.74630400000,3.09940300000000,2.50000000000000;4501.36841500000,3.00825800000000,2.50000000000000;4442.56037800000,2.96919600000000,2.50000000000000;4356.07796900000,2.90409300000000,2.50000000000000;4276.51415400000,2.83899000000000,2.50000000000000;4231.54330200000,2.79992800000000,2.50000000000000;4190.03174600000,2.77388600000000,2.50000000000000;4148.52019000000,2.74784500000000,2.50000000000000;4065.49707800000,2.69576200000000,2.50000000000000;4023.98552200000,2.66972100000000,2.50000000000000;3982.47396600000,2.61763800000000,2.50000000000000;3899.45085500000,2.46139000000000,2.50000000000000;3812.96844600000,2.30514200000000,2.50000000000000;3733.40463100000,2.16191500000000,2.50000000000000;3646.92222300000,2.03170800000000,2.50000000000000;3563.89911100000,1.92754200000000,2.50000000000000;3480.87599900000,1.84941800000000,2.50000000000000;3439.36444300000,1.79733600000000,2.50000000000000;3397.85288700000,1.77129400000000,2.50000000000000;3314.82977600000,1.66712900000000,2.50000000000000;3273.31822000000,1.62806700000000,2.50000000000000;3231.80666400000,1.60202500000000,2.50000000000000;3148.78355200000,1.51088100000000,2.50000000000000;3065.76044000000,1.40671500000000,2.50000000000000;2979.27803200000,1.38067400000000,2.50000000000000;2896.25492000000,1.39369500000000,2.50000000000000;2813.23180800000,1.40671500000000,2.50000000000000;2726.74940000000,1.43275700000000,2.50000000000000;2647.18558500000,1.45879800000000,2.50000000000000;2564.16247300000,1.48483900000000,2.50000000000000;2481.13936100000,1.51088100000000,2.50000000000000;2394.65695300000,1.53692200000000,2.50000000000000;2311.63384100000,1.56296300000000,2.50000000000000;2225.15143300000,1.58900500000000,2.50000000000000;2145.58761800000,1.58900500000000,2.50000000000000;2059.10521000000,1.61504600000000,2.50000000000000;1976.08209800000,1.62806700000000,2.50000000000000;1855.00672600000,1.64108700000000,2.50000000000000;1768.52431800000,1.64108700000000,2.50000000000000;1685.50120600000,1.66712900000000,2.50000000000000;1599.01879800000,1.66712900000000,2.50000000000000;1519.45498300000,1.69317000000000,2.50000000000000;1432.97257500000,1.69317000000000,2.50000000000000;1391.46101900000,1.71921200000000,2.50000000000000;1308.43790700000,1.71921200000000,2.50000000000000;1266.92635100000,1.71921200000000,2.50000000000000;1228.87409200000,1.74525300000000,2.50000000000000;1183.90323900000,1.74525300000000,2.50000000000000;1100.88012800000,1.74525300000000,2.50000000000000;1017.85701600000,1.77129400000000,2.50000000000000;976.345460000000,1.79733600000000,2.50000000000000;931.374608000000,1.83639800000000,2.50000000000000;848.351496000000,1.92754200000000,2.50000000000000;806.839940000000,1.97962500000000,2.50000000000000;758.409791000000,2.20097700000000,2.50000000000000;709.979643000000,2.52649300000000,2.50000000000000;668.468087000000,2.77388600000000,2.50000000000000;633.875124000000,3.00825800000000,2.50000000000000;623.497235000000,3.25565100000000,2.50000000000000;623.497235000000,3.49002400000000,2.50000000000000;613.119346000000,3.72439600000000,2.50000000000000;606.200753000000,3.84158200000000,2.50000000000000;606.200753000000,4.07595400000000,2.50000000000000;606.200753000000,4.32334700000000,2.50000000000000;606.200753000000,4.54469800000000,2.50000000000000;606.200753000000,4.77907000000000,2.50000000000000;606.200753000000,5.01344300000000,2.50000000000000;623.497235000000,5.24781500000000,2.50000000000000;633.875124000000,5.49520800000000,2.50000000000000;647.712309000000,5.71655900000000,2.50000000000000;668.468087000000,5.96395200000000,2.50000000000000;689.223865000000,6.19832400000000,2.50000000000000;741.113310000000,6.54988200000000,2.50000000000000;775.706273000000,6.78425500000000,2.50000000000000;709.979643000000,6.31551000000000,2.50000000000000;2117.91324700000,8.54204600000000,2;2079.86098800000,8.54204600000000,2;1993.37857900000,8.48996300000000,2;1910.35546800000,8.41183900000000,2;1827.33235600000,8.33371500000000,2;1744.30924400000,8.24257000000000,2;1661.28613200000,8.16444600000000,2;1578.26302000000,8.07330200000000,2;1533.29216800000,8.02121900000000,2;1453.72835300000,7.95611500000000,2;1367.24594500000,7.89101200000000,2;1280.76353600000,7.81288800000000,2;1201.19972100000,7.74778500000000,2;1118.17660900000,7.66966100000000,2;1073.20575700000,7.65664000000000,2;993.641941000000,7.56549500000000,2;907.159533000000,7.40924700000000,2;865.647977000000,7.33112300000000,2;786.084162000000,7.17487500000000,2;741.113310000000,7.12279200000000,2;699.601754000000,7.01862700000000,2;658.090198000000,6.81029600000000,2;626.956531000000,6.57592400000000,2;588.904271000000,6.32853100000000,2;564.689197000000,6.09415900000000,2;533.555530000000,5.75562100000000,2;512.799752000000,5.52124900000000,2;512.799752000000,2.53951400000000,2;547.392716000000,2.30514200000000,2;533.555530000000,2.42232800000000,2;557.770605000000,2.18795600000000,2;571.607790000000,2.07077000000000,2;599.282160000000,1.97962500000000,2;678.845976000000,1.87546000000000,2;730.735421000000,1.83639800000000,2;806.839940000000,1.74525300000000,2;889.863052000000,1.62806700000000,2;972.886164000000,1.53692200000000,2;1017.85701600000,1.51088100000000,2;1100.88012800000,1.51088100000000,2;1142.39168300000,1.51088100000000,2;1183.90323900000,1.48483900000000,2;1266.92635100000,1.48483900000000,2;1349.94946300000,1.48483900000000,2;1432.97257500000,1.48483900000000,2;1477.94342700000,1.48483900000000,2;1519.45498300000,1.45879800000000,2;1605.93739100000,1.45879800000000,2;1716.63487300000,1.45879800000000,2;1810.03587400000,1.45879800000000,2;1851.54743000000,1.43275700000000,2;1934.57054200000,1.43275700000000,2;2017.59365400000,1.43275700000000,2;2059.10521000000,1.40671500000000,2;2145.58761800000,1.36765300000000,2;2228.61073000000,1.34161200000000,2;2311.63384100000,1.30255000000000,2;2394.65695300000,1.27650900000000,2;2477.68006500000,1.22442600000000,2;2560.70317700000,1.19838400000000,2;2647.18558500000,1.15932200000000,2;2685.23784400000,1.15932200000000,2;2771.72025300000,1.12026000000000,2;2854.74336400000,1.08119800000000,2;2937.76647600000,1.04213600000000,2;2979.27803200000,1.01609500000000,2;3065.76044000000,1.06817800000000,2;3186.83581200000,1.19838400000000,2;3314.82977600000,1.32859100000000,2;3442.82374000000,1.43275700000000,2;3567.35840700000,1.58900500000000,2;3691.89307500000,1.70619100000000,2;3816.42774300000,1.82337700000000,2;3899.45085500000,1.90150100000000,2;3982.47396600000,1.97962500000000,2;4023.98552200000,2.03170800000000,2;4107.00863400000,2.05774900000000,2;4190.03174600000,2.09681100000000,2;4273.05485800000,2.14889400000000,2;4401.04882200000,2.21399700000000,2;4525.58348900000,2.27910100000000,2;4608.60660100000,2.30514200000000,2;4691.62971300000,2.35722500000000,2;4816.16438100000,2.43534900000000,2;4982.21060400000,2.51347300000000,2;5068.69301200000,2.50045200000000,2;5193.22768000000,2.43534900000000,2;5321.22164400000,2.35722500000000,2;5442.29701500000,2.27910100000000,2;5566.83168300000,2.22701800000000,2;5694.82564700000,2.16191500000000,2;5781.30805500000,2.10983200000000,2;5860.87187100000,2.07077000000000,2;5902.38342700000,2.05774900000000,2;5961.19146400000,2.03170800000000,2;6000,2.01868700000000,2;6000,7.99517800000000,2;5923.13920500000,8.07330200000000,2;5836.65679700000,8.17746700000000,2;5712.12212900000,8.28163200000000,2;5584.12816500000,8.41183900000000,2;5504.56434900000,8.48996300000000,2;5376.57038500000,8.62017000000000,2;5293.54727300000,8.69829400000000,2;5169.01260600000,8.82850100000000,2;5041.01864200000,8.94568700000000,2;5002.96638200000,8.99777000000000,2;4836.92015900000,8.99777000000000,2;4681.25182400000,8.99777000000000,2;4501.36841500000,8.99777000000000,2;4290.35133900000,8.99777000000000,2;4124.30511600000,8.99777000000000,2;3958.25889200000,8.99777000000000,2;3792.21266800000,8.99777000000000,2;3626.16644500000,8.99777000000000,2;3498.17248100000,8.99777000000000,2;3328.66696100000,8.99777000000000,2;3148.78355200000,8.99777000000000,2;2996.57451400000,8.99777000000000,2;3038.08607000000,8.99777000000000,2;2955.06295800000,8.97172800000000,2;4964.91412300000,8.99777000000000,2;4944.15834500000,2.50045200000000,2;2830.52829000000,8.89360400000000,2;2747.50517800000,8.88058400000000,2;2661.02277000000,8.82850100000000,2;2619.51121400000,8.80245900000000,2;2536.48810200000,8.76339700000000,2;2456.92428700000,8.71131500000000,2;2370.44187900000,8.68527300000000,2;2287.41876700000,8.64621100000000,2;2200.93635900000,8.59412900000000,2;2159.42480300000,8.56808700000000,2;2915.84158400000,8.95207700000000,2;3080.35034300000,9.00319500000000,2;3121.47753200000,9.00319500000000,2;3207.15917700000,9.00319500000000,2;3248.28636700000,9.00319500000000,2;3289.41355700000,9.00319500000000,2;3375.09520200000,9.00319500000000,2;3433.35872000000,9.00319500000000,2;3539.60396000000,9.00319500000000,2;3580.73115000000,9.00319500000000,2;3669.84006100000,9.00319500000000,2;3731.53084500000,9.00319500000000,2;3844.63061700000,9.00319500000000,2;3889.18507200000,9.00319500000000,2;4019.42117300000,9.00319500000000,2;4067.40289400000,9.00319500000000,2;4166.79360200000,9.00319500000000,2;4204.49352600000,9.00319500000000,2;4249.04798200000,9.00319500000000,2;4372.42955100000,9.00319500000000,2;4427.26580400000,9.00319500000000,2;4547.22010700000,9.00319500000000,2;4595.20182800000,9.00319500000000,2;4629.47448600000,9.00319500000000,2;4718.58339700000,9.00319500000000,2;4756.28332100000,9.00319500000000,2;4790.55597900000,9.00319500000000,2;4879.66489000000,9.00319500000000,2;4924.21934500000,9.00319500000000,2;5085.30083800000,8.90095800000000,2;5123.00076200000,8.87539900000000,2;5212.10967300000,8.78594200000000,2;5256.66412800000,8.73482400000000,2;5332.06397600000,8.67092700000000,2;5421.17288700000,8.58147000000000,2;5462.30007600000,8.54313100000000,2;5544.55445500000,8.45367400000000,2;5630.23610100000,8.37699700000000,2;5671.36329000000,8.33865800000000,2;5753.61766900000,8.24920100000000,2;5801.59939100000,8.19808300000000,2;5880.42650400000,8.12140600000000,2;5962.68088300000,8.04472800000000,2;997.101238000000,7.78684700000000,1.50000000000000;1083.58364600000,7.85195000000000,1.50000000000000;1166.60675800000,7.93007400000000,1.50000000000000;1249.62987000000,7.99517800000000,1.50000000000000;1346.49016700000,8.09934300000000,1.50000000000000;1419.13538900000,8.16444600000000,1.50000000000000;1502.15850100000,8.25559100000000,1.50000000000000;1585.18161300000,8.33371500000000,1.50000000000000;1668.20472500000,8.43788100000000,1.50000000000000;1751.22783700000,8.48996300000000,1.50000000000000;1837.71024500000,8.59412900000000,1.50000000000000;1920.73335700000,8.67225300000000,1.50000000000000;2003.75646800000,8.76339700000000,1.50000000000000;2086.77958000000,8.80245900000000,1.50000000000000;2169.80269200000,8.82850100000000,1.50000000000000;2256.28510000000,8.88058400000000,1.50000000000000;2335.84891600000,8.91964600000000,1.50000000000000;2422.33132400000,8.94568700000000,1.50000000000000;2505.35443600000,8.99777000000000,1.50000000000000;2588.37754700000,9.03683200000000,1.50000000000000;2674.85995500000,9.08891400000000,1.50000000000000;2757.88306700000,9.12797600000000,1.50000000000000;2837.44688300000,9.15401800000000,1.50000000000000;2923.92929100000,9.20610000000000,1.50000000000000;3006.95240300000,9.25818300000000,1.50000000000000;3141.86495900000,9.25818300000000,1.50000000000000;3214.51018200000,9.25818300000000,1.50000000000000;3252.56244200000,9.28422400000000,1.50000000000000;3384.01570200000,9.28422400000000,1.50000000000000;3505.09107300000,9.28422400000000,1.50000000000000;3636.54433400000,9.28422400000000,1.50000000000000;3716.10814900000,9.28422400000000,1.50000000000000;3757.61970500000,9.31026600000000,1.50000000000000;3882.15437300000,9.31026600000000,1.50000000000000;3965.17748500000,9.31026600000000,1.50000000000000;4006.68904100000,9.31026600000000,1.50000000000000;4093.17144900000,9.32328700000000,1.50000000000000;4172.73526400000,9.33630700000000,1.50000000000000;4259.21767200000,9.34932800000000,1.50000000000000;4342.24078400000,9.36234900000000,1.50000000000000;4425.26389600000,9.36234900000000,1.50000000000000;4511.74630400000,9.38839000000000,1.50000000000000;4594.76941600000,9.41443100000000,1.50000000000000;4674.33323100000,9.41443100000000,1.50000000000000;4719.30408400000,9.44047300000000,1.50000000000000;4798.86789900000,9.44047300000000,1.50000000000000;4840.37945500000,9.47953500000000,1.50000000000000;4926.86186300000,9.46651400000000,1.50000000000000;4968.37341900000,9.49255500000000,1.50000000000000;5009.88497500000,9.49255500000000,1.50000000000000;5096.36738300000,9.38839000000000,1.50000000000000;5134.41964300000,9.34932800000000,1.50000000000000;5179.39049500000,9.32328700000000,1.50000000000000;5300.46586600000,9.18005900000000,1.50000000000000;5428.45983000000,9.06287300000000,1.50000000000000;5552.99449800000,8.94568700000000,1.50000000000000;5684.44775800000,8.82850100000000,1.50000000000000;5802.06383300000,8.68527300000000,1.50000000000000;5888.54624100000,8.62017000000000,1.50000000000000;5971.56935300000,8.51600500000000,1.50000000000000;6000,8.48996300000000,1.50000000000000;6000,1.39369500000000,1.50000000000000;5954.27287200000,1.39369500000000,1.50000000000000;5912.76131600000,1.40671500000000,1.50000000000000;5871.24976000000,1.40671500000000,1.50000000000000;5826.27890800000,1.43275700000000,1.50000000000000;5784.76735200000,1.43275700000000,1.50000000000000;5743.25579600000,1.45879800000000,1.50000000000000;5660.23268400000,1.45879800000000,1.50000000000000;5618.72112800000,1.48483900000000,1.50000000000000;5577.20957200000,1.48483900000000,1.50000000000000;5535.69801600000,1.51088100000000,1.50000000000000;5452.67490400000,1.51088100000000,1.50000000000000;5411.16334900000,1.53692200000000,1.50000000000000;5366.19249600000,1.52390100000000,1.50000000000000;5324.68094000000,1.54994300000000,1.50000000000000;5283.16938500000,1.54994300000000,1.50000000000000;5245.11712500000,1.58900500000000,1.50000000000000;5162.09401300000,1.57598400000000,1.50000000000000;5120.58245700000,1.60202500000000,1.50000000000000;5079.07090100000,1.60202500000000,1.50000000000000;5037.55934500000,1.62806700000000,1.50000000000000;4954.53623400000,1.62806700000000,1.50000000000000;4909.56538100000,1.62806700000000,1.50000000000000;4868.05382600000,1.60202500000000,1.50000000000000;4785.03071400000,1.60202500000000,1.50000000000000;4702.00760200000,1.60202500000000,1.50000000000000;4660.49604600000,1.58900500000000,1.50000000000000;4598.22871200000,1.58900500000000,1.50000000000000;4532.50208200000,1.58900500000000,1.50000000000000;4490.99052600000,1.56296300000000,1.50000000000000;4407.96741400000,1.56296300000000,1.50000000000000;4328.40359900000,1.56296300000000,1.50000000000000;4283.43274700000,1.53692200000000,1.50000000000000;4203.86893100000,1.53692200000000,1.50000000000000;4117.38652300000,1.53692200000000,1.50000000000000;4072.41567100000,1.51088100000000,1.50000000000000;3989.39255900000,1.51088100000000,1.50000000000000;3951.34029900000,1.45879800000000,1.50000000000000;3864.85789100000,1.36765300000000,1.50000000000000;3781.83477900000,1.30255000000000,1.50000000000000;3702.27096400000,1.19838400000000,1.50000000000000;3574.27700000000,1.08119800000000,1.50000000000000;3491.25388800000,1.00307400000000,1.50000000000000;3404.77148000000,0.964012000000000,1.50000000000000;3321.74836800000,0.924950000000000,1.50000000000000;3197.21370100000,0.846826000000000,1.50000000000000;3114.19058900000,0.794744000000000,1.50000000000000;3031.16747700000,0.768702000000000,1.50000000000000;2989.65592100000,0.742661000000000,1.50000000000000;2906.63280900000,0.794744000000000,1.50000000000000;2823.60969700000,0.833806000000000,1.50000000000000;2695.61573300000,0.911930000000000,1.50000000000000;2612.59262200000,0.950992000000000,1.50000000000000;2529.56951000000,0.990054000000000,1.50000000000000;2405.03484200000,1.05515700000000,1.50000000000000;2322.01173000000,1.08119800000000,1.50000000000000;2194.01776600000,1.14630200000000,1.50000000000000;2110.99465400000,1.17234300000000,1.50000000000000;2027.97154300000,1.21140500000000,1.50000000000000;1944.94843100000,1.22442600000000,1.50000000000000;1778.90220700000,1.22442600000000,1.50000000000000;1861.92531900000,1.22442600000000,1.50000000000000;1733.93135500000,1.25046700000000,1.50000000000000;1602.47809500000,1.25046700000000,1.50000000000000;1446.80976000000,1.25046700000000,1.50000000000000;1277.30424000000,1.25046700000000,1.50000000000000;1235.79268400000,1.27650900000000,1.50000000000000;1152.76957200000,1.27650900000000,1.50000000000000;1069.74646100000,1.27650900000000,1.50000000000000;1007.47912700000,1.27650900000000,1.50000000000000;945.211793000000,1.31557100000000,1.50000000000000;851.810792000000,1.38067400000000,1.50000000000000;720.357532000000,1.45879800000000,1.50000000000000;644.253013000000,1.52390100000000,1.50000000000000;568.148493000000,1.57598400000000,1.50000000000000;512.799752000000,1.62806700000000,1.50000000000000;505.881160000000,6.53686200000000,1.50000000000000;540.474123000000,6.77123400000000,1.50000000000000;571.607790000000,6.97956500000000,1.50000000000000;665.008791000000,7.14883400000000,1.50000000000000;751.491199000000,7.33112300000000,1.50000000000000;834.514311000000,7.48737100000000,1.50000000000000;917.537422000000,7.63059900000000,1.50000000000000;509.340456000000,7.16185400000000,1;592.363568000000,7.31810200000000,1;678.845976000000,7.46133000000000,1;761.869088000000,7.60455700000000,1;841.432903000000,7.74778500000000,1;924.456015000000,7.87799100000000,1;1000.56053400000,7.99517800000000,1;1093.96153500000,8.09934300000000,1;1173.52535000000,8.16444600000000,1;1260.00775900000,8.25559100000000,1;1343.03087000000,8.33371500000000,1;1426.05398200000,8.42486000000000,1;1509.07709400000,8.48996300000000,1;1592.10020600000,8.59412900000000,1;1671.66402100000,8.65923200000000,1;1761.60572600000,8.75037700000000,1;1841.16954100000,8.82850100000000,1;1927.65194900000,8.91964600000000,1;2003.75646800000,8.99777000000000,1;2097.15746900000,9.03683200000000,1;2180.18058100000,9.08891400000000,1;2263.20369300000,9.11495600000000,1;2342.76750800000,9.15401800000000,1;2432.70921300000,9.20610000000000,1;2515.73232400000,9.24516200000000,1;2595.29614000000,9.27120400000000,1;2678.31925200000,9.32328700000000,1;2764.80166000000,9.36234900000000,1;2847.82477200000,9.41443100000000,1;2930.84788300000,9.44047300000000,1;3010.41169900000,9.49255500000000,1;3100.35340300000,9.49255500000000,1;3141.86495900000,9.51859700000000,1;3228.34736700000,9.51859700000000,1;3349.42273900000,9.51859700000000,1;3394.39359100000,9.54463800000000,1;3473.95740700000,9.54463800000000,1;3598.49207400000,9.54463800000000,1;3681.51518600000,9.55765900000000,1;3723.02674200000,9.55765900000000,1;3809.50915000000,9.55765900000000,1;3851.02070600000,9.55765900000000,1;3892.53226200000,9.58370000000000,1;3979.01467000000,9.57067900000000,1;4017.06693000000,9.58370000000000,1;4100.09004100000,9.62276200000000,1;4186.57244900000,9.66182400000000,1;4269.59556100000,9.70088600000000,1;4394.13022900000,9.75296900000000,1;4477.15334100000,9.79203100000000,1;4605.14730500000,9.83109300000000,1;4733.14126900000,9.88317600000000,1;4850.75734400000,9.93525800000000,1;4937.23975200000,9.98734100000000,1;5002.96638200000,10.0133820000000,1;5148.25682800000,9.85713400000000,1;5310.84375500000,9.70088600000000,1;5438.83771900000,9.55765900000000,1;5566.83168300000,9.44047300000000,1;5687.90705500000,9.29724500000000,1;5815.90101900000,9.18005900000000,1;5898.92413000000,9.08891400000000,1;5940.43568600000,9.06287300000000,1;6000,8.98474900000000,1;6002.70302000000,0.846826000000000,1;5943.89498300000,0.846826000000000,1;5895.46483400000,0.872868000000000,1;5812.44172200000,0.872868000000000,1;5774.38946300000,0.898909000000000,1;5729.41861000000,0.898909000000000,1;5646.39549900000,0.898909000000000,1;5604.88394300000,0.924950000000000,1;5518.40153500000,0.924950000000000,1;5480.34927500000,0.937971000000000,1;5397.32616300000,0.937971000000000,1;5352.35531100000,0.964012000000000,1;5227.82064300000,0.964012000000000,1;5186.30908700000,0.990054000000000,1;5103.28597600000,0.990054000000000,1;5061.77442000000,1.01609500000000,1;4885.35030700000,1.00307400000000,1;4598.22871200000,1.00307400000000,1;4446.01967400000,1.00307400000000,1;4293.81063600000,1.00307400000000,1;4767.73423200000,0.990054000000000,1;4179.65385700000,1.00307400000000,1;4017.06693000000,1.00307400000000,1;3892.53226200000,0.950992000000000,1;3764.53829800000,0.885888000000000,1;3601.95137100000,0.807764000000000,1;3473.95740700000,0.742661000000000,1;3390.93429500000,0.716619000000000,1;3311.37047900000,0.664537000000000,1;3224.88807100000,0.625475000000000,1;3138.40566300000,0.573392000000000,1;3065.76044000000,0.534330000000000,1;2972.35943900000,0.534330000000000,1;2837.44688300000,0.586413000000000,1;2681.77854800000,0.677557000000000,1;2560.70317700000,0.729640000000000,1;2432.70921300000,0.794744000000000,1;2263.20369300000,0.885888000000000,1;2135.20972900000,0.937971000000000,1;2014.13435700000,1.00307400000000,1;1972.62280100000,1.01609500000000,1;1841.16954100000,1.01609500000000,1;1695.87909500000,1.01609500000000,1;1543.67005700000,1.01609500000000,1;1246.17057300000,1.01609500000000,1;983.264053000000,1.00307400000000,1;824.136422000000,1.00307400000000,1;661.549494000000,1.00307400000000,1;505.881160000000,1.00307400000000,1;505.881160000000,7.56549500000000,0.500000000000000;592.363568000000,7.72174300000000,0.500000000000000;675.386680000000,7.86497100000000,0.500000000000000;758.409791000000,8.02121900000000,0.500000000000000;844.892200000000,8.20350800000000,0.500000000000000;924.456015000000,8.35975600000000,0.500000000000000;1007.47912700000,8.50298400000000,0.500000000000000;1093.96153500000,8.59412900000000,0.500000000000000;1173.52535000000,8.65923200000000,0.500000000000000;1260.00775900000,8.75037700000000,0.500000000000000;1343.03087000000,8.82850100000000,0.500000000000000;1471.02483400000,8.95870800000000,0.500000000000000;1595.55950200000,9.08891400000000,0.500000000000000;1723.55346600000,9.20610000000000,0.500000000000000;1844.62883700000,9.32328700000000,0.500000000000000;1803.11728200000,9.29724500000000,0.500000000000000;1931.11124600000,9.41443100000000,0.500000000000000;2007.21576500000,9.49255500000000,0.500000000000000;2093.69817300000,9.51859700000000,0.500000000000000;2138.66902500000,9.51859700000000,0.500000000000000;2180.18058100000,9.53161700000000,0.500000000000000;2225.15143300000,9.54463800000000,0.500000000000000;2263.20369300000,9.55765900000000,0.500000000000000;2346.22680500000,9.57067900000000,0.500000000000000;2422.33132400000,9.57067900000000,0.500000000000000;2512.27302800000,9.62276200000000,0.500000000000000;2598.75543600000,9.64880300000000,0.500000000000000;2678.31925200000,9.64880300000000,0.500000000000000;2764.80166000000,9.67484500000000,0.500000000000000;2892.79562400000,9.72692700000000,0.500000000000000;3013.87099500000,9.75296900000000,0.500000000000000;3103.81270000000,9.76599000000000,0.500000000000000;3224.88807100000,9.76599000000000,0.500000000000000;3262.94033100000,9.79203100000000,0.500000000000000;3387.47499800000,9.79203100000000,0.500000000000000;3432.44585100000,9.80505200000000,0.500000000000000;3518.92825900000,9.81807200000000,0.500000000000000;3598.49207400000,9.84411400000000,0.500000000000000;3723.02674200000,9.83109300000000,0.500000000000000;3767.99759400000,9.85713400000000,0.500000000000000;3892.53226200000,9.85713400000000,0.500000000000000;3930.58452100000,9.88317600000000,0.500000000000000;4020.52622600000,9.89619600000000,0.500000000000000;4134.68300500000,9.93525800000000,0.500000000000000;4266.13626500000,10.0003620000000,0.500000000000000;4356.07796900000,10.0394240000000,0.500000000000000;4477.15334100000,10.1435890000000,0.500000000000000;4646.65886100000,10.2607750000000,0.500000000000000;4812.70508400000,10.3649410000000,0.500000000000000;4940.69904800000,10.4560850000000,0.500000000000000;5009.88497500000,10.5081680000000,0.500000000000000;5096.36738300000,10.4040030000000,0.500000000000000;5224.36134700000,10.2868170000000,0.500000000000000;5386.94827400000,10.1175480000000,0.500000000000000;5521.86083100000,9.98734100000000,0.500000000000000;5691.36635100000,9.80505200000000,0.500000000000000;5815.90101900000,9.67484500000000,0.500000000000000;5943.89498300000,9.55765900000000,0.500000000000000;6002.70302000000,9.46651400000000,0.500000000000000;6000,0.430165000000000,0.500000000000000;5892.00553800000,0.430165000000000,0.500000000000000;5947.35427900000,0.430165000000000,0.500000000000000;5850.49398200000,0.443185000000000,0.500000000000000;5805.52313000000,0.443185000000000,0.500000000000000;5722.50001800000,0.443185000000000,0.500000000000000;5639.47690600000,0.443185000000000,0.500000000000000;5597.96535000000,0.482247000000000,0.500000000000000;5480.34927500000,0.482247000000000,0.500000000000000;5390.40757100000,0.482247000000000,0.500000000000000;5348.89601500000,0.495268000000000,0.500000000000000;5258.95431000000,0.495268000000000,0.500000000000000;5137.87893900000,0.508289000000000,0.500000000000000;5096.36738300000,0.521309000000000,0.500000000000000;5016.80356700000,0.521309000000000,0.500000000000000;4971.83271500000,0.521309000000000,0.500000000000000;4874.97241800000,0.521309000000000,0.500000000000000;4695.08900900000,0.521309000000000,0.500000000000000;4584.39152700000,0.521309000000000,0.500000000000000;4473.69404400000,0.521309000000000,0.500000000000000;4269.59556100000,0.521309000000000,0.500000000000000;4148.52019000000,0.521309000000000,0.500000000000000;4044.74130000000,0.521309000000000,0.500000000000000;3968.63678100000,0.508289000000000,0.500000000000000;3844.10211300000,0.469227000000000,0.500000000000000;3716.10814900000,0.443185000000000,0.500000000000000;3591.57348200000,0.404123000000000,0.500000000000000;3508.55037000000,0.391103000000000,0.500000000000000;3467.03881400000,0.365061000000000,0.500000000000000;3342.50414600000,0.352041000000000,0.500000000000000;3256.02173800000,0.325999000000000,0.500000000000000;3176.45792300000,0.299958000000000,0.500000000000000;3089.97551400000,0.273916000000000,0.500000000000000;3048.46395900000,0.260896000000000,0.500000000000000;3006.95240300000,0.260896000000000,0.500000000000000;2965.44084700000,0.260896000000000,0.500000000000000;2882.41773500000,0.273916000000000,0.500000000000000;2799.39462300000,0.299958000000000,0.500000000000000;2712.91221500000,0.325999000000000,0.500000000000000;2605.67402900000,0.339020000000000,0.500000000000000;2498.43584300000,0.365061000000000,0.500000000000000;2380.81976800000,0.417144000000000,0.500000000000000;2252.82580400000,0.443185000000000,0.500000000000000;2135.20972900000,0.482247000000000,0.500000000000000;2048.72732100000,0.508289000000000,0.500000000000000;2000.29717200000,0.508289000000000,0.500000000000000;1924.19265300000,0.521309000000000,0.500000000000000;1723.55346600000,0.521309000000000,0.500000000000000;1529.83287200000,0.521309000000000,0.500000000000000;1249.62987000000,0.521309000000000,0.500000000000000;983.264053000000,0.521309000000000,0.500000000000000;727.276124000000,0.521309000000000,0.500000000000000;502.421863000000,0.521309000000000,0.500000000000000;503.046458000000,7.84025600000000,0;503.046458000000,9.06709300000000,0;503.046458000000,10.6645370000000,0;503.046458000000,9.75718800000000,0;506.473724000000,8.41533500000000,0;718.964204000000,10.6645370000000,0;931.454684000000,10.6517570000000,0;1167.93602400000,10.6389780000000,0;1448.97182000000,10.6261980000000,0;1311.88118800000,10.6645370000000,0;797.791318000000,10.6645370000000,0;1688.88042700000,10.6773160000000,0;1921.93450100000,10.6773160000000,0;2192.68850000000,10.6773160000000,0;2552.55140900000,10.6900960000000,0;2861.00533100000,10.7028750000000,0;3090.63214000000,10.6773160000000,0;3337.39527800000,10.7412140000000,0;3601.29474500000,10.6900960000000,0;3830.92155400000,10.6773160000000,0;4108.53008400000,10.6773160000000,0;4468.39299300000,10.6773160000000,0;4776.84691500000,10.6773160000000,0;5027.03731900000,10.6773160000000,0;5325.20944400000,10.6773160000000,0;5606.24524000000,10.6645370000000,0;5825.59025100000,10.6645370000000,0;6003.80807300000,10.6773160000000,0;6000.38080700000,9.84664500000000,0;5996.95354200000,10.2172520000000,0;503.046458000000,0.00639000000000000,0;712.109673000000,0.00639000000000000,0;948.591013000000,0.00639000000000000,0;1226.19954300000,-0.00639000000000000,0;1462.68088300000,-0.00639000000000000,0;1664.88956600000,-0.00639000000000000,0;1846.53465300000,-0.00639000000000000,0;2117.28865200000,-0.00639000000000000,0;2302.36100500000,-0.00639000000000000,0;2473.72429600000,-0.00639000000000000,0;2655.36938300000,-0.00639000000000000,0;2816.45087600000,-0.00639000000000000,0;3059.78674800000,0.00639000000000000,0;2943.25971100000,0.00639000000000000,0;3207.15917700000,-0.00639000000000000,0;3429.93145500000,-0.00639000000000000,0;3738.38537700000,-0.00639000000000000,0;4005.71211000000,-0.00639000000000000,0;4290.17517100000,-0.00639000000000000,0;4598.62909400000,0.00639000000000000,0;4797.41051000000,0.00639000000000000,0;4992.76466100000,-0.0319490000000000,0;5277.22772300000,0.00639000000000000,0;5630.23610100000,0.00639000000000000,0;5897.56283300000,0.00639000000000000,0;6003.80807300000,-0.00639000000000000,0;512.799752000000,7.91054300000000,0;588.904271000000,8.05210100000000,0;665.008791000000,8.20652800000000,0;754.950495000000,8.41243100000000,0;824.136422000000,8.55398900000000,0;917.537422000000,8.73415400000000,0;1028.23490500000,8.86284300000000,0;1159.68816500000,8.97866400000000,0;1298.06001800000,9.13309100000000,0;1450.26905600000,9.24891100000000,0;1599.01879800000,9.42907600000000,0;1713.17557700000,9.53202700000000,0;1820.41376300000,9.60924100000000,0;1893.05898600000,9.68645400000000,0;1996.83787600000,9.75079900000000,0;2145.58761800000,9.80227400000000,0;2301.25595200000,9.86661900000000,0;2401.57554600000,9.91809500000000,0;2536.48810200000,9.95670100000000,0;2678.31925200000,9.98243900000000,0;2806.31321600000,10.0596530000000,0;2944.68506900000,10.0853910000000,0;3020.78958800000,10.1626040000000,0;3190.29510800000,10.1239970000000,0;3404.77148000000,10.1368660000000,0;3591.57348200000,10.1754730000000,0;3781.83477900000,10.2012110000000,0;3958.25889200000,10.2269490000000,0;4134.68300500000,10.3556380000000,0;4293.81063600000,10.5100650000000,0;4456.39756300000,10.5229340000000,0;4643.19956400000,10.5486720000000,0;5317.76234800000,10.5100650000000,0;5473.43068200000,10.3813760000000,0;5622.18042400000,10.2012110000000,0;5760.55227700000,10.0467840000000,0;5898.92413000000,9.90522600000000,0;6002.70302000000,9.82801200000000,0;4290.35133900000,10.4071140000000,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddGDI, RmvGDI, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/GDI.png'));
    title('Ricardo GDI Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:0.5:5] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0 10.5]);
    xlim([500 6000]);
	%}
end


%% 10: Add/Remove technology: HCCI
% map_data = [HCCI5;HCCI4p5;HCCI4;HCCI3p5;HCCI3;HCCI2p5;...
% HCCI2;HCCI1p5;HCCI1;HCCI0p5;HCCI0];

if (AddHCCI==1 || RmvHCCI==1)
	%     './data/HCCI_1.mat'
	map_data = [1010.16949200000,2.19277100000000,7;1042.37288100000,2.18473900000000,7;1083.05084700000,2.16465900000000,7;1123.72881400000,2.14457800000000,7;1162.71186400000,2.12449800000000,7;1205.08474600000,2.10441800000000,7;1242.37288100000,2.08433700000000,7;1249.15254200000,2.03614500000000,7;1257.62711900000,1.98795200000000,7;1267.79661000000,1.93574300000000,7;1276.27118600000,1.89156600000000,7;1284.74576300000,1.83935700000000,7;1291.52542400000,1.79116500000000,7;1316.94915300000,1.74297200000000,7;1350.84745800000,1.69477900000000,7;1374.57627100000,1.64658600000000,7;1391.52542400000,1.62650600000000,7;1400,1.59839400000000,7;1408.47457600000,1.57429700000000,7;1400,1.54618500000000,7;1406.77966100000,1.49799200000000,7;1440.67796600000,1.39759000000000,7;1416.94915300000,1.39759000000000,7;1406.77966100000,1.44176700000000,7;1493.22033900000,1.36947800000000,7;1542.37288100000,1.37349400000000,7;1591.52542400000,1.37751000000000,7;1647.45762700000,1.38955800000000,7;1686.44067800000,1.39759000000000,7;1700,1.41767100000000,7;1774.57627100000,1.45783100000000,7;1750.84745800000,1.47972700000000,7;1722.03389800000,1.44558300000000,7;1806.77966100000,1.41143900000000,7;1825.42372900000,1.36875900000000,7;1857.62711900000,1.32181100000000,7;1876.27118600000,1.29193500000000,7;1891.52542400000,1.27059500000000,7;1903.38983100000,1.20230700000000,7;1920.33898300000,1.19377100000000,7;1955.93220300000,1.17669900000000,7;1998.30508500000,1.16389500000000,7;2040.67796600000,1.15109100000000,7;2081.35593200000,1.13401900000000,7;2127.11864400000,1.11268000000000,7;2177.96610200000,1.09134000000000,7;2213.55932200000,1.07853600000000,7;2257.62711900000,1.06573200000000,7;2301.69491500000,1.04866000000000,7;2359.32203400000,1.03158800000000,7;2418.64406800000,1.01878400000000,7;2484.74576300000,0.988908000000000,7;2545.76271200000,0.971836000000000,7;2608.47457600000,0.929156000000000,7;2666.10169500000,0.907816000000000,7;2715.25423700000,0.886476000000000,7;2755.93220300000,0.869404000000000,7;2776.27118600000,0.843796000000000,7;2798.30508500000,0.830992000000000,7;2818.64406800000,0.813920000000000,7;2838.98305100000,0.801116000000000,7;1008.47457600000,3.13997700000000,7;1050.84745800000,3.13570900000000,7;1091.52542400000,3.13570900000000,7;1132.20339000000,3.12717300000000,7;1176.27118600000,3.12717300000000,7;1218.64406800000,3.11863700000000,7;1261.01694900000,3.11863700000000,7;1300,3.11863700000000,7;1342.37288100000,3.11436900000000,7;1383.05084700000,3.10583300000000,7;1425.42372900000,3.11010100000000,7;1467.79661000000,3.11010100000000,7;1505.08474600000,3.11010100000000,7;1533.89830500000,3.04608100000000,7;1559.32203400000,2.99913300000000,7;1576.27118600000,2.94791700000000,7;1591.52542400000,2.90096900000000,7;1610.16949200000,2.84975300000000,7;1635.59322000000,2.80280500000000,7;1659.32203400000,2.75585700000000,7;1693.22033900000,2.70464100000000,7;1718.64406800000,2.65342600000000,7;1742.37288100000,2.60647800000000,7;1784.74576300000,2.59367400000000,7;1866.10169500000,2.59794200000000,7;1825.42372900000,2.59794200000000,7;1910.16949200000,2.59794200000000,7;1950.84745800000,2.59794200000000,7;2000,2.59794200000000,7;2074.57627100000,2.54672600000000,7;2032.20339000000,2.57660200000000,7;1972.88135600000,2.59794200000000,7;2116.94915300000,2.51685000000000,7;2157.62711900000,2.49124200000000,7;2196.61016900000,2.46990200000000,7;2249.15254200000,2.44429400000000,7;2284.74576300000,2.40161400000000,7;2328.81355900000,2.36320200000000,7;2369.49152500000,2.32905800000000,7;2408.47457600000,2.29064600000000,7;2450.84745800000,2.24369800000000,7;2493.22033900000,2.19248200000000,7;2527.11864400000,2.14553400000000,7;2554.23728800000,2.09431800000000,7;2596.61016900000,2.04737000000000,7;2635.59322000000,2.00895800000000,7;2672.88135600000,1.97481400000000,7;2718.64406800000,1.94067000000000,7;2759.32203400000,1.91933000000000,7;2805.08474600000,1.94920600000000,7;2844.06779700000,1.97054600000000,7;2883.05084700000,1.98761800000000,7;2925.42372900000,2.00895800000000,7;2966.10169500000,2.02603000000000,7;3008.47457600000,2.03456600000000,7;3052.54237300000,2.01322600000000,7;3089.83050800000,1.99188600000000,7;3108.47457600000,1.94067000000000,7;3127.11864400000,1.91079400000000,7;3091.52542400000,1.84250700000000,7;3057.62711900000,1.79129100000000,7;3018.64406800000,1.74861100000000,7;2977.96610200000,1.71019900000000,7;2947.45762700000,1.66325100000000,7;2920.33898300000,1.58642700000000,7;2927.11864400000,1.51813900000000,7;2940.67796600000,1.47119100000000,7;2967.79661000000,1.41997500000000,7;2993.22033900000,1.36875900000000,7;3006.77966100000,1.34315100000000,7;3027.11864400000,1.34741900000000,7;3067.79661000000,1.33888300000000,7;3108.47457600000,1.33034700000000,7;3150.84745800000,1.32181100000000,7;3193.22033900000,1.30900700000000,7;3233.89830500000,1.30047100000000,7;3254.23728800000,1.30473900000000,7;3261.01694900000,1.23218300000000,7;3266.10169500000,1.18523500000000,7;3279.66101700000,1.13401900000000,7;3291.52542400000,1.08707200000000,7;3305.08474600000,1.04012400000000,7;3316.94915300000,0.988908000000000,7;3332.20339000000,0.941960000000000,7;3344.06779700000,0.903548000000000,7;3318.64406800000,0.877940000000000,7;3293.22033900000,0.877940000000000,7;3261.01694900000,0.860868000000000,7;3232.20339000000,0.848064000000000,7;3211.86440700000,0.839528000000000,7;3193.22033900000,0.822456000000000,7;3155.93220300000,0.818188000000000,7;3140.67796600000,0.792580000000000,7;2200,1.52208800000000,13;2250.84745800000,1.54618500000000,13;2291.52542400000,1.52208800000000,13;2332.20339000000,1.48996000000000,13;2366.10169500000,1.43775100000000,13;2383.05084700000,1.40160600000000,13;2342.37288100000,1.37349400000000,13;2300,1.35341400000000,13;2257.62711900000,1.32128500000000,13;2216.94915300000,1.34136500000000,13;2176.27118600000,1.35743000000000,13;2142.37288100000,1.37349400000000,13;2116.94915300000,1.40160600000000,13;2149.15254200000,1.44979900000000,13;2167.79661000000,1.48192800000000,13;1010.16949200000,3.59665200000000,1.50000000000000;1050.84745800000,3.57531200000000,1.50000000000000;1091.52542400000,3.56677600000000,1.50000000000000;1133.89830500000,3.54970400000000,1.50000000000000;1176.27118600000,3.53690000000000,1.50000000000000;1218.64406800000,3.51982800000000,1.50000000000000;1259.32203400000,3.51129200000000,1.50000000000000;1300,3.52836400000000,1.50000000000000;1342.37288100000,3.53690000000000,1.50000000000000;1383.05084700000,3.55824000000000,1.50000000000000;1425.42372900000,3.57531200000000,1.50000000000000;1467.79661000000,3.58811600000000,1.50000000000000;1500,3.61372400000000,1.50000000000000;1542.37288100000,3.63506400000000,1.50000000000000;1584.74576300000,3.64786800000000,1.50000000000000;1632.20339000000,3.66067200000000,1.50000000000000;1667.79661000000,3.67347600000000,1.50000000000000;1710.16949200000,3.69481600000000,1.50000000000000;1752.54237300000,3.71188800000000,1.50000000000000;1793.22033900000,3.72042400000000,1.50000000000000;1835.59322000000,3.73322800000000,1.50000000000000;1876.27118600000,3.74176400000000,1.50000000000000;1920.33898300000,3.74176400000000,1.50000000000000;1961.01694900000,3.75030000000000,1.50000000000000;2000,3.76310400000000,1.50000000000000;2042.37288100000,3.74176400000000,1.50000000000000;2084.74576300000,3.71188800000000,1.50000000000000;2127.11864400000,3.67347600000000,1.50000000000000;2167.79661000000,3.63506400000000,1.50000000000000;2211.86440700000,3.58384800000000,1.50000000000000;2255.93220300000,3.53690000000000,1.50000000000000;2272.88135600000,3.48568400000000,1.50000000000000;2296.61016900000,3.43873600000000,1.50000000000000;2318.64406800000,3.39178900000000,1.50000000000000;2338.98305100000,3.34057300000000,1.50000000000000;2361.01694900000,3.29362500000000,1.50000000000000;2384.74576300000,3.24667700000000,1.50000000000000;2410.16949200000,3.19546100000000,1.50000000000000;2430.50847500000,3.14424500000000,1.50000000000000;2452.54237300000,3.09729700000000,1.50000000000000;2474.57627100000,3.04608100000000,1.50000000000000;2498.30508500000,2.99486500000000,1.50000000000000;2527.11864400000,2.94791700000000,1.50000000000000;2552.54237300000,2.90096900000000,1.50000000000000;2579.66101700000,2.85402100000000,1.50000000000000;2610.16949200000,2.80280500000000,1.50000000000000;2628.81355900000,2.75158900000000,1.50000000000000;2650.84745800000,2.70037300000000,1.50000000000000;2661.01694900000,2.65342600000000,1.50000000000000;2676.27118600000,2.60647800000000,1.50000000000000;2693.22033900000,2.55526200000000,1.50000000000000;2711.86440700000,2.50831400000000,1.50000000000000;2730.50847500000,2.45709800000000,1.50000000000000;2759.32203400000,2.41868600000000,1.50000000000000;2844.06779700000,2.41868600000000,1.50000000000000;2927.11864400000,2.41868600000000,1.50000000000000;3003.38983100000,2.41868600000000,1.50000000000000;2977.96610200000,2.41868600000000,1.50000000000000;3025.42372900000,2.39307800000000,1.50000000000000;3052.54237300000,2.36320200000000,1.50000000000000;3084.74576300000,2.31198600000000,1.50000000000000;3111.86440700000,2.26503800000000,1.50000000000000;3135.59322000000,2.21382200000000,1.50000000000000;2784.74576300000,2.41868600000000,1.50000000000000;1022.03389800000,2.07630500000000,0.500000000000000;1023.72881400000,1.97992000000000,0.500000000000000;1022.03389800000,1.82730900000000,0.500000000000000;1020.33898300000,1.90763100000000,0.500000000000000;1025.42372900000,1.78313300000000,0.500000000000000;1033.89830500000,1.73092400000000,0.500000000000000;1042.37288100000,1.67871500000000,0.500000000000000;1084.74576300000,1.66265100000000,0.500000000000000;1125.42372900000,1.63453800000000,0.500000000000000;1167.79661000000,1.61445800000000,0.500000000000000;1208.47457600000,1.58232900000000,0.500000000000000;1250.84745800000,1.57429700000000,0.500000000000000;1259.32203400000,1.52610400000000,0.500000000000000;1267.79661000000,1.47791200000000,0.500000000000000;1267.79661000000,1.42570300000000,0.500000000000000;1284.74576300000,1.38955800000000,0.500000000000000;1325.42372900000,1.37751000000000,0.500000000000000;1366.10169500000,1.36144600000000,0.500000000000000;1408.47457600000,1.33734900000000,0.500000000000000;1450.84745800000,1.31325300000000,0.500000000000000;1493.22033900000,1.30120500000000,0.500000000000000;1530.50847500000,1.28915700000000,0.500000000000000;1576.27118600000,1.28915700000000,0.500000000000000;1618.64406800000,1.26907600000000,0.500000000000000;1657.62711900000,1.25301200000000,0.500000000000000;1701.69491500000,1.22891600000000,0.500000000000000;1742.37288100000,1.20883500000000,0.500000000000000;1783.05084700000,1.18072300000000,0.500000000000000;1867.79661000000,1.14457800000000,0.500000000000000;1827.11864400000,1.16465900000000,0.500000000000000;1916.94915300000,1.14457800000000,0.500000000000000;1950.84745800000,1.12449800000000,0.500000000000000;1993.22033900000,1.10441800000000,0.500000000000000;2035.59322000000,1.08433700000000,0.500000000000000;2091.52542400000,1.06827300000000,0.500000000000000;2137.28813600000,1.05220900000000,0.500000000000000;2159.32203400000,1.03212900000000,0.500000000000000;2203.38983100000,1.01606400000000,0.500000000000000;2240.67796600000,0.995984000000000,0.500000000000000;2301.69491500000,0.983936000000000,0.500000000000000;2328.81355900000,0.975904000000000,0.500000000000000;2369.49152500000,0.955823000000000,0.500000000000000;2408.47457600000,0.935743000000000,0.500000000000000;2450.84745800000,0.915663000000000,0.500000000000000;2493.22033900000,0.895582000000000,0.500000000000000;2535.59322000000,0.887550000000000,0.500000000000000;2576.27118600000,0.871486000000000,0.500000000000000;2620.33898300000,0.859438000000000,0.500000000000000;2659.32203400000,0.839357000000000,0.500000000000000;2701.69491500000,0.815261000000000,0.500000000000000;2742.37288100000,0.791165000000000,0.500000000000000;3261.01694900000,0.799197000000000,0.500000000000000;3284.74576300000,0.807229000000000,0.500000000000000;3327.11864400000,0.827309000000000,0.500000000000000;3367.79661000000,0.851406000000000,0.500000000000000;3410.16949200000,0.867470000000000,0.500000000000000;3450.84745800000,0.887550000000000,0.500000000000000;3484.74576300000,0.903614000000000,0.500000000000000;3464.40678000000,0.895582000000000,0.500000000000000;1011.86440700000,3.66494000000000,0.500000000000000;1049.15254200000,3.64786800000000,0.500000000000000;1091.52542400000,3.62226000000000,0.500000000000000;1133.89830500000,3.60518800000000,0.500000000000000;1176.27118600000,3.58811600000000,0.500000000000000;1218.64406800000,3.57531200000000,0.500000000000000;1259.32203400000,3.56677600000000,0.500000000000000;1301.69491500000,3.58811600000000,0.500000000000000;1333.89830500000,3.60518800000000,0.500000000000000;1354.23728800000,3.65640400000000,0.500000000000000;1376.27118600000,3.70335200000000,0.500000000000000;1418.64406800000,3.75030000000000,0.500000000000000;1459.32203400000,3.79298000000000,0.500000000000000;1501.69491500000,3.83992800000000,0.500000000000000;1544.06779700000,3.87407200000000,0.500000000000000;1584.74576300000,3.89968000000000,0.500000000000000;1625.42372900000,3.90821600000000,0.500000000000000;1669.49152500000,3.92955600000000,0.500000000000000;1710.16949200000,3.93809200000000,0.500000000000000;1750.84745800000,3.95089600000000,0.500000000000000;1793.22033900000,3.92528800000000,0.500000000000000;1835.59322000000,3.92102000000000,0.500000000000000;1877.96610200000,3.90394800000000,0.500000000000000;1920.33898300000,3.89114400000000,0.500000000000000;1961.01694900000,3.88260800000000,0.500000000000000;2001.69491500000,3.86980400000000,0.500000000000000;2042.37288100000,3.84846400000000,0.500000000000000;2086.44067800000,3.83992800000000,0.500000000000000;2127.11864400000,3.82285600000000,0.500000000000000;2169.49152500000,3.81432000000000,0.500000000000000;2211.86440700000,3.78444400000000,0.500000000000000;2252.54237300000,3.75030000000000,0.500000000000000;2293.22033900000,3.73322800000000,0.500000000000000;2337.28813600000,3.70335200000000,0.500000000000000;2371.18644100000,3.66067200000000,0.500000000000000;2396.61016900000,3.61799200000000,0.500000000000000;2415.25423700000,3.55824000000000,0.500000000000000;2427.11864400000,3.51556000000000,0.500000000000000;2432.20339000000,3.46434400000000,0.500000000000000;2435.59322000000,3.42166500000000,0.500000000000000;2447.45762700000,3.37044900000000,0.500000000000000;2462.71186400000,3.31923300000000,0.500000000000000;2483.05084700000,3.27228500000000,0.500000000000000;2505.08474600000,3.22106900000000,0.500000000000000;2544.06779700000,3.20399700000000,0.500000000000000;2562.71186400000,3.19546100000000,0.500000000000000;2591.52542400000,3.15704900000000,0.500000000000000;2618.64406800000,3.11863700000000,0.500000000000000;2652.54237300000,3.06742100000000,0.500000000000000;2662.71186400000,3.01620500000000,0.500000000000000;2694.91525400000,2.96925700000000,0.500000000000000;2728.81355900000,2.91804100000000,0.500000000000000;2769.49152500000,2.87109300000000,0.500000000000000;2811.86440700000,2.82841300000000,0.500000000000000;2852.54237300000,2.79426900000000,0.500000000000000;2893.22033900000,2.75158900000000,0.500000000000000;2937.28813600000,2.70464100000000,0.500000000000000;2976.27118600000,2.68330100000000,0.500000000000000;3020.33898300000,2.64489000000000,0.500000000000000;3062.71186400000,2.61501400000000,0.500000000000000;3098.30508500000,2.56379800000000,0.500000000000000;3128.81355900000,2.51685000000000,0.500000000000000;3149.15254200000,2.46136600000000,0.500000000000000;3164.40678000000,2.41868600000000,0.500000000000000;3181.35593200000,2.36320200000000,0.500000000000000;3194.91525400000,2.31625400000000,0.500000000000000;3203.38983100000,2.26930600000000,0.500000000000000;3203.38983100000,2.22235800000000,0.500000000000000;3227.11864400000,2.19248200000000,0.500000000000000;3269.49152500000,2.17541000000000,0.500000000000000;3310.16949200000,2.16260600000000,0.500000000000000;3352.54237300000,2.14126600000000,0.500000000000000;3393.22033900000,2.12419400000000,0.500000000000000;3471.18644100000,2.08578200000000,0.500000000000000;3435.59322000000,2.10285400000000,0.500000000000000;3479.66101700000,2.03456600000000,0.500000000000000;3477.96610200000,1.98761800000000,0.500000000000000;3476.27118600000,1.93640200000000,0.500000000000000;3476.27118600000,1.88518700000000,0.500000000000000;3477.96610200000,1.83823900000000,0.500000000000000;3477.96610200000,1.79129100000000,0.500000000000000;3477.96610200000,1.74434300000000,0.500000000000000;3476.27118600000,1.69312700000000,0.500000000000000;3476.27118600000,1.64191100000000,0.500000000000000;3476.27118600000,1.59069500000000,0.500000000000000;3476.27118600000,1.53947900000000,0.500000000000000;3494.91525400000,1.49253100000000,0.500000000000000;3498.30508500000,1.29620300000000,0.500000000000000;3479.66101700000,1.29620300000000,0.500000000000000;3481.35593200000,1.24925500000000,0.500000000000000;3484.74576300000,1.20230700000000,0.500000000000000;3486.44067800000,1.15109100000000,0.500000000000000;3486.44067800000,1.09987600000000,0.500000000000000;3486.44067800000,0.954764000000000,0.500000000000000;3484.74576300000,1.05292800000000,0.500000000000000;3486.44067800000,0.997444000000000,0.500000000000000;1008.47457600000,2.11565800000000,0.500000000000000;1008.47457600000,3.73749600000000,0;1008.47457600000,3.83566000000000,0;1008.47457600000,3.93382400000000,0;1006.77966100000,4.10027600000000,0;1103.38983100000,4.10027600000000,0;1230.50847500000,4.10027600000000,0;1374.57627100000,4.10027600000000,0;1530.50847500000,4.10027600000000,0;1696.61016900000,4.10454400000000,0;1845.76271200000,4.10454400000000,0;2023.72881400000,4.10027600000000,0;2206.77966100000,4.10027600000000,0;2386.44067800000,4.10027600000000,0;2615.25423700000,4.10454400000000,0;2794.91525400000,4.10454400000000,0;2962.71186400000,4.10454400000000,0;3167.79661000000,4.10027600000000,0;3345.76271200000,4.10027600000000,0;3505.08474600000,4.10027600000000,0;3505.08474600000,3.71615600000000,0;3505.08474600000,3.37898500000000,0;3505.08474600000,3.05034900000000,0;3505.08474600000,2.70037300000000,0;3503.38983100000,2.37600600000000,0;3505.08474600000,2.14126600000000,0;3505.08474600000,1.96627800000000,0;3503.38983100000,1.83823900000000,0;3501.69491500000,1.70166300000000,0;3503.38983100000,1.60776700000000,0;3503.38983100000,1.57362300000000,0;3503.38983100000,1.21084300000000,0;3506.77966100000,1.13401900000000,0;3508.47457600000,1.06146400000000,0;3510.16949200000,0.993176000000000,0;3510.16949200000,0.924888000000000,0;3508.47457600000,0.843796000000000,0;3505.08474600000,0.788312000000000,0;3445.76271200000,0.788312000000000,0;3391.52542400000,0.784044000000000,0;3335.59322000000,0.788312000000000,0;1006.77966100000,1.74434300000000,0;1008.47457600000,1.58215900000000,0;1008.47457600000,1.38156300000000,0;1006.77966100000,1.15962700000000,0;1006.77966100000,0.980372000000000,0;1008.47457600000,0.788312000000000,0;1074.57627100000,0.792580000000000,0;1174.57627100000,0.784044000000000,0;1254.23728800000,0.784044000000000,0;1330.50847500000,0.784044000000000,0;1476.27118600000,0.788312000000000,0;1615.25423700000,0.784044000000000,0;1747.45762700000,0.792580000000000,0;1883.05084700000,0.788312000000000,0;2010.16949200000,0.788312000000000,0;2193.22033900000,0.779776000000000,0;2337.28813600000,0.788312000000000,0;2488.13559300000,0.788312000000000,0;2642.37288100000,0.788312000000000,0;1049.15254200000,3.72896000000000,0;1094.91525400000,3.70762000000000,0;1162.71186400000,3.69054800000000,0;1220.33898300000,3.68201200000000,0;1271.18644100000,3.68628000000000,0;1306.77966100000,3.70335200000000,0;1344.06779700000,3.78444400000000,0;1383.05084700000,3.85700000000000,0;1425.42372900000,3.89114400000000,0;1474.57627100000,3.93809200000000,0;1528.81355900000,3.98504000000000,0;1588.13559300000,4.00638000000000,0;1652.54237300000,4.01918400000000,0;1705.08474600000,4.03198800000000,0;1766.10169500000,4.03625600000000,0;1830.50847500000,4.02345200000000,0;1891.52542400000,4.01064800000000,0;1966.10169500000,3.99357600000000,0;2042.37288100000,3.96796800000000,0;2111.86440700000,3.95089600000000,0;2176.27118600000,3.92955600000000,0;2238.98305100000,3.89114400000000,0;2303.38983100000,3.86126800000000,0;2374.57627100000,3.79724800000000,0;2428.81355900000,3.71188800000000,0;2467.79661000000,3.61799200000000,0;2484.74576300000,3.46434400000000,0;2516.94915300000,3.36618100000000,0;2571.18644100000,3.35764500000000,0;2642.37288100000,3.24667700000000,0;2701.69491500000,3.14851300000000,0;2771.18644100000,3.03327700000000,0;2850.84745800000,2.95218500000000,0;2923.72881400000,2.89670100000000,0;3006.77966100000,2.81134100000000,0;3086.44067800000,2.75585700000000,0;3154.23728800000,2.66196100000000,0;3200,2.52111800000000,0;3245.76271200000,2.38454200000000,0;3254.23728800000,2.30345000000000,0;3320.33898300000,2.27357400000000,0;3377.96610200000,2.24796600000000,0;3450.84745800000,2.21382200000000,0;3489.83050800000,2.18394600000000,0;1011.86440700000,1.62910700000000,0;1027.11864400000,1.56081900000000,0;1072.88135600000,1.53521100000000,0;1118.64406800000,1.48399500000000,0;1174.57627100000,1.47972700000000,0;1220.33898300000,1.43704700000000,0;1244.06779700000,1.30047100000000,0;1294.91525400000,1.24071900000000,0;1301.69491500000,1.26632700000000,0;1386.44067800000,1.21937900000000,0;1462.71186400000,1.18950300000000,0;1530.50847500000,1.15535900000000,0;1576.27118600000,1.14682300000000,0;1644.06779700000,1.10414400000000,0;1708.47457600000,1.07000000000000,0;1764.40678000000,1.04866000000000,0;1830.50847500000,1.02305200000000,0;1911.86440700000,1.00598000000000,0;1971.18644100000,0.959032000000000,0;2052.54237300000,0.941960000000000,0;2103.38983100000,0.929156000000000,0;2155.93220300000,0.890744000000000,0;2223.72881400000,0.886476000000000,0;2362.71186400000,0.848064000000000,0;2293.22033900000,0.886476000000000,0;2425.42372900000,0.835260000000000,0;2472.88135600000,0.813920000000000,0];
	[engine_out, bsfc_improve_pcnt] = engine_tech_mod( engine_out, AddHCCI, RmvHCCI, map_data );
	
	% Plot the Ricardo contour, and the interpolated contour
	%{
    figure
    subplot(1,2,1); image(imread('./data/HCCI.png'));
    title('Ricardo HCCI Map (Picture)');
    subplot(1,2,2); contourf( (engine_new.fuel_map_speed_radps .*convert.radps2rpm),...
        engine_new.fuel_map_torque_Nm*(4*pi/(displacement/1000)*1e-5), bsfc_improve_pcnt, [0:0.5:13] );
    title('Interpolation Result (Extrapolated for Real Engine Data)');xlabel('RPM'); ylabel('BMEP (Bar)');
    ylim([0.5 4]);
    xlim([1000 3500]);
	%}
end

function [ engine, bsfc_improve_pcnt ] = engine_tech_mod( engine, AddTech, RmvTech, map_data )

%% Load Ricardo map's contour data points which were picked using map_interpreter
% map_data = [DcpDvvl9p5;DcpDvvl9;DcpDvvl8p5;DcpDvvl8;DcpDvvl7p5;...
%     DcpDvvl7;DcpDvvl6p5;DcpDvvl6;DcpDvvl5p5;DcpDvvl5;DcpDvvl4p5;...
%     DcpDvvl4;DcpDvvl3p5;DcpDvvl3;DcpDvvl2p5;DcpDvvl2;DcpDvvl1p5;...
%     DcpDvvl1;DcpDvvl0p5;DcpDvvl0];
if ischar(map_data)
	load(map_data,'map_data');
end

%% Interpolate these data points to be a full BSFC improvement v.s. RPM contour map.
%     [xout,yout,zout] = scatter2surf(map_data(:,1),map_data(:,2),map_data(:,3),...
%        [1000:110:6000],[0:0.5:11],...
%        'method','scatterinterp','interp','natural','xnormalize','ynormalize',...
%        'contours',[0:.5:9.5] );
%     bsfc_improve_pcnt(isnan(bsfc_improve_pcnt))=0;    % For scatter2surf not having Extrapolation

%% scatteredInterpolant
data_xscale = max(map_data(:,1)) - min(map_data(:,1));
data_yscale = max(map_data(:,2)) - min(map_data(:,2));

F = scatteredInterpolant( map_data(:,1)/data_xscale,map_data(:,2)/data_yscale,map_data(:,3),'natural', 'linear' ); %extrapolant: nearest, linear, or none

xin = (engine.fuel_map_speed_radps .* convert.radps2rpm);    %Convert rad/s to RPM
yin = (engine.fuel_map_torque_Nm) * 4 * pi / (engine.displacement_L /1000)*1e-5;     %Convert Nm to BMEP(2.4L)
[x_grid_norm, y_grid_norm] =  meshgrid( xin/data_xscale, yin/data_yscale);

bsfc_improve_pcnt = F( x_grid_norm, y_grid_norm );

%% Update engine_new's fuel map
if (AddTech)
	engine.fuel_map_gps = engine.fuel_map_gps .* (1-bsfc_improve_pcnt/100);
elseif (RmvTech)
	engine.fuel_map_gps = engine.fuel_map_gps ./ (1-bsfc_improve_pcnt/100);
end
