function [ transmission_out ] = REVS_modify_transmission( transmission_in, varargin )
%REVS_modify_transmission takes in an transmission
%   and returns the result in a new transmission.
%
%   [ transmission_out ] = REVS_modify_transmission( transmission_in, varargin )
%
% varargs:
% scale_factor        = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
% scale_max_torque    = parse_varargs( varargin, 'rated_torque', [], 'numeric' );
% pump_scale_factor   = parse_varargs( varargin, 'pump_scale_factor', scale_factor, 'numeric' ); % scales pump friction (not pump inertia)
% scale_inertia_exp   = parse_varargs( varargin, 'scale_inertia_exp', 1.5, 'numeric' );          % 0 = no scaling

%% Parse varargs
% Check if enough varargin was passed
narginchk( 1, inf );

% Validate required arguments
validate_arg( transmission_in, {'class_REVS_AMT_transmission','class_REVS_AT_transmission','class_REVS_DCT_transmission','class_REVS_CVT_transmission','struct'});

scale_factor        = parse_varargs( varargin, 'scale_factor', 1, 'numeric' );
scale_max_torque    = parse_varargs( varargin, 'rated_torque', [], 'numeric' );

if ~isempty(scale_max_torque)
    scale_factor = scale_max_torque / transmission_in.rated_torque_Nm;
end

pump_scale_factor   = parse_varargs( varargin, 'pump_scale_factor', scale_factor, 'numeric' ); % scales pump friction (not pump inertia)
scale_inertia_exp   = parse_varargs( varargin, 'scale_inertia_exp', 1.5, 'numeric' );

%% -----------------------------------------------------------------------
transmission_out = transmission_in;

transmission_out.rated_torque_Nm = transmission_out.rated_torque_Nm * scale_factor;

if isprop(transmission_out,'torque_converter')
    transmission_out.torque_converter.pump_inertia_kgm2                 = transmission_out.torque_converter.pump_inertia_kgm2 * scale_factor;
    transmission_out.torque_converter.turbine_inertia_kgm2              = transmission_out.torque_converter.turbine_inertia_kgm2 * scale_factor;
end

if isprop(transmission_out,'pump_loss_Nm')
    transmission_out.pump_loss_Nm.table                                 = transmission_out.pump_loss_Nm.table * pump_scale_factor;
end

if isprop(transmission_out.control,'line_pressure_map_bar')
    transmission_out.control.line_pressure_map_bar.axis_2.breakpoints   = transmission_out.control.line_pressure_map_bar.axis_2.breakpoints * scale_factor;
end

transmission_out.gear.input_torque_loss_Nm.table                    = max(0, scale_factor * transmission_out.gear.input_torque_loss_Nm.table);

transmission_out.gear.inertia_kgm2                                  = scale_factor.^scale_inertia_exp * transmission_out.gear.inertia_kgm2;
transmission_out.gear.common_input_inertia_kgm2                     = scale_factor.^scale_inertia_exp * transmission_out.gear.common_input_inertia_kgm2;
transmission_out.gear.common_output_inertia_kgm2                    = scale_factor.^scale_inertia_exp * transmission_out.gear.common_output_inertia_kgm2;

end
