"""Python SWMM5 interface"""

import ctypes
import platform
import datetime

_plat= platform.system()
if _plat=='Linux':
    _lib = ctypes.CDLL("libswmm5.so")
elif _plat=='Windows':
    _lib = ctypes.WinDLL(".\swmm5.dll")
else:
    Exception('Platform '+ _plat +' unsupported')

def getVersion():
    return _lib.swmm_getVersion()

def run(f1, f2, f3 = ''):
    return _lib.swmm_run(ctypes.c_char_p(f1.encode()), 
                        ctypes.c_char_p(f2.encode()), 
                        ctypes.c_char_p(f3.encode()))

def open(f1, f2, f3 = ''):
    return _lib.swmm_open(ctypes.c_char_p(f1.encode()), 
                        ctypes.c_char_p(f2.encode()), 
                        ctypes.c_char_p(f3.encode()))

def start(saveFlag):
    return _lib.swmm_start(ctypes.c_int(saveFlag))

def step():
    elapsed_time = ctypes.c_double()
    _lib.swmm_step(ctypes.byref(elapsed_time))
    return elapsed_time.value

def stride(strideStep):
    elapsed_time = ctypes.c_double()
    _lib.swmm_stride(ctypes.c_int(strideStep), ctypes.byref(elapsed_time))
    return elapsed_time.value

def end():
    _lib.swmm_end()

def getMassBalErr():     
    runoff = ctypes.c_float()
    flow = ctypes.c_float()
    qual = ctypes.c_float()
    _lib.swmm_getMassBalErr(
        ctypes.byref(runoff), ctypes.byref(flow), ctypes.byref(qual))
    return runoff.value, flow.value, qual.value

def report():
    return _lib.swmm_report()

def close():
  _lib.swmm_close()

def getWarnings():
    return _lib.swmm_getWarnings()

def getError():
    errmsg = ctypes.create_string_buffer(240)
    _lib.swmm_getError(ctypes.byref(errmsg), ctypes.c_int(240))
    return errmsg.value.decode()

def getCount(objtype):
    return _lib.swmm_getCount(ctypes.c_int(objtype))

def getName(objtype, index, size):
    name = ctypes.create_string_buffer(size)
    _lib.swmm_getName(
        ctypes.c_int(objtype), ctypes.c_int(index), ctypes.byref(name), ctypes.c_int(size))
    return name.value.decode()

def getIndex(objtype, name):
    return _lib.swmm_getIndex(ctypes.c_int(objtype), ctypes.c_char_p(name.encode()))

def getValue(property, index):
    _lib.swmm_getValue.restype = c_double   
    return _lib.swmm_getValue(ctypes.c_int(property), ctypes.c_int(index))

def getSavedValue(property, index, period):
    _lib.swmm_getSavedValue.restype = c_double   
    return _lib.swmm_getSavedValue(
        ctypes.c_int(property), ctypes.c_int(index), ctypes.c_int(period))

def setValue(property, index, value):
    _lib.swmm_setValue(
        ctypes.c_int(property), ctypes.c_int(index), ctypes.c_double(value))

def writeLine(line):
    _lib.swmm_writeLine(ctypes.c_char_p(line.encode()))

def decodeDate(date):

    """  Decodes a SWMM DateTime value to a Python DateTime value. """
    """  Use Python's datetime.weekday()  method to get day of week """
    """  (where Monday = 0 and Sunday = 6) if needed. """

    year= ctypes.c_int()
    month= ctypes.c_int()
    day= ctypes.c_int()
    hour= ctypes.c_int()
    minute= ctypes.c_int()
    second= ctypes.c_int()
    dayofweek= ctypes.c_int()
    _lib.swmm_decodeDate(
        ctypes.c_double(date), ctypes.byref(year), ctypes.byref(month),
        ctypes.byref(day),  ctypes.byref(hour), ctypes.byref(minute),
        ctypes.byref(second), ctypes.byref(dayofweek))
    d = datetime.datetime(
        year.value, month.value, day.value, hour.value, minute.value, second.value)
    return d

swmm_GAGE     = 0
swmm_SUBCATCH = 1
swmm_NODE     = 2
swmm_LINK     = 3
swmm_SYSTEM   = 100

swmm_JUNCTION = 0
swmm_OUTFALL  = 1
swmm_STORAGE  = 2
swmm_DIVIDER  = 3

swmm_CONDUIT = 0
swmm_PUMP    = 1
swmm_ORIFICE = 2
swmm_WEIR    = 3
swmm_OUTLET  = 4

swmm_GAGE_RAINFALL = 100

swmm_SUBCATCH_AREA      = 200
swmm_SUBCATCH_RAINGAGE  = 201
swmm_SUBCATCH_RAINFALL  = 202
swmm_SUBCATCH_EVAP      = 203
swmm_SUBCATCH_INFIL     = 204
swmm_SUBCATCH_RUNOFF    = 205
swmm_SUBCATCH_RPTFLAG   = 206

swmm_NODE_TYPE     = 300
swmm_NODE_ELEV     = 301
swmm_NODE_MAXDEPTH = 302
swmm_NODE_DEPTH    = 303
swmm_NODE_HEAD     = 304
swmm_NODE_VOLUME   = 305
swmm_NODE_LATFLOW  = 306
swmm_NODE_INFLOW   = 307
swmm_NODE_OVERFLOW = 308
swmm_NODE_RPTFLAG  = 309

swmm_LINK_TYPE       = 400
swmm_LINK_NODE1      = 401
swmm_LINK_NODE2      = 402
swmm_LINK_LENGTH     = 403
swmm_LINK_SLOPE      = 404
swmm_LINK_FULLDEPTH  = 405
swmm_LINK_FULLFLOW   = 406
swmm_LINK_SETTING    = 407
swmm_LINK_TIMEOPEN   = 408
swmm_LINK_TIMECLOSED = 409
swmm_LINK_FLOW       = 410
swmm_LINK_DEPTH      = 411
swmm_LINK_VELOCITY   = 412
swmm_LINK_TOPWIDTH   = 413
swmm_LINK_RPTFLAG    = 414

swmm_STARTDATE    = 0
swmm_CURRENTDATE  = 1
swmm_ELAPSEDTIME  = 2
swmm_ROUTESTEP    = 3
swmm_MAXROUTESTEP = 4
swmm_REPORTSTEP   = 5
swmm_TOTALSTEPS   = 6
swmm_NOREPORT     = 7
swmm_FLOWUNITS    = 8

swmm_CFS = 0
swmm_GPM = 1
swmm_MGD = 2
swmm_CMS = 3
swmm_LPS = 4
swmm_MLD = 5
